"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function createTree(context) {
    const root = new lib_1.App();
    if (context) {
        Object.keys(context).forEach(key => root.node.setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(root, 'Child1');
    const child2 = new lib_1.Construct(root, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1
    };
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
/**
 * Returns a string with a tree representation of this construct and it's children.
 */
function toTreeString(node, depth = 0) {
    let out = '';
    for (let i = 0; i < depth; ++i) {
        out += '  ';
    }
    const name = node.node.id || '';
    out += `${node.constructor.name}${name.length > 0 ? ' [' + name + ']' : ''}\n`;
    for (const child of node.node.children) {
        out += toTreeString(child, depth + 1);
    }
    return out;
}
module.exports = {
    'the "Root" construct is a special construct which can be used as the root of the tree'(test) {
        const root = new lib_1.App();
        test.equal(root.node.id, '', 'if not specified, name of a root construct is an empty string');
        test.ok(!root.node.scope, 'no parent');
        test.equal(root.node.children.length, 0, 'a construct is created without children'); // no children
        test.done();
    },
    'constructs cannot be created with an empty name unless they are root'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.Construct(root, ''));
        test.done();
    },
    'construct.name returns the name of the construct'(test) {
        const t = createTree();
        test.equal(t.child1.node.id, 'Child1');
        test.equal(t.child2.node.id, 'Child2');
        test.equal(t.child1_1.node.id, 'Child11');
        test.equal(t.child1_2.node.id, 'Child12');
        test.equal(t.child1_1_1.node.id, 'Child111');
        test.equal(t.child2_1.node.id, 'Child21');
        test.done();
    },
    'construct id can use any character except the path separator'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'valid');
        new lib_1.Construct(root, 'ValiD');
        new lib_1.Construct(root, 'Va123lid');
        new lib_1.Construct(root, 'v');
        new lib_1.Construct(root, '  invalid');
        new lib_1.Construct(root, 'invalid   ');
        new lib_1.Construct(root, '123invalid');
        new lib_1.Construct(root, 'in valid');
        new lib_1.Construct(root, 'in_Valid');
        new lib_1.Construct(root, 'in-Valid');
        new lib_1.Construct(root, 'in\\Valid');
        new lib_1.Construct(root, 'in.Valid');
        test.done();
    },
    'if construct id contains path seperators, they will be replaced by double-dash'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
        test.deepEqual(c.node.id, 'Boom--Boom--Bam');
        test.done();
    },
    'if "undefined" is forcefully used as an "id", it will be treated as an empty string'(test) {
        const c = new lib_1.Construct(undefined, undefined);
        test.deepEqual(c.node.id, '');
        test.done();
    },
    "dont allow unresolved tokens to be used in construct IDs"(test) {
        // GIVEN
        const root = new lib_1.App();
        const token = lib_1.Lazy.stringValue({ produce: () => 'lazy' });
        // WHEN + THEN
        test.throws(() => new lib_1.Construct(root, `MyID: ${token}`), /Cannot use tokens in construct ID: MyID: \${Token/);
        test.done();
    },
    'construct.uniqueId returns a tree-unique alphanumeric id of this construct'(test) {
        const root = new lib_1.App();
        const child1 = new lib_1.Construct(root, 'This is the first child');
        const child2 = new lib_1.Construct(child1, 'Second level');
        const c1 = new lib_1.Construct(child2, 'My construct');
        const c2 = new lib_1.Construct(child1, 'My construct');
        test.deepEqual(c1.node.path, 'This is the first child/Second level/My construct');
        test.deepEqual(c2.node.path, 'This is the first child/My construct');
        test.deepEqual(c1.node.uniqueId, 'ThisisthefirstchildSecondlevelMyconstruct202131E0');
        test.deepEqual(c2.node.uniqueId, 'ThisisthefirstchildMyconstruct8C288DF9');
        test.done();
    },
    'cannot calculate uniqueId if the construct path is ["Default"]'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Default');
        test.throws(() => c.node.uniqueId, /Unable to calculate a unique id for an empty set of components/);
        test.done();
    },
    'construct.getChildren() returns an array of all children'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Child1');
        new lib_1.Construct(root, 'Child2');
        test.equal(child.node.children.length, 0, 'no children');
        test.equal(root.node.children.length, 2, 'two children are expected');
        test.done();
    },
    'construct.findChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.tryFindChild(child.node.id), child, 'findChild(name) can be used to retrieve the child from a parent');
        test.ok(!root.node.tryFindChild('NotFound'), 'findChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.getChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.findChild(child.node.id), child, 'getChild(name) can be used to retrieve the child from a parent');
        test.throws(() => {
            root.node.findChild('NotFound');
        }, '', 'getChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.toString() and construct.toTreeString() can be used for diagnostics'(test) {
        const t = createTree();
        test.equal(t.root.toString(), '<root>');
        test.equal(t.child1_1_1.toString(), 'Child1/Child11/Child111');
        test.equal(t.child2.toString(), 'Child2');
        test.equal(toTreeString(t.root), 'App\n  Construct [Child1]\n    Construct [Child11]\n      Construct [Child111]\n    Construct [Child12]\n  Construct [Child2]\n    Construct [Child21]\n');
        test.done();
    },
    'construct.getContext(key) can be used to read a value from context defined at the root level'(test) {
        const context = {
            ctx1: 12,
            ctx2: 'hello'
        };
        const t = createTree(context);
        test.equal(t.root.node.tryGetContext('ctx1'), 12);
        test.equal(t.child1_1_1.node.tryGetContext('ctx2'), 'hello');
        test.done();
    },
    'construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack'(test) {
        const root = new lib_1.App();
        root.node.setContext('c1', 'root');
        root.node.setContext('c2', 'root');
        const child1 = new lib_1.Construct(root, 'child1');
        child1.node.setContext('c2', 'child1');
        child1.node.setContext('c3', 'child1');
        const child2 = new lib_1.Construct(root, 'child2');
        const child3 = new lib_1.Construct(child1, 'child1child1');
        child3.node.setContext('c1', 'child3');
        child3.node.setContext('c4', 'child3');
        test.equal(root.node.tryGetContext('c1'), 'root');
        test.equal(root.node.tryGetContext('c2'), 'root');
        test.equal(root.node.tryGetContext('c3'), undefined);
        test.equal(child1.node.tryGetContext('c1'), 'root');
        test.equal(child1.node.tryGetContext('c2'), 'child1');
        test.equal(child1.node.tryGetContext('c3'), 'child1');
        test.equal(child2.node.tryGetContext('c1'), 'root');
        test.equal(child2.node.tryGetContext('c2'), 'root');
        test.equal(child2.node.tryGetContext('c3'), undefined);
        test.equal(child3.node.tryGetContext('c1'), 'child3');
        test.equal(child3.node.tryGetContext('c2'), 'child1');
        test.equal(child3.node.tryGetContext('c3'), 'child1');
        test.equal(child3.node.tryGetContext('c4'), 'child3');
        test.done();
    },
    'construct.setContext(key, value) can only be called before adding any children'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'child1');
        test.throws(() => root.node.setContext('k', 'v'));
        test.done();
    },
    'fails if context key contains unresolved tokens'(test) {
        const root = new lib_1.App();
        test.throws(() => root.node.setContext(`my-${lib_1.Aws.REGION}`, 'foo'), /Invalid context key/);
        test.throws(() => root.node.tryGetContext(lib_1.Aws.REGION), /Invalid context key/);
        test.done();
    },
    'construct.pathParts returns an array of strings of all names from root to node'(test) {
        const tree = createTree();
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'Child1/Child11/Child111');
        test.deepEqual(tree.child2.node.path, 'Child2');
        test.done();
    },
    'if a root construct has a name, it should be included in the path'(test) {
        const tree = createTree({});
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'Child1/Child11/Child111');
        test.done();
    },
    'construct can not be created with the name of a sibling'(test) {
        const root = new lib_1.App();
        // WHEN
        new lib_1.Construct(root, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(root, 'SameName');
        }, /There is already a Construct with name 'SameName' in App/);
        // WHEN
        const c0 = new lib_1.Construct(root, 'c0');
        new lib_1.Construct(c0, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(c0, 'SameName');
        }, /There is already a Construct with name 'SameName' in Construct \[c0\]/);
        test.done();
    },
    'addMetadata(type, data) can be used to attach metadata to constructs FIND_ME'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        test.deepEqual(con.node.metadata, [], 'starts empty');
        con.node.addMetadata('key', 'value');
        con.node.addMetadata('number', 103);
        con.node.addMetadata('array', [123, 456]);
        test.deepEqual(con.node.metadata[0].type, 'key');
        test.deepEqual(con.node.metadata[0].data, 'value');
        test.deepEqual(con.node.metadata[1].data, 103);
        test.deepEqual(con.node.metadata[2].data, [123, 456]);
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace[0].indexOf('FIND_ME') !== -1, 'First stack line should include this function\s name');
        test.done();
    },
    'addMetadata(type, undefined/null) is ignored'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'Foo');
        con.node.addMetadata('Null', null);
        con.node.addMetadata('Undefined', undefined);
        con.node.addMetadata('True', true);
        con.node.addMetadata('False', false);
        con.node.addMetadata('Empty', '');
        const exists = (key) => con.node.metadata.find(x => x.type === key);
        test.ok(!exists('Null'));
        test.ok(!exists('Undefined'));
        test.ok(exists('True'));
        test.ok(exists('False'));
        test.ok(exists('Empty'));
        test.done();
    },
    'addWarning(message) can be used to add a "WARNING" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addWarning('This construct is deprecated, use the other one instead');
        test.deepEqual(con.node.metadata[0].type, cxapi.WARNING_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'This construct is deprecated, use the other one instead');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addError(message) can be used to add a "ERROR" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addError('Stop!');
        test.deepEqual(con.node.metadata[0].type, cxapi.ERROR_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Stop!');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addInfo(message) can be used to add an "INFO" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addInfo('Hey there, how do you do?');
        test.deepEqual(con.node.metadata[0].type, cxapi.INFO_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Hey there, how do you do?');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'multiple children of the same type, with explicit names are welcome'(test) {
        const root = new lib_1.App();
        new MyBeautifulConstruct(root, 'mbc1');
        new MyBeautifulConstruct(root, 'mbc2');
        new MyBeautifulConstruct(root, 'mbc3');
        new MyBeautifulConstruct(root, 'mbc4');
        test.equal(root.node.children.length, 4);
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.validate() can be implemented to perform validation, ConstructNode.validate(construct.node) will return all errors from the subtree (DFS)'(test) {
        class MyConstruct extends lib_1.Construct {
            validate() {
                return ['my-error1', 'my-error2'];
            }
        }
        class YourConstruct extends lib_1.Construct {
            validate() {
                return ['your-error1'];
            }
        }
        class TheirConstruct extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new YourConstruct(this, 'YourConstruct');
            }
            validate() {
                return ['their-error'];
            }
        }
        class TestStack extends lib_1.App {
            constructor() {
                super();
                new MyConstruct(this, 'MyConstruct');
                new TheirConstruct(this, 'TheirConstruct');
            }
            validate() {
                return ['stack-error'];
            }
        }
        const stack = new TestStack();
        const errors = lib_1.ConstructNode.validate(stack.node).map((v) => ({ path: v.source.node.path, message: v.message }));
        // validate DFS
        test.deepEqual(errors, [
            { path: 'MyConstruct', message: 'my-error1' },
            { path: 'MyConstruct', message: 'my-error2' },
            { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
            { path: 'TheirConstruct', message: 'their-error' },
            { path: '', message: 'stack-error' }
        ]);
        test.done();
    },
    'construct.lock() protects against adding children anywhere under this construct (direct or indirect)'(test) {
        class LockableConstruct extends lib_1.Construct {
            lockMe() {
                this.node._lock();
            }
            unlockMe() {
                this.node._unlock();
            }
        }
        const stack = new lib_1.App();
        const c0a = new LockableConstruct(stack, 'c0a');
        const c0b = new lib_1.Construct(stack, 'c0b');
        const c1a = new lib_1.Construct(c0a, 'c1a');
        const c1b = new lib_1.Construct(c0a, 'c1b');
        c0a.lockMe();
        // now we should still be able to add children to c0b, but not to c0a or any its children
        new lib_1.Construct(c0b, 'c1a');
        test.throws(() => new lib_1.Construct(c0a, 'fail1'), /Cannot add children to "c0a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1a, 'fail2'), /Cannot add children to "c0a\/c1a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1b, 'fail3'), /Cannot add children to "c0a\/c1b" during synthesis/);
        c0a.unlockMe();
        new lib_1.Construct(c0a, 'c0aZ');
        new lib_1.Construct(c1a, 'c1aZ');
        new lib_1.Construct(c1b, 'c1bZ');
        test.done();
    },
    'findAll returns a list of all children in either DFS or BFS'(test) {
        // GIVEN
        const c1 = new lib_1.Construct(undefined, '1');
        const c2 = new lib_1.Construct(c1, '2');
        new lib_1.Construct(c1, '3');
        new lib_1.Construct(c2, '4');
        new lib_1.Construct(c2, '5');
        // THEN
        test.deepEqual(c1.node.findAll().map(x => x.node.id), c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id)); // default is PreOrder
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id), ['1', '2', '4', '5', '3']);
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.POSTORDER).map(x => x.node.id), ['4', '5', '2', '3', '1']);
        test.done();
    },
    'ancestors returns a list of parents up to root'(test) {
        const { child1_1_1 } = createTree();
        test.deepEqual(child1_1_1.node.scopes.map(x => x.node.id), ['', 'Child1', 'Child11', 'Child111']);
        test.done();
    },
    '"root" returns the root construct'(test) {
        const { child1, child2, child1_1_1, root } = createTree();
        test.ok(child1.node.root === root);
        test.ok(child2.node.root === root);
        test.ok(child1_1_1.node.root === root);
        test.done();
    },
    'defaultChild': {
        'returns the child with id "Resource"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Resource');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns the child with id "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'can override defaultChild'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'Resource');
            const defaultChild = new lib_1.Construct(root, 'OtherResource');
            root.node.defaultChild = defaultChild;
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns "undefined" if there is no default'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'child2');
            test.equal(root.node.defaultChild, undefined);
            test.done();
        },
        'fails if there are both "Resource" and "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            new lib_1.Construct(root, 'Resource');
            test.throws(() => root.node.defaultChild, /Cannot determine default child for . There is both a child with id "Resource" and id "Default"/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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