"use strict";
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
class SomeToken {
    constructor() {
        this.foo = 60;
        this.goo = 40;
    }
    resolve() {
        return this.foo + this.goo;
    }
}
module.exports = nodeunit_1.testCase({
    'capitalizeResourceProperties capitalizes all keys of an object (recursively) from camelCase to PascalCase'(test) {
        const c = new lib_1.Stack();
        test.equal(util_1.capitalizePropertyNames(c, undefined), undefined);
        test.equal(util_1.capitalizePropertyNames(c, 12), 12);
        test.equal(util_1.capitalizePropertyNames(c, 'hello'), 'hello');
        test.deepEqual(util_1.capitalizePropertyNames(c, ['hello', 88]), ['hello', 88]);
        test.deepEqual(util_1.capitalizePropertyNames(c, { Hello: 'world', hey: 'dude' }), { Hello: 'world', Hey: 'dude' });
        test.deepEqual(util_1.capitalizePropertyNames(c, [1, 2, { three: 3 }]), [1, 2, { Three: 3 }]);
        test.deepEqual(util_1.capitalizePropertyNames(c, { Hello: 'world', recursive: { foo: 123, there: { another: ['hello', { world: 123 }] } } }), { Hello: 'world', Recursive: { Foo: 123, There: { Another: ['hello', { World: 123 }] } } });
        // make sure tokens are resolved and result is also capitalized
        test.deepEqual(util_1.capitalizePropertyNames(c, { hello: { resolve: () => ({ foo: 'bar' }) }, world: new SomeToken() }), { Hello: { Foo: 'bar' }, World: 100 });
        test.done();
    },
    'ignoreEmpty': {
        '[]'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty([])), undefined);
            test.done();
        },
        '{}'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty({})), undefined);
            test.done();
        },
        'undefined/null'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(undefined)), undefined);
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(null)), null);
            test.done();
        },
        'primitives'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(12)), 12);
            test.strictEqual(stack.resolve(util_1.ignoreEmpty("12")), "12");
            test.done();
        },
        'non-empty arrays/objects'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(util_1.ignoreEmpty([1, 2, 3, undefined])), [1, 2, 3]); // undefined array values is cleaned up by "resolve"
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ o: 1, b: 2, j: 3 })), { o: 1, b: 2, j: 3 });
            test.done();
        },
        'resolve first'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => 123 } })), { xoo: 123 });
            test.strictEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => undefined } })), undefined);
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => [] } })), { xoo: [] });
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => [undefined, undefined] } })), { xoo: [] });
            test.done();
        }
    },
    'filterUnderined': {
        'is null-safe (aka treats null and undefined the same)'(test) {
            test.deepEqual(util_1.filterUndefined({ 'a null': null, 'a not null': true }), { 'a not null': true });
            test.done();
        },
        'removes undefined, but leaves the rest'(test) {
            test.deepEqual(util_1.filterUndefined({ 'an undefined': undefined, 'yes': true }), { yes: true });
            test.done();
        }
    }
});
//# sourceMappingURL=data:application/json;base64,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