from ipaddress import IPv4Network

import httpx
import pytest
from pytest_httpx import HTTPXMock


@pytest.mark.mock
class TestMockedL0API:
    """Test L0 API endpoints with mocked responses"""

    @pytest.mark.asyncio
    async def test_get_address_balance(
        self, network, httpx_mock: HTTPXMock, mock_l0_api_responses
    ):
        network.config("integrationnet")
        address = "DAG0zJW14beJtZX2BY2KA9gLbpaZ8x6vgX4KVPVX"
        httpx_mock.add_response(
            url="https://l0-lb-integrationnet.constellationnetwork.io/dag/DAG0zJW14beJtZX2BY2KA9gLbpaZ8x6vgX4KVPVX/balance",
            json=mock_l0_api_responses["address_balance"],
        )
        result = await network.get_address_balance(address)
        assert result.model_dump() == {
            "ordinal": 3381908,
            "balance": 5699930,
            "address": None,
            "meta": None,
        }

    @pytest.mark.asyncio
    async def test_get_total_supply(
        self, network, httpx_mock: HTTPXMock, mock_l0_api_responses
    ):
        network.config("integrationnet")
        httpx_mock.add_response(
            url="https://l0-lb-integrationnet.constellationnetwork.io/dag/total-supply",
            json=mock_l0_api_responses["total_supply"],
        )
        result = await network.l0_api.get_total_supply()
        assert result.model_dump() == {
            "ordinal": 3381985,
            "total_supply": 97227345262286877,
        }

    @pytest.mark.asyncio
    async def test_get_cluster_info(
        self, network, httpx_mock: HTTPXMock, mock_l0_api_responses
    ):
        network.config("integrationnet")
        httpx_mock.add_response(
            url="https://l0-lb-integrationnet.constellationnetwork.io/cluster/info",
            json=mock_l0_api_responses["cluster_info"],
        )
        results = await network.l0_api.get_cluster_info()
        assert [r.model_dump() for r in results] == [
            {
                "alias": None,
                "id": "615b72d69facdbd915b234771cd4ffe49692a573f7ac05fd212701afe9b703eb8ab2ab117961f819e6d5eaf5ad073456cf56a0422c67b17606d95f50460a919d",
                "ip": IPv4Network("5.161.233.213/32"),
                "state": "Ready",
                "session": 1748983955866,
                "public_port": 9000,
                "p2p_port": 9001,
                "reputation": None,
            }
        ]

    @pytest.mark.asyncio
    async def test_get_latest_l0_snapshot(
        self, network, httpx_mock: HTTPXMock, mock_l0_api_responses
    ):
        network.config("integrationnet")
        httpx_mock.add_response(
            url="https://l0-lb-integrationnet.constellationnetwork.io/global-snapshots/latest",
            json=mock_l0_api_responses["latest_global_snapshot"],
        )
        result = await network.l0_api.get_latest_snapshot()
        assert result.model_dump() == {
            "value": {
                "ordinal": 3382066,
                "height": 346,
                "sub_height": 5502,
                "last_snapshot_hash": "01feca0848f4793fa644af1389a4b2e32cb56a1b1ba315562bf8b7a67a77b715",
                "blocks": [],
                "state_channel_snapshots": {
                    "DAG5kfY9GoHF1CYaY8tuRJxmB3JSzAEARJEAkA2C": [
                        {
                            "value": {
                                "last_snapshot_hash": "571e90c082e8551b89716fc1ba4719af9a36f782a13d0cafea86e4ae6351f669",
                                "content": [
                                    27,
                                    -40,
                                    31,
                                    0,
                                    -128,
                                    -86,
                                    -86,
                                    -86,
                                    -22,
                                    -1,
                                    116,
                                    -73,
                                    -77,
                                    67,
                                    -125,
                                    -20,
                                    42,
                                    26,
                                    -73,
                                    -22,
                                    6,
                                    104,
                                    -24,
                                    -61,
                                    12,
                                    116,
                                    65,
                                    103,
                                    65,
                                    95,
                                    10,
                                    -14,
                                    32,
                                    -101,
                                    102,
                                    6,
                                    84,
                                    116,
                                    68,
                                    66,
                                    -72,
                                    103,
                                    86,
                                    55,
                                    36,
                                    4,
                                    85,
                                    85,
                                    85,
                                    -43,
                                    -1,
                                    -35,
                                    47,
                                    118,
                                    -67,
                                    59,
                                    -88,
                                    31,
                                    -3,
                                    104,
                                    -121,
                                    -109,
                                    -127,
                                    93,
                                    78,
                                    110,
                                    6,
                                    122,
                                    112,
                                    3,
                                    -65,
                                    -8,
                                    -47,
                                    -81,
                                    126,
                                    -13,
                                    -77,
                                    -37,
                                    -59,
                                    1,
                                    28,
                                    -44,
                                    -51,
                                    -43,
                                    -64,
                                    -3,
                                    112,
                                    -75,
                                    -125,
                                    31,
                                    46,
                                    14,
                                    -18,
                                    23,
                                    -65,
                                    -7,
                                    -35,
                                    111,
                                    14,
                                    126,
                                    114,
                                    -13,
                                    -101,
                                    -33,
                                    -52,
                                    1,
                                    28,
                                    64,
                                    -51,
                                    -4,
                                    -30,
                                    14,
                                    118,
                                    -13,
                                    -125,
                                    93,
                                    -4,
                                    -26,
                                    -89,
                                    -125,
                                    -125,
                                    -87,
                                    10,
                                    -128,
                                    -61,
                                    -35,
                                    14,
                                    -18,
                                    122,
                                    114,
                                    -80,
                                    -29,
                                    -55,
                                    15,
                                    -9,
                                    -93,
                                    31,
                                    28,
                                    -64,
                                    47,
                                    78,
                                    85,
                                    85,
                                    85,
                                    -11,
                                    79,
                                    98,
                                    6,
                                    -32,
                                    14,
                                    -32,
                                    -96,
                                    126,
                                    -77,
                                    69,
                                    69,
                                    92,
                                    84,
                                    85,
                                    84,
                                    -59,
                                    124,
                                    -65,
                                    63,
                                    -81,
                                    31,
                                    -15,
                                    -14,
                                    125,
                                    -82,
                                    -89,
                                    -97,
                                    -41,
                                    -37,
                                    123,
                                    63,
                                    -65,
                                    -58,
                                    -53,
                                    -11,
                                    -124,
                                    -118,
                                    -56,
                                    27,
                                    110,
                                    -41,
                                    -41,
                                    121,
                                    -2,
                                    -14,
                                    -11,
                                    113,
                                    61,
                                    33,
                                    -36,
                                    -82,
                                    -5,
                                    -9,
                                    -4,
                                    -57,
                                    118,
                                    94,
                                    70,
                                    -73,
                                    -21,
                                    37,
                                    -18,
                                    -113,
                                    79,
                                    -81,
                                    -15,
                                    -19,
                                    -2,
                                    -11,
                                    -19,
                                    -15,
                                    -113,
                                    -72,
                                    127,
                                    -67,
                                    -98,
                                    -82,
                                    30,
                                    -95,
                                    -79,
                                    -45,
                                    41,
                                    -100,
                                    -67,
                                    -40,
                                    -80,
                                    69,
                                    92,
                                    72,
                                    89,
                                    59,
                                    91,
                                    104,
                                    -23,
                                    41,
                                    38,
                                    72,
                                    64,
                                    69,
                                    95,
                                    -68,
                                    -105,
                                    9,
                                    90,
                                    115,
                                    -70,
                                    -61,
                                    54,
                                    86,
                                    -10,
                                    -21,
                                    118,
                                    -27,
                                    -53,
                                    91,
                                    -3,
                                    113,
                                    -65,
                                    -98,
                                    126,
                                    -1,
                                    124,
                                    -69,
                                    -34,
                                    -25,
                                    -49,
                                    120,
                                    -17,
                                    -5,
                                    -53,
                                    -25,
                                    -29,
                                    -7,
                                    -37,
                                    -3,
                                    122,
                                    -6,
                                    121,
                                    -11,
                                    124,
                                    123,
                                    -97,
                                    -118,
                                    -57,
                                    -12,
                                    -11,
                                    -12,
                                    -5,
                                    -49,
                                    123,
                                    -68,
                                    -41,
                                    -45,
                                    79,
                                    123,
                                    -83,
                                    -81,
                                    38,
                                    -35,
                                    67,
                                    26,
                                    32,
                                    62,
                                    117,
                                    40,
                                    71,
                                    -114,
                                    106,
                                    58,
                                    -99,
                                    -99,
                                    91,
                                    89,
                                    102,
                                    4,
                                    20,
                                    -35,
                                    -109,
                                    12,
                                    -42,
                                    -114,
                                    -95,
                                    36,
                                    90,
                                    75,
                                    -111,
                                    106,
                                    47,
                                    62,
                                    -62,
                                    -76,
                                    -49,
                                    100,
                                    -29,
                                    -11,
                                    113,
                                    83,
                                    121,
                                    25,
                                    -65,
                                    60,
                                    -82,
                                    39,
                                    84,
                                    16,
                                    117,
                                    -3,
                                    -72,
                                    41,
                                    23,
                                    -98,
                                    -32,
                                    105,
                                    91,
                                    84,
                                    22,
                                    41,
                                    -66,
                                    4,
                                    79,
                                    -113,
                                    -70,
                                    86,
                                    53,
                                    98,
                                    80,
                                    12,
                                    91,
                                    -56,
                                    28,
                                    94,
                                    -115,
                                    -30,
                                    -24,
                                    -126,
                                    39,
                                    15,
                                    71,
                                    -58,
                                    -111,
                                    -34,
                                    120,
                                    114,
                                    47,
                                    -94,
                                    -84,
                                    45,
                                    -62,
                                    83,
                                    -46,
                                    -26,
                                    108,
                                    40,
                                    120,
                                    -118,
                                    33,
                                    -102,
                                    21,
                                    13,
                                    -39,
                                    82,
                                    79,
                                    77,
                                    29,
                                    89,
                                    34,
                                    58,
                                    -58,
                                    11,
                                    99,
                                    109,
                                    -107,
                                    45,
                                    103,
                                    105,
                                    -52,
                                    70,
                                    27,
                                    -77,
                                    3,
                                    100,
                                    57,
                                    80,
                                    -3,
                                    106,
                                    5,
                                    -71,
                                    15,
                                    123,
                                    22,
                                    -50,
                                    30,
                                    80,
                                    -83,
                                    4,
                                    88,
                                    -93,
                                    -62,
                                    -103,
                                    39,
                                    -20,
                                    112,
                                    -111,
                                    -23,
                                    -76,
                                    28,
                                    32,
                                    63,
                                    10,
                                    30,
                                    -20,
                                    -80,
                                    -59,
                                    3,
                                    -47,
                                    -95,
                                    -100,
                                    -37,
                                    -53,
                                    66,
                                    -126,
                                    105,
                                    -42,
                                    -38,
                                    -12,
                                    33,
                                    -36,
                                    -59,
                                    -55,
                                    -74,
                                    96,
                                    -22,
                                    98,
                                    -43,
                                    61,
                                    6,
                                    1,
                                    -99,
                                    -108,
                                    91,
                                    49,
                                    102,
                                    103,
                                    59,
                                    -118,
                                    -58,
                                    17,
                                    68,
                                    -86,
                                    -16,
                                    65,
                                    -30,
                                    105,
                                    -43,
                                    16,
                                    85,
                                    81,
                                    51,
                                    -37,
                                    37,
                                    -24,
                                    66,
                                    96,
                                    102,
                                    -18,
                                    82,
                                    -89,
                                    46,
                                    -121,
                                    46,
                                    46,
                                    -84,
                                    96,
                                    -95,
                                    67,
                                    103,
                                    -46,
                                    -73,
                                    -105,
                                    -72,
                                    -93,
                                    -122,
                                    -125,
                                    49,
                                    81,
                                    -30,
                                    28,
                                    69,
                                    56,
                                    -74,
                                    -91,
                                    54,
                                    37,
                                    51,
                                    3,
                                    25,
                                    10,
                                    -71,
                                    -29,
                                    94,
                                    68,
                                    -125,
                                    67,
                                    -80,
                                    -74,
                                    -20,
                                    -122,
                                    85,
                                    14,
                                    91,
                                    -78,
                                    -124,
                                    -92,
                                    71,
                                    5,
                                    39,
                                    -43,
                                    -122,
                                    98,
                                    21,
                                    -86,
                                    112,
                                    101,
                                    -48,
                                    94,
                                    75,
                                    -122,
                                    54,
                                    -44,
                                    108,
                                    -99,
                                    68,
                                    60,
                                    37,
                                    -40,
                                    -63,
                                    70,
                                    115,
                                    36,
                                    17,
                                    14,
                                    61,
                                    48,
                                    85,
                                    22,
                                    70,
                                    123,
                                    103,
                                    -80,
                                    122,
                                    -101,
                                    116,
                                    -119,
                                    -79,
                                    -97,
                                    -99,
                                    -77,
                                    -93,
                                    -105,
                                    114,
                                    -69,
                                    -113,
                                    -99,
                                    112,
                                    -86,
                                    -47,
                                    2,
                                    -23,
                                    -67,
                                    -42,
                                    106,
                                    -79,
                                    9,
                                    -30,
                                    65,
                                    -107,
                                    43,
                                    -14,
                                    -34,
                                    -105,
                                    40,
                                    -107,
                                    -85,
                                    18,
                                    58,
                                    -84,
                                    90,
                                    126,
                                    28,
                                    81,
                                    -77,
                                    -102,
                                    109,
                                    60,
                                    -11,
                                    -120,
                                    -113,
                                    7,
                                    49,
                                    12,
                                    107,
                                    -107,
                                    -79,
                                    34,
                                    -99,
                                    -18,
                                    5,
                                    -35,
                                    -42,
                                    -128,
                                    -83,
                                    -18,
                                    -42,
                                    -123,
                                    -64,
                                    -28,
                                    -62,
                                    25,
                                    124,
                                    28,
                                    13,
                                    -103,
                                    29,
                                    34,
                                    69,
                                    -69,
                                    -89,
                                    86,
                                    -60,
                                    73,
                                    12,
                                    -81,
                                    -43,
                                    123,
                                    -80,
                                    -86,
                                    -116,
                                    -94,
                                    122,
                                    -10,
                                    58,
                                    82,
                                    103,
                                    -100,
                                    -85,
                                    -116,
                                    78,
                                    111,
                                    -100,
                                    -76,
                                    12,
                                    -27,
                                    -41,
                                    -13,
                                    -7,
                                    118,
                                    -67,
                                    -49,
                                    127,
                                    -30,
                                    -7,
                                    117,
                                    -6,
                                    -105,
                                    122,
                                    60,
                                    -1,
                                    -104,
                                    -128,
                                    70,
                                    -40,
                                    -61,
                                    92,
                                    0,
                                    -125,
                                    -50,
                                    -90,
                                    -96,
                                    58,
                                    65,
                                    60,
                                    42,
                                    20,
                                    -18,
                                    -121,
                                    -14,
                                    108,
                                    44,
                                    -63,
                                    -30,
                                    89,
                                    -22,
                                    -119,
                                    16,
                                    5,
                                    -70,
                                    99,
                                    31,
                                    -36,
                                    -71,
                                    -40,
                                    56,
                                    72,
                                    -116,
                                    4,
                                    125,
                                    -79,
                                    -23,
                                    -9,
                                    123,
                                    124,
                                    -103,
                                    -65,
                                    -67,
                                    125,
                                    127,
                                    125,
                                    92,
                                    79,
                                    120,
                                    -69,
                                    -98,
                                    95,
                                    31,
                                    -17,
                                    111,
                                    -3,
                                    -67,
                                    -52,
                                    -101,
                                    68,
                                    54,
                                    45,
                                    -20,
                                    9,
                                    10,
                                    -91,
                                    31,
                                    110,
                                    52,
                                    8,
                                    52,
                                    55,
                                    -59,
                                    -47,
                                    10,
                                    97,
                                    105,
                                    -39,
                                    -75,
                                    -52,
                                    27,
                                    79,
                                    45,
                                    11,
                                    -56,
                                    -77,
                                    -40,
                                    1,
                                    -38,
                                    107,
                                    109,
                                    -19,
                                    35,
                                    -78,
                                    99,
                                    -11,
                                    2,
                                    -60,
                                    -60,
                                    28,
                                    -66,
                                    108,
                                    41,
                                    14,
                                    -90,
                                    -101,
                                    92,
                                    9,
                                    116,
                                    -117,
                                    42,
                                    -118,
                                    -125,
                                    -43,
                                    115,
                                    -123,
                                    -54,
                                    -84,
                                    45,
                                    88,
                                    -84,
                                    -57,
                                    -90,
                                    -68,
                                    123,
                                    29,
                                    -16,
                                    -84,
                                    -123,
                                    92,
                                    -128,
                                    64,
                                    64,
                                    -85,
                                    61,
                                    50,
                                    -50,
                                    -34,
                                    10,
                                    -53,
                                    106,
                                    -127,
                                    -93,
                                    11,
                                    -60,
                                    100,
                                    -125,
                                    95,
                                    116,
                                    109,
                                    12,
                                    -119,
                                    84,
                                    -92,
                                    -109,
                                    57,
                                    -46,
                                    67,
                                    -114,
                                    -72,
                                    8,
                                    -12,
                                    -120,
                                    35,
                                    -75,
                                    56,
                                    27,
                                    -84,
                                    -51,
                                    -50,
                                    96,
                                    40,
                                    1,
                                    10,
                                    49,
                                    43,
                                    -62,
                                    -9,
                                    -100,
                                    -47,
                                    -38,
                                    -48,
                                    -123,
                                    127,
                                    66,
                                    82,
                                    119,
                                    27,
                                    29,
                                    -29,
                                    93,
                                    75,
                                    54,
                                    -109,
                                    47,
                                    -107,
                                    38,
                                    -54,
                                    -90,
                                    65,
                                    111,
                                    81,
                                    -89,
                                    -78,
                                    56,
                                    43,
                                    11,
                                    -122,
                                    15,
                                    -115,
                                    -11,
                                    25,
                                    -40,
                                    96,
                                    1,
                                    29,
                                    45,
                                    101,
                                    80,
                                    -86,
                                    32,
                                    -96,
                                    -69,
                                    -74,
                                    1,
                                    111,
                                    -71,
                                    -100,
                                    -84,
                                    -40,
                                    62,
                                    -66,
                                    -36,
                                    -70,
                                    -114,
                                    -107,
                                    119,
                                    98,
                                    46,
                                    20,
                                    100,
                                    -54,
                                    -45,
                                    -94,
                                    -123,
                                    38,
                                    -122,
                                    -124,
                                    -39,
                                    -103,
                                    59,
                                    -83,
                                    -86,
                                    97,
                                    -30,
                                    72,
                                    -99,
                                    29,
                                    -119,
                                    -48,
                                    -58,
                                    104,
                                    114,
                                    -125,
                                    -18,
                                    68,
                                    115,
                                    -125,
                                    -57,
                                    -76,
                                    -46,
                                    25,
                                    -35,
                                    -102,
                                    25,
                                    -26,
                                    110,
                                    -63,
                                    -57,
                                    -77,
                                    106,
                                    123,
                                    -104,
                                    33,
                                    -92,
                                    -80,
                                    -80,
                                    89,
                                    -5,
                                    100,
                                    22,
                                    110,
                                    -34,
                                    -53,
                                    -96,
                                    118,
                                    -61,
                                    64,
                                    27,
                                    -64,
                                    8,
                                    1,
                                    -9,
                                    -20,
                                    -59,
                                    90,
                                    -44,
                                    -101,
                                    -84,
                                    22,
                                    -51,
                                    90,
                                    -78,
                                    -72,
                                    90,
                                    -128,
                                    -111,
                                    -105,
                                    -92,
                                    -103,
                                    30,
                                    -102,
                                    -109,
                                    -91,
                                    116,
                                    -106,
                                    -99,
                                    57,
                                    37,
                                    -82,
                                    66,
                                    96,
                                    -92,
                                    -32,
                                    -115,
                                    48,
                                    2,
                                    101,
                                    4,
                                    39,
                                    4,
                                    -36,
                                    39,
                                    -72,
                                    3,
                                    108,
                                    3,
                                    -32,
                                    72,
                                    -97,
                                    -61,
                                    99,
                                    -27,
                                    -79,
                                    107,
                                    84,
                                    33,
                                    25,
                                    -116,
                                    50,
                                    -12,
                                    84,
                                    51,
                                    26,
                                    -69,
                                    -70,
                                    59,
                                    122,
                                    54,
                                    22,
                                    -64,
                                    -54,
                                    -62,
                                    110,
                                    1,
                                    42,
                                    83,
                                    -40,
                                    -14,
                                    -49,
                                    31,
                                    -73,
                                    -21,
                                    -2,
                                    -120,
                                    -57,
                                    -4,
                                    -6,
                                    -2,
                                    -10,
                                    118,
                                    -82,
                                    -89,
                                    -97,
                                    -85,
                                    -59,
                                    -74,
                                    -33,
                                    -2,
                                    -5,
                                    -81,
                                    57,
                                    -9,
                                    -61,
                                    -97,
                                    -49,
                                    37,
                                    -120,
                                    -42,
                                    89,
                                    -45,
                                    -46,
                                    108,
                                    27,
                                    88,
                                    -112,
                                    -70,
                                    119,
                                    31,
                                    -35,
                                    92,
                                    33,
                                    29,
                                    94,
                                    94,
                                    85,
                                    -106,
                                    -52,
                                    -90,
                                    -85,
                                    28,
                                    56,
                                    100,
                                    -71,
                                    -57,
                                    68,
                                    -83,
                                    -91,
                                    -36,
                                    113,
                                    -35,
                                    -82,
                                    -116,
                                    -105,
                                    120,
                                    -83,
                                    -71,
                                    -1,
                                    -6,
                                    -2,
                                    -10,
                                    118,
                                    -82,
                                    -89,
                                    -117,
                                    -9,
                                    -38,
                                    -36,
                                    9,
                                    16,
                                    -85,
                                    5,
                                    21,
                                    -114,
                                    -90,
                                    -100,
                                    117,
                                    -74,
                                    -72,
                                    -70,
                                    66,
                                    120,
                                    112,
                                    23,
                                    14,
                                    120,
                                    -118,
                                    -17,
                                    -106,
                                    86,
                                    -34,
                                    -29,
                                    53,
                                    -75,
                                    82,
                                    -71,
                                    93,
                                    100,
                                    -82,
                                    -37,
                                    -11,
                                    18,
                                    -9,
                                    -57,
                                    -1,
                                    -49,
                                    -3,
                                    30,
                                    95,
                                    -86,
                                    62,
                                    120,
                                    106,
                                    87,
                                    76,
                                    -7,
                                    -46,
                                    58,
                                    -86,
                                    -92,
                                    19,
                                    71,
                                    -125,
                                    -110,
                                    -61,
                                    76,
                                    58,
                                    124,
                                    16,
                                    120,
                                    -110,
                                    13,
                                    64,
                                    79,
                                    71,
                                    41,
                                    -43,
                                    54,
                                    78,
                                    59,
                                    29,
                                    116,
                                    -56,
                                    -106,
                                    16,
                                    -34,
                                    -65,
                                    -49,
                                    120,
                                    -95,
                                    -7,
                                    -6,
                                    -3,
                                    -27,
                                    -27,
                                    108,
                                    -62,
                                    95,
                                    94,
                                    94,
                                    -34,
                                    -2,
                                    -4,
                                    -12,
                                    109,
                                    94,
                                    -5,
                                    -110,
                                    -50,
                                    -53,
                                    -16,
                                    88,
                                    -6,
                                    -39,
                                    -13,
                                    92,
                                    88,
                                    -19,
                                    121,
                                    92,
                                    75,
                                    84,
                                    125,
                                    105,
                                    -38,
                                    -111,
                                    72,
                                    -107,
                                    22,
                                    -46,
                                    -88,
                                    53,
                                    -72,
                                    86,
                                    66,
                                    -78,
                                    40,
                                    4,
                                    37,
                                    -15,
                                    86,
                                    56,
                                    -116,
                                    -5,
                                    -70,
                                    93,
                                    49,
                                    -73,
                                    16,
                                    42,
                                    94,
                                    -32,
                                    94,
                                    -75,
                                    -19,
                                    -53,
                                    -53,
                                    91,
                                    -58,
                                    -53,
                                    -30,
                                    63,
                                    46,
                                    -97,
                                    -2,
                                    -9,
                                    90,
                                    -41,
                                    -45,
                                    37,
                                    91,
                                    -79,
                                    78,
                                    -86,
                                    70,
                                    -37,
                                    10,
                                    -87,
                                    -88,
                                    -39,
                                    25,
                                    -42,
                                    107,
                                    102,
                                    -100,
                                    -107,
                                    57,
                                    13,
                                    105,
                                    -71,
                                    -106,
                                    43,
                                    -107,
                                    -127,
                                    -61,
                                    -18,
                                    -98,
                                    40,
                                    33,
                                    -12,
                                    77,
                                    24,
                                    -30,
                                    -87,
                                    -9,
                                    -9,
                                    -37,
                                    -37,
                                    31,
                                    -13,
                                    -6,
                                    127,
                                    111,
                                    -11,
                                    -121,
                                    59,
                                    -80,
                                    -122,
                                    -63,
                                    127,
                                    -126,
                                    -5,
                                    -72,
                                    93,
                                    -13,
                                    -19,
                                    -83,
                                    -66,
                                    -2,
                                    -6,
                                    -2,
                                    -10,
                                    -27,
                                    125,
                                    -18,
                                    -9,
                                    -21,
                                    105,
                                    -21,
                                    50,
                                    89,
                                    -73,
                                    -85,
                                    -29,
                                    17,
                                    -65,
                                    124,
                                    -5,
                                    -10,
                                    -14,
                                    92,
                                    -15,
                                    120,
                                    126,
                                    123,
                                    77,
                                    -81,
                                    -94,
                                    127,
                                    -3,
                                    -37,
                                    -41,
                                    120,
                                    126,
                                    -3,
                                    -12,
                                    -120,
                                    -57,
                                    92,
                                    79,
                                    -65,
                                    35,
                                    -15,
                                    -115,
                                    -27,
                                    -122,
                                    -72,
                                    110,
                                    -120,
                                    116,
                                    67,
                                    -128,
                                    -37,
                                    94,
                                    55,
                                    68,
                                    -69,
                                    33,
                                    -32,
                                    13,
                                    81,
                                    111,
                                    44,
                                    55,
                                    -11,
                                    -37,
                                    -58,
                                    -37,
                                    -26,
                                    27,
                                    -110,
                                    126,
                                    -66,
                                    -115,
                                    111,
                                    -21,
                                    -98,
                                    15,
                                    37,
                                    -83,
                                    120,
                                    -87,
                                    -17,
                                    47,
                                    -15,
                                    -104,
                                    -2,
                                    52,
                                    -40,
                                    118,
                                    116,
                                    -91,
                                    21,
                                    -47,
                                    89,
                                    24,
                                    124,
                                    -116,
                                    -109,
                                    -60,
                                    123,
                                    79,
                                    74,
                                    39,
                                    23,
                                    -12,
                                    -50,
                                    51,
                                    -77,
                                    92,
                                    103,
                                    -98,
                                    -36,
                                    -122,
                                    83,
                                    -106,
                                    -113,
                                    25,
                                    88,
                                    -86,
                                    77,
                                    -91,
                                    -56,
                                    -106,
                                    -68,
                                    62,
                                    110,
                                    -41,
                                    127,
                                    102,
                                    12,
                                    24,
                                    -54,
                                    -6,
                                    52,
                                    120,
                                    -11,
                                    -66,
                                    88,
                                    -63,
                                    121,
                                    -93,
                                    -3,
                                    91,
                                    -68,
                                    -49,
                                    -21,
                                    -29,
                                    -97,
                                    -39,
                                    -29,
                                    88,
                                    69,
                                    85,
                                    -70,
                                    21,
                                    95,
                                    49,
                                    43,
                                    -77,
                                    19,
                                    24,
                                    -65,
                                    -51,
                                    4,
                                    -14,
                                    84,
                                    14,
                                    -125,
                                    6,
                                    104,
                                    8,
                                    -69,
                                    -12,
                                    44,
                                    -108,
                                    106,
                                    118,
                                    8,
                                    -18,
                                    -125,
                                    -30,
                                    123,
                                    -40,
                                    100,
                                    -95,
                                    -81,
                                    109,
                                    80,
                                    90,
                                    108,
                                    -50,
                                    97,
                                    6,
                                    -67,
                                    -12,
                                    72,
                                    29,
                                    -12,
                                    35,
                                    -5,
                                    -70,
                                    93,
                                    -9,
                                    -71,
                                    -33,
                                    -67,
                                    -18,
                                    15,
                                    -105,
                                    -8,
                                    118,
                                    84,
                                    6,
                                    67,
                                    -6,
                                    -72,
                                    93,
                                    -33,
                                    -34,
                                    -33,
                                    -34,
                                    -114,
                                    -7,
                                    -11,
                                    -25,
                                    -18,
                                    -77,
                                    127,
                                    -78,
                                    100,
                                    -37,
                                    16,
                                    -43,
                                    57,
                                    -63,
                                    -116,
                                    -78,
                                    -43,
                                    35,
                                    -112,
                                    78,
                                    -95,
                                    -107,
                                    -87,
                                    -41,
                                    -24,
                                    52,
                                    -124,
                                    119,
                                    0,
                                    -109,
                                    19,
                                    -121,
                                    33,
                                    64,
                                    -15,
                                    -18,
                                    53,
                                    71,
                                    113,
                                    59,
                                    17,
                                    -70,
                                    59,
                                    43,
                                    20,
                                    -74,
                                    64,
                                    53,
                                    107,
                                    -71,
                                    111,
                                    -22,
                                    -75,
                                    -10,
                                    -20,
                                    86,
                                    72,
                                    41,
                                    15,
                                    56,
                                    42,
                                    -124,
                                    -26,
                                    101,
                                    -95,
                                    110,
                                    -37,
                                    86,
                                    98,
                                    89,
                                    -73,
                                    38,
                                    42,
                                    95,
                                    -73,
                                    -21,
                                    -2,
                                    -4,
                                    -27,
                                    53,
                                    30,
                                    -33,
                                    -33,
                                    -25,
                                    122,
                                    -70,
                                    24,
                                    68,
                                    -127,
                                    8,
                                    112,
                                    -41,
                                    -50,
                                    105,
                                    -51,
                                    17,
                                    -76,
                                    110,
                                    -11,
                                    -100,
                                    61,
                                    -76,
                                    106,
                                    -101,
                                    59,
                                    -110,
                                    -110,
                                    -98,
                                    -88,
                                    30,
                                    -115,
                                    -12,
                                    77,
                                    -95,
                                    -23,
                                    -80,
                                    73,
                                    54,
                                    -126,
                                    51,
                                    30,
                                    115,
                                    46,
                                    6,
                                    34,
                                    4,
                                    -120,
                                    -84,
                                    37,
                                    41,
                                    34,
                                    54,
                                    -102,
                                    -74,
                                    -79,
                                    9,
                                    -102,
                                    -95,
                                    122,
                                    -121,
                                    -43,
                                    54,
                                    -34,
                                    -63,
                                    -45,
                                    104,
                                    59,
                                    40,
                                    118,
                                    44,
                                    -49,
                                    118,
                                    -82,
                                    93,
                                    38,
                                    65,
                                    -51,
                                    -70,
                                    -99,
                                    -7,
                                    -6,
                                    -8,
                                    -4,
                                    113,
                                    -5,
                                    121,
                                    -3,
                                    -120,
                                    -105,
                                    -17,
                                    -19,
                                    -100,
                                    23,
                                    -116,
                                    -106,
                                    -104,
                                    18,
                                    100,
                                    30,
                                    65,
                                    -78,
                                    -83,
                                    97,
                                    -25,
                                    98,
                                    -46,
                                    -77,
                                    102,
                                    91,
                                    -61,
                                    89,
                                    75,
                                    -54,
                                    79,
                                    -117,
                                    -38,
                                    -126,
                                    -75,
                                    -10,
                                    20,
                                    24,
                                    74,
                                    -38,
                                    -94,
                                    78,
                                    -30,
                                    -90,
                                    58,
                                    -50,
                                    66,
                                    -123,
                                    28,
                                    17,
                                    22,
                                    38,
                                    40,
                                    -74,
                                    -73,
                                    29,
                                    -113,
                                    105,
                                    54,
                                    -99,
                                    68,
                                    18,
                                    -92,
                                    -38,
                                    -55,
                                    -5,
                                    -48,
                                    68,
                                    15,
                                    -113,
                                    -24,
                                    49,
                                    4,
                                    49,
                                    33,
                                    -92,
                                    -12,
                                    99,
                                    59,
                                    -52,
                                    -128,
                                    22,
                                    -74,
                                    -14,
                                    83,
                                    16,
                                    1,
                                    60,
                                    99,
                                    31,
                                    21,
                                    -59,
                                    109,
                                    -34,
                                    122,
                                    6,
                                    -83,
                                    17,
                                    -61,
                                    -106,
                                    16,
                                    19,
                                    -83,
                                    -128,
                                    -59,
                                    81,
                                    -35,
                                    -24,
                                    -66,
                                    -41,
                                    113,
                                    -74,
                                    83,
                                    -87,
                                    51,
                                    -93,
                                    29,
                                    58,
                                    59,
                                    55,
                                    110,
                                    32,
                                    2,
                                    35,
                                    33,
                                    41,
                                    4,
                                    -38,
                                    90,
                                    -36,
                                    -46,
                                    39,
                                    124,
                                    -52,
                                    -85,
                                    86,
                                    24,
                                    69,
                                    -106,
                                    8,
                                    -63,
                                    -126,
                                    67,
                                    -72,
                                    -47,
                                    49,
                                    102,
                                    -50,
                                    120,
                                    67,
                                    115,
                                    121,
                                    -71,
                                    46,
                                    40,
                                    -64,
                                    -21,
                                    -29,
                                    -13,
                                    -57,
                                    13,
                                    20,
                                    -31,
                                    46,
                                    -114,
                                    28,
                                    -93,
                                    -88,
                                    -118,
                                    77,
                                    -89,
                                    66,
                                    -55,
                                    89,
                                    -55,
                                    -12,
                                    28,
                                    53,
                                    -122,
                                    -52,
                                    104,
                                    108,
                                    -111,
                                    0,
                                    107,
                                    93,
                                    -78,
                                    -74,
                                    7,
                                    -86,
                                    -96,
                                    -83,
                                    16,
                                    20,
                                    30,
                                    -97,
                                    -83,
                                    72,
                                    114,
                                    44,
                                    23,
                                    -28,
                                    -85,
                                    -108,
                                    78,
                                    98,
                                    -75,
                                    -24,
                                    -54,
                                    26,
                                    109,
                                    41,
                                    -104,
                                    46,
                                    73,
                                    48,
                                    -98,
                                    -114,
                                    -95,
                                    35,
                                    97,
                                    7,
                                    53,
                                    -28,
                                    -100,
                                    78,
                                    19,
                                    -44,
                                    90,
                                    -83,
                                    -101,
                                    -103,
                                    -67,
                                    -104,
                                    11,
                                    -7,
                                    36,
                                    16,
                                    1,
                                    -20,
                                    13,
                                    116,
                                    -104,
                                    73,
                                    45,
                                    -73,
                                    -111,
                                    86,
                                    82,
                                    -97,
                                    74,
                                    46,
                                    78,
                                    60,
                                    -19,
                                    -5,
                                    64,
                                    29,
                                    92,
                                    -102,
                                    13,
                                    83,
                                    36,
                                    -30,
                                    71,
                                    -85,
                                    -53,
                                    -63,
                                    -117,
                                    81,
                                    113,
                                    81,
                                    92,
                                    31,
                                    -97,
                                    63,
                                    110,
                                    63,
                                    -81,
                                    31,
                                    37,
                                    47,
                                    4,
                                    -71,
                                    22,
                                    -26,
                                    -35,
                                    50,
                                    81,
                                    114,
                                    124,
                                    -29,
                                    14,
                                    -50,
                                    -75,
                                    -7,
                                    76,
                                    102,
                                    22,
                                    -100,
                                    60,
                                    -39,
                                    82,
                                    -51,
                                    76,
                                    -125,
                                    77,
                                    110,
                                    -20,
                                    -95,
                                    112,
                                    2,
                                    -64,
                                    -60,
                                    -75,
                                    122,
                                    -99,
                                    5,
                                    -51,
                                    8,
                                    -8,
                                    4,
                                    -13,
                                    88,
                                    85,
                                    -71,
                                    -103,
                                    -6,
                                    -18,
                                    -92,
                                    117,
                                    -128,
                                    14,
                                    6,
                                    81,
                                    -111,
                                    -96,
                                    -98,
                                    17,
                                    32,
                                    81,
                                    21,
                                    60,
                                    102,
                                    -96,
                                    -62,
                                    -75,
                                    -9,
                                    -31,
                                    -47,
                                    -77,
                                    39,
                                    27,
                                    -76,
                                    0,
                                    53,
                                    110,
                                    -118,
                                    66,
                                    -36,
                                    -19,
                                    74,
                                    -15,
                                    36,
                                    -82,
                                    -126,
                                    -75,
                                    -67,
                                    29,
                                    19,
                                    42,
                                    -39,
                                    124,
                                    98,
                                    32,
                                    18,
                                    44,
                                    38,
                                    -83,
                                    15,
                                    39,
                                    65,
                                    -111,
                                    -99,
                                    73,
                                    -80,
                                    85,
                                    48,
                                    0,
                                    -77,
                                    107,
                                    -48,
                                    61,
                                    -48,
                                    66,
                                    114,
                                    -57,
                                    -20,
                                    -85,
                                    -105,
                                    96,
                                    -86,
                                    44,
                                    34,
                                    77,
                                    -53,
                                    -80,
                                    -125,
                                    25,
                                    -20,
                                    57,
                                    -72,
                                    41,
                                    114,
                                    -87,
                                    66,
                                    46,
                                    67,
                                    -104,
                                    -93,
                                    50,
                                    -21,
                                    -52,
                                    -72,
                                    -57,
                                    54,
                                    -77,
                                    92,
                                    59,
                                    -112,
                                    -62,
                                    -106,
                                    96,
                                    -93,
                                    -102,
                                    -110,
                                    -32,
                                    -55,
                                    -48,
                                    -94,
                                    112,
                                    -92,
                                    28,
                                    -77,
                                    -23,
                                    -10,
                                    -27,
                                    78,
                                    114,
                                    54,
                                    -83,
                                    16,
                                    18,
                                    -122,
                                    115,
                                    50,
                                    -60,
                                    79,
                                    64,
                                    54,
                                    -112,
                                    73,
                                    -88,
                                    56,
                                    -25,
                                    -71,
                                    -66,
                                    107,
                                    -51,
                                    -119,
                                    -79,
                                    -75,
                                    123,
                                    -125,
                                    115,
                                    28,
                                    110,
                                    -23,
                                    -128,
                                    92,
                                    36,
                                    -88,
                                    41,
                                    93,
                                    14,
                                    65,
                                    85,
                                    -21,
                                    32,
                                    54,
                                    91,
                                    38,
                                    68,
                                    -54,
                                    -90,
                                    -27,
                                    -52,
                                    -89,
                                    78,
                                    -37,
                                    -103,
                                    17,
                                    31,
                                    -53,
                                    -48,
                                    107,
                                    89,
                                    -89,
                                    -124,
                                    55,
                                    73,
                                    -76,
                                    66,
                                    -64,
                                    -42,
                                    -93,
                                    -109,
                                    121,
                                    24,
                                    70,
                                    -64,
                                    108,
                                    -53,
                                    0,
                                    58,
                                    -71,
                                    -11,
                                    -48,
                                    -103,
                                    62,
                                    -87,
                                    123,
                                    -97,
                                    22,
                                    43,
                                    -126,
                                    -32,
                                    88,
                                    -114,
                                    26,
                                    55,
                                    67,
                                    -31,
                                    -121,
                                    -101,
                                    -111,
                                    -27,
                                    -70,
                                    51,
                                    -33,
                                    -110,
                                    -118,
                                    91,
                                    24,
                                    -118,
                                    -86,
                                    53,
                                    79,
                                    16,
                                    -78,
                                    68,
                                    -17,
                                    14,
                                    60,
                                    29,
                                    103,
                                    -30,
                                    100,
                                    -39,
                                    -79,
                                    10,
                                    55,
                                    -82,
                                    22,
                                    29,
                                    48,
                                    -77,
                                    22,
                                    11,
                                    -47,
                                    102,
                                    22,
                                    6,
                                    -68,
                                    -120,
                                    -63,
                                    -59,
                                    -24,
                                    72,
                                    -85,
                                    -80,
                                    124,
                                    -104,
                                    -32,
                                    -104,
                                    -127,
                                    -100,
                                    -56,
                                    -111,
                                    45,
                                    21,
                                    13,
                                    20,
                                    -72,
                                    100,
                                    37,
                                    -58,
                                    57,
                                    -67,
                                    -126,
                                    -80,
                                    1,
                                    88,
                                    41,
                                    -117,
                                    -96,
                                    3,
                                    41,
                                    101,
                                    61,
                                    77,
                                    -68,
                                    -106,
                                    -85,
                                    -124,
                                    21,
                                    -110,
                                    29,
                                    11,
                                    68,
                                    81,
                                    -37,
                                    121,
                                    89,
                                    -125,
                                    -63,
                                    86,
                                    -94,
                                    46,
                                    54,
                                    -37,
                                    57,
                                    83,
                                    -103,
                                    29,
                                    109,
                                    -78,
                                    -88,
                                    -101,
                                    -92,
                                    74,
                                    -55,
                                    -80,
                                    115,
                                    111,
                                    -124,
                                    -113,
                                    48,
                                    39,
                                    -39,
                                    -27,
                                    108,
                                    -42,
                                    7,
                                    39,
                                    -85,
                                    122,
                                    109,
                                    -127,
                                    25,
                                    105,
                                    -38,
                                    -91,
                                    -25,
                                    0,
                                    74,
                                    -90,
                                    42,
                                    27,
                                    23,
                                    12,
                                    3,
                                    -106,
                                    72,
                                    -118,
                                    -32,
                                    9,
                                    -64,
                                    -123,
                                    -103,
                                    -32,
                                    -52,
                                    -22,
                                    -49,
                                    -73,
                                    -21,
                                    -52,
                                    -4,
                                    -10,
                                    30,
                                    -81,
                                    -9,
                                    -88,
                                    -57,
                                    -13,
                                    -37,
                                    107,
                                    -86,
                                    -43,
                                    21,
                                    -17,
                                    -113,
                                    -25,
                                    19,
                                    -11,
                                    56,
                                    -124,
                                    -13,
                                    -10,
                                    -31,
                                    -65,
                                    38,
                                    126,
                                    -75,
                                    71,
                                    10,
                                    -37,
                                    115,
                                    -62,
                                    108,
                                    -9,
                                    57,
                                    -1,
                                    123,
                                    -83,
                                    127,
                                    63,
                                    -49,
                                    -97,
                                    -41,
                                    63,
                                    121,
                                    70,
                                    -33,
                                    32,
                                    30,
                                    2,
                                    13,
                                    58,
                                    -86,
                                    59,
                                    37,
                                    -67,
                                    11,
                                    -105,
                                    -87,
                                    -53,
                                    -2,
                                    -72,
                                    93,
                                    63,
                                    -26,
                                    -67,
                                    -105,
                                    -76,
                                    46,
                                    -8,
                                    11,
                                    -4,
                                    -27,
                                    49,
                                    -98,
                                    99,
                                    -1,
                                    -112,
                                    -52,
                                    106,
                                    -51,
                                    50,
                                    -119,
                                    -36,
                                    -67,
                                    -96,
                                    33,
                                    -93,
                                    -105,
                                    -102,
                                    0,
                                    112,
                                    -74,
                                    -80,
                                    1,
                                    26,
                                    65,
                                    68,
                                    59,
                                    -71,
                                    50,
                                    112,
                                    -93,
                                    -14,
                                    90,
                                    -79,
                                    71,
                                    -113,
                                    -18,
                                    -35,
                                    126,
                                    -52,
                                    101,
                                    23,
                                    -83,
                                    -19,
                                    -35,
                                    53,
                                    -59,
                                    -95,
                                    24,
                                    -106,
                                    47,
                                    -38,
                                    54,
                                    -118,
                                    -79,
                                    -68,
                                    29,
                                    91,
                                    -121,
                                    24,
                                    -49,
                                    -63,
                                    100,
                                    -29,
                                    50,
                                    -13,
                                    -60,
                                    -27,
                                    99,
                                    58,
                                    39,
                                    -124,
                                    -45,
                                    -114,
                                    28,
                                    47,
                                    63,
                                    56,
                                    59,
                                    -45,
                                    -123,
                                    -82,
                                    -113,
                                    -101,
                                    -77,
                                    -28,
                                    -76,
                                    -24,
                                    97,
                                    62,
                                    -63,
                                    100,
                                    -78,
                                    19,
                                    99,
                                    65,
                                    -54,
                                    8,
                                    74,
                                    43,
                                    42,
                                    53,
                                    -79,
                                    -45,
                                    70,
                                    -111,
                                    18,
                                    62,
                                    81,
                                    -20,
                                    85,
                                    123,
                                    -113,
                                    96,
                                    52,
                                    -98,
                                    51,
                                    67,
                                    10,
                                    2,
                                    99,
                                    -54,
                                    -63,
                                    -51,
                                    45,
                                    75,
                                    65,
                                    14,
                                    58,
                                    27,
                                    -20,
                                    -96,
                                    -80,
                                    44,
                                    -111,
                                    -107,
                                    94,
                                    36,
                                    -112,
                                    7,
                                    -64,
                                    106,
                                    88,
                                    -118,
                                    -12,
                                    32,
                                    -80,
                                    114,
                                    -39,
                                    108,
                                    74,
                                    15,
                                    -49,
                                    -83,
                                    -88,
                                    -42,
                                    -66,
                                    -77,
                                    72,
                                    40,
                                    -34,
                                    68,
                                    15,
                                    36,
                                    48,
                                    -99,
                                    89,
                                    -75,
                                    34,
                                    -48,
                                    6,
                                    -63,
                                    120,
                                    -99,
                                    3,
                                    -87,
                                    96,
                                    -124,
                                    123,
                                    -25,
                                    -119,
                                    -67,
                                    -46,
                                    27,
                                    84,
                                    56,
                                    68,
                                    -113,
                                    -72,
                                    -44,
                                    106,
                                    -31,
                                    -20,
                                    96,
                                    67,
                                    -42,
                                    114,
                                    -9,
                                    81,
                                    36,
                                    -82,
                                    24,
                                    -38,
                                    -37,
                                    27,
                                    -104,
                                    -52,
                                    -25,
                                    12,
                                    -55,
                                    114,
                                    8,
                                    94,
                                    -60,
                                    9,
                                    109,
                                    -69,
                                    -44,
                                    74,
                                    -125,
                                    61,
                                    73,
                                    -71,
                                    102,
                                    41,
                                    44,
                                    119,
                                    -20,
                                    -104,
                                    53,
                                    76,
                                    49,
                                    114,
                                    125,
                                    124,
                                    -2,
                                    0,
                                ],
                                "fee": 500000,
                            },
                            "proofs": [
                                {
                                    "id": "49496e22cffa3314958aa12fc16c658ce5ed0a8da032823a6100c39d7ef5198221888350c1d40cd35c8892d779e9d50b4c8a0f542168c6a586967b1c6dd5b153",
                                    "signature": "304502210092079e1d99e27ce9578fa4c55d760be58ea64fc1d6eedfd550e28e854e9e9a0d0220133f3bd1d95f57951872046f15fb290c3f61b7f470562928533b99087085680b",
                                },
                                {
                                    "id": "a2496d09b7325f7e96d0f774c8fd45670779ec0614b672db23d2cf8342c13aaa6a64146996f8aed365eb12412c9b39f2eade3e45f610464212b8f69a660271d5",
                                    "signature": "3044022062212f32742e5faa5a0c132bf8ac1d4dea1459c8be7256987751ccf2db96e188022051c324a47b4767a26871ca85f711d99722a8ea2cd6b715bacf933adc0ff97a90",
                                },
                                {
                                    "id": "d741b547225b6ba6f1ba38be192ab7550b7610ef54e7fee88a9666b79a12a6741d1565241fba5c2a812be66edd878824f927a42430ffba48fa0bd0264a5483bf",
                                    "signature": "3046022100cbb97cd67251ed83c55af1a7cac9fe4fd8eb245f03e78a0dfef0adf252b5a6a2022100a927c871ebe97a1da6a917bc67ec803fce3e7c7e66e85af7c7ba69543d879397",
                                },
                            ],
                        }
                    ]
                },
                "rewards": [],
                "epoch_progress": 765850,
                "next_facilitators": [
                    "e0c1ee6ec43510f0e16d2969a7a7c074a5c8cdb477c074fe9c32a9aad8cbc8ff1dff60bb81923e0db437d2686a9b65b86c403e6a21fa32b6acc4e61be4d70925"
                ],
                "tips": {
                    "deprecated": [
                        {
                            "block": {
                                "height": 346,
                                "hash": "095df6f2c7c6100f4c4d613f43225628b5c1d39bf29ae52944b45e8e7f107153",
                            },
                            "deprecatedAt": 3372537,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "1991fdb2e4dd57c34d85ea5132f340d828cdd027c900a5bc8814876c9bd1237c",
                            },
                            "deprecatedAt": 3376564,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "3a17dc6b1a527792985c1463e6653b09f277a2cfc9d44511e20f24a06c7a15e0",
                            },
                            "deprecatedAt": 3374627,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "696382d147fd1e22a63efe55c1b3402f0f3443f28197264e697bbd68f1fc5b59",
                            },
                            "deprecatedAt": 3373425,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "88937bd01a698bd68d3ada48ceb046720600add03565fcaa65b18997ba5115a5",
                            },
                            "deprecatedAt": 3376564,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "a2ae2fc4aaae5d48c497174c82334630ae7a8d732fcc625d679697022b0af647",
                            },
                            "deprecatedAt": 3372673,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "b308259a3e272d6b96a2611e7874afc04ddba7e94a7a47b58c3e70901a77d3a7",
                            },
                            "deprecatedAt": 3373425,
                        },
                        {
                            "block": {
                                "height": 346,
                                "hash": "c869b6ec210db99303a31d81b9734fb77a17f7329adb0048412c3fe7ef8ba3aa",
                            },
                            "deprecatedAt": 3374627,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "ef96e384774bb45f6b093c4b2279414196de6e1babe43ce9bd5c92f9603fb755",
                            },
                            "deprecatedAt": 3379320,
                        },
                    ],
                    "remained_active": [
                        {
                            "block": {
                                "height": 347,
                                "hash": "1bd4cbc605a0f20bd0f83d0d5703a061947a15839630ad475a893943c99e9d7c",
                            },
                            "usageCount": 1,
                            "introducedAt": 3372537,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "2a974ca5535542a04377d644a70a662bb514b848598970cd10d2eb6430ee542c",
                            },
                            "usageCount": 0,
                            "introducedAt": 3371529,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "588f49ab72d255ed3724f7d0aacc0aa414c4bf73360235e3c2f49d7815e274e1",
                            },
                            "usageCount": 0,
                            "introducedAt": 3373425,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "60c190b3f81e751eaf161efc5aa6e3f0c9ce2e8b13103f5244091c92e9bc96ca",
                            },
                            "usageCount": 1,
                            "introducedAt": 3374627,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "7cc4c31684ce127d5d80b3f5ab53c4ae87d3040b2477baee0f600af45b64b9e0",
                            },
                            "usageCount": 0,
                            "introducedAt": 3372673,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "acaba4022b1ed3606ce405c9d51a1df4f12792a4e401e3a47ff9603f1a8821a7",
                            },
                            "usageCount": 0,
                            "introducedAt": 3376564,
                        },
                        {
                            "block": {
                                "height": 347,
                                "hash": "dd7ff843e92423dc5807db7a9c1d9d50d48a77316f045c744778dc64a47776e0",
                            },
                            "usageCount": 0,
                            "introducedAt": 3371305,
                        },
                        {
                            "block": {
                                "height": 348,
                                "hash": "12da9b2ae4f9e94e70751214b05bb6b93b03f7db3faaea9e143385fa3c556155",
                            },
                            "usageCount": 0,
                            "introducedAt": 3379320,
                        },
                        {
                            "block": {
                                "height": 348,
                                "hash": "3dcec3fb0c975cd5ec44e1d08f2071301ad18041d96154e53d5a96d180c9ee17",
                            },
                            "usageCount": 0,
                            "introducedAt": 3379315,
                        },
                    ],
                },
                "state_proof": {
                    "lastStateChannelSnapshotHashesProof": "6a64c0cc461d0bfdc3c958680647ecc57bd1258b1b0185bd435e0163f0ae8827",
                    "lastTxRefsProof": "1b1f20d01230bdbaff7a531f6afc327146ce777fe7e66dfb5c02b767d363a261",
                    "balancesProof": "e1f70490540d6afc06a21566414da85fa48a5fb112e63143a6df69f19e1a248e",
                    "lastCurrencySnapshotsProof": {
                        "leaf_count": 60,
                        "hash": "8f27acd704c9a7652708f6ad3947a27d7f053d787d203c6cdf10203d190845c3",
                    },
                },
                "allow_spend_blocks": [],
                "token_lock_blocks": [],
                "spend_actions": {},
                "update_node_parameters": {},
                "artifacts": [],
                "active_delegated_stakes": {},
                "delegated_stakes_withdrawals": {},
                "active_node_collaterals": {},
                "node_collateral_withdrawals": {},
                "version": "0.0.1",
            },
            "proofs": [
                {
                    "id": "003a68d1beedcfa3cbda378b48a620b81aec7735a04f90693ee99adccb547c2b0b9863455ee7d0bec5133095fbd357048da73c4bc19531b8938375f1990a887e",
                    "signature": "30450221009689856c6251611258c90e55c0b55b73611b46c10a23275e786b9e0cf85604d902207bfb5ed6bdb11e41e2304697a6b20fca470d721896739239a3713478b67b8c60",
                },
                {
                    "id": "021d126e817fbce4d7092b5f759c29ef95aafc6c136f29baf834cd051285c430f3d5f51296bc21318d6d3206b3f96083fb4f4fed9e1989655143717ee119bb84",
                    "signature": "3046022100d23ca6ae03a508920ce801838116595b9bfbefeb9d5a0fe17eb359c8da4f95ea022100ec188e911d9da9ea7e0ca3e4f6f18a170dcaa825c9ae194a5c784162f46649c4",
                },
                {
                    "id": "033f7a98bc68c97cc91f906cca49bfd600e1d9cfc510f386bf46f7950cd67478c459c505ce283ec5ec11910b319982f978fd4438de660edff5d7b23e342a4f76",
                    "signature": "3045022100eadb59490b8fb1ff7f9424f962366c19ea7599b4522a89fe5bfd8ac22ac3d86802206a09693154de61e90c247e5a36ae6fdc52f33e439ff7a98b9d15534fd2d4a9af",
                },
                {
                    "id": "03620bc96fedba75e379fc1bf3157d1f5d53e5020c4793da2c78a159338050a56ef4c01bfc772612c1d0571868d649adfbd3129975455f2f36375f5651a31d3b",
                    "signature": "3045022100a156ac82b3d63aa2bcdc7ab06c04c86b8e7b808987300c8bfd29b1f3d0c718ef02207eed399bcff40808a21c3a314a09f7d972a8a9ba583b398f379a2dd213383183",
                },
                {
                    "id": "0388667180d549c8e36ab0a459f57f8059f402f9d0086ecb6f39ff1ef4409e901c404ad1819b432b9f7cca97bf735f3d76fe9fa6b873ea2d0e7e83d19ae21e4d",
                    "signature": "304502207e070050d8dfa5f25a20b8ff845c1abb043d88aad823b94ec2fd7c09b5e396cc022100e4ccdbbcd599e0b486146a55499eec4b704a064daa360d4405df615c1ff4d661",
                },
                {
                    "id": "0407a91f44e86efe548108defcbd2fda4b974a90faf13e4465482edcbf550297d89e81c0c69ed7fbdddb3623a940ec21b73aee5dd21ddff4d9ce047731f94e0d",
                    "signature": "304402207e08d909f6cc3556cf743471bc29606eb52dc1025236dae1620c2dfd42cc9d3502204fafce7bf010d0d53d7fb7c13c7b3739194b4c837fb237dd75ffa84810cd05a6",
                },
                {
                    "id": "041855d1b7d73f0906314abc3231a750d62910e24761fbd50ad4be887d708bf48fd958f6ec88aa7851dd645fbfc98a22219e1bb9db28bcbca05ef53529062516",
                    "signature": "3046022100e6f3d9d7bde794a7e3f757159f4d9e5d496faaaada99b318b94743dd133ba2cb022100b5cdb1bfbe6d352f6eb1bf2607a266de7ca26f87387ad0f89751158a4a73baa4",
                },
                {
                    "id": "070a0de46e6552210027ca7b1de995bc83bae0140c97085a9983e4d184749c56535deea5f0e794d6d42e4fd0fc89d85610063165907b4591e63a7b7929308c34",
                    "signature": "3046022100f4d41190e52a1001489c65a37a1d77b4f4a410fb57430872c32656843e4b9f7b0221008282fd53dd4eb35e9b9035ff9abef233bb712586725de37db732ded97a274f7b",
                },
                {
                    "id": "074d9710028e7f0324848ceabf2945b9a9441a584c033933804c9559184a64dd67411f2b9679daa01b280ff53ca01c5ceef6a1474f64e5b97d737f2ecc732196",
                    "signature": "3045022042ea2cd54b8cce6e1392d71c929d5b0a1c060b50842a774d603fcf5d33f49635022100ecab4a3222f753d016d62a36d28d86ba8af11330618f3ad60dc66b28e8734194",
                },
                {
                    "id": "0a81f11f842ebeb0fa2b1818df70c8636f8f7eca6c6917063455369d34191c1028f18edfd2b2cf7ea5fcf47d0a54d9768c46c24ba1385a8d268f660042ae9ed5",
                    "signature": "3046022100d85dcd9cd271217ec2aaaee07ce7e93e152c394d159b7c4838b0d874eea5d02a022100c89d6a7d9e4a5cc84d3858c32bb1e0104b1b22e7662d3c6768a579bffa4fd094",
                },
                {
                    "id": "0abdc87966b4add2d0a14f798f9050f024f208d0e7cdd6f1274c3a6e9043237af7341259051da1a194071a4778134f5696a7a4468ba006b72b0f757c16491215",
                    "signature": "304402207d2d7a078d239ed69ea9be0729b38ab1f803517d4bd940928d14547daa4375b6022052fa5ec034d76553783570fb003f11d95fe75d30dd806acbf4cb00d5f139c74f",
                },
                {
                    "id": "0bf1800320b5033920028c165256eb0cafcdd29a33e0190a77569940e25fd8efcbe41cca489e4fc275b129c928a552033927617d87681bfff03d035bf723b16c",
                    "signature": "3044022017d1b7f5fe7cd849065105ae52c09f15466edbf13f3a7fc9212847a4c4d8ed9a022077fbc40ac4aaa3c818887d72f8ef0a56e8d26a3ce47600cba1419e883077a0ec",
                },
                {
                    "id": "0cb9294a7a3304653927f251ce7ab9541e2a066f81734ffae570ac5bb8fcb7a7c71074421d756ea43426b448b102345f9b9122f9d7107b01b18b3861745dcb31",
                    "signature": "304502204842b9f1a7a44f96848ddc55727589777fd49e9c556df7cd028e3996d12c776f022100ab9b75174a1eba3b1736b8872ca635192e9fdbf18dc6a4f6190e99d288cf2351",
                },
                {
                    "id": "135de5b0eeed1c006fb424b3870e8f2e43209e34421dfd35e0b389171b7da1825ae29d311b93420a677213dbae41b5537d06ea9727fd9244c9af6ea4db501d7f",
                    "signature": "3044022012d2f15f569753e1407a1bf6260494c66800ce719de33f2be40186a27119213302207e6278ac8bfe86c6915b1448101d09a6d4ca346c44cf438df4cb5c70b7df3986",
                },
                {
                    "id": "14333ce45b59571b12abcb28854663540b64f5e2ac0b5320dcc8511e40fd0f0aa2248869ffce8c035af5e29412dda94fe70d262c0a3bfb1e250d159a388682e6",
                    "signature": "30440220765f2e4b80eea608317afee1d969aeecb77bffbe26132a3c2b689b52d661284802201c916b1841c4642cd5254442ab6d4f01221ccf67a513f199951de00a7c89031a",
                },
                {
                    "id": "153cfc2b70c6781c61026459df5b650357d6cbf1f1383cde73f23e9509f6e49535d602ea3c77f386a35b2a3a557d06b6d2125fc7a4fa6a633cf05621f2856a8c",
                    "signature": "304402205812b794e56449825b2b24558293ed5169115a6cc7754e25f51aa13cf66cf122022068e532563d800825ece227769cf50331da766d647a4a05536c26d777f9fba953",
                },
                {
                    "id": "18d004e6ca93743313ecafd3f060020540065c38a7c13e5479f31b289ace4d3ff68011b10606b46e6af204775a1caf60ac1e8ccfcf26f756deff9748cdd2a2b9",
                    "signature": "3045022027ab8255d1c7104ac5dd08248059bf4ddcfe21a77b5745fdac5fa3f853887ba5022100a5084399c0a04703a2662c6a8d5d28178e7ec24aa4fd9018cdb718da420876b7",
                },
                {
                    "id": "1b5408bd0e32da4567bc0b9793a602f9d5a768ae29706c778e19638e827de65a34bc8606ec7939c1f01348e22d8bf22cba1961ac23837c2d509ca22323da8a57",
                    "signature": "3046022100b870f9c9cfe8485123819fb74ede8a8818ac8932688914843246a94e3a03966e02210083ee4de431283a64c4a7ea73dea1562bf0b6870955599bd3b0de000628ecf991",
                },
                {
                    "id": "1b9abcf35398c95428b2d47f5bfa2ce21816a837f52b562e53b283f8b00cc49761ffbf7fd25e050e8e1a60dc2e490c3dfc053bc595a1e2f002c6df5cadba93d4",
                    "signature": "3045022100b504998b19c08d2fe31028c83595c5b9c53b77c0b6cc082b70b80f1dc2f520f10220736ef5cab0854c38832855b2ab789aa0ffe7a8030e72f88d46d67e78c4653081",
                },
                {
                    "id": "1bcf21b5d50b15e0e640bbfe47645d94212f2189def14a70ed3a20a28afcf4c611b7feb8839973b7aac21d661e4b419b75610f50bf8b6e12a316b9d8d4221589",
                    "signature": "3044022009cebfa8c208ac344b7b542f55905c87275544f8238b8fb01f99aaaca0d43aec022076c6505fd4dce3f3732ec116b7b9e84c618a79a7d233b499a60f8b318195ea48",
                },
                {
                    "id": "1bee0bdbae4163e098ca6d684ded1ffc065bca1f9f9cc1c8a1bfa81025116a86b4945553e50acb63ca22cadc94ad88fe24dcc42876f25fb6d0f34a89744b4028",
                    "signature": "3045022100c215f0f86cfceaf266e18a3369ae6bff8936d52dc729151e9a684b3dc9745525022045a38226666cd71614d8df5bd10f91835bcdd0ed3f9084d167e1facb72406d9c",
                },
                {
                    "id": "1d9e854801a1efb6662b0dc269bacc82c66b2158126f572229abe42533a9ff35e26150066a401c9d56bf43edd6d5d72cc19a56b312485d644945b6521cce5b39",
                    "signature": "304502210088fbd02bd551c71b1ac27b7df4f1939edb23662013fe85cc79112788e431b57f02206ce1729661034aeab149042ef65bb2ef074752390b5b0e9479cdd81ccd09a08a",
                },
                {
                    "id": "1df54a54c32b805ee125fce24a2a65dcab623a64bc24c1b2a18b96ccd4b6e637163c227a57a170f3af0f527e93aa48b15c7ecef2e9716403c5a0017b2c84b50d",
                    "signature": "3046022100b62261fddd3bbcac9b1e5e24bd64c407eec4a3a21563267a081c312ef2e9896f022100a8a4be6f9e6a3933e9eb6d60a44ee927b22254b5c3189d159f48fff7ce4edc44",
                },
                {
                    "id": "1f1e8c7b87ced63edb417c330a1a6669b87eaa64d3a0b920c54afb9f7efef37a7250c68f9ecb449fd864af2eeaf616ba0e26cb67c79b56db09c984b780d8cf50",
                    "signature": "3046022100c97a0673f1ee63d3387de54c9e68cee23f3578b37d01b0f16e93a27bb33f1dc0022100ef68aa26bb9b7aaeeadfd9d3d40ae74bc76e38ddb768137d842a5c5fcfb711c0",
                },
                {
                    "id": "1f3d00a0aa002b0fd98b78a726e7d2d8fee93cde0a49ad8e7d1ec580ea2f758c717cfc82defe67eac42702dcaedf241a842ee2aa9a023377d0fc24508cd77c8f",
                    "signature": "3046022100d3ab2f07de5017cf4a3dc5b46d4f0b3f27032625dcacca003010874fa521a26c022100847bdb90372911d925abddcc8ed4924ae3a9b361bbc88f72eb2fa2dc11a77297",
                },
                {
                    "id": "282deaafc1a71f828a9ec9e4500055f5cea7632d7b435bd3b206c3f06a7b551cbea09c5d2c6143e9c1825fc7a5b82f36752a822770dc2ee68eb06895347f0ffe",
                    "signature": "3044022038a1eaf83ad5aa0739ded76310b0bcb7affe734645d46f1df878bc74d2861ec202206c128dc15d89bffcaa7d029b19e4cf0ec79d8b0cd51863f8e271241b591629c8",
                },
                {
                    "id": "28d023de73edadd0a0121096513b3d4076d43c6fc4fae14c902eb462a66abb60003ac4b956609ca879c6a7b8450e7a5a1979e3437da8f0dc53ab629534918287",
                    "signature": "30440220327f41ea47b09ca3126de3853e5ddfee122ac54dfd2c4714808cf9abdc8c3c56022049b7bfe40fd3726eb9fdc79836d00457c8fdc493045f1cd6886e2e993de90f20",
                },
                {
                    "id": "2981e43256edc9fc93723c068b0815d68aa714be7c8f8d8d4dd397d4dc28271863a21a78a6e7f9254a8c02c347d9a38fc776b00680cad91a0ed5fcbbdbcfe76c",
                    "signature": "3045022100fdde4fb14fcec8552dc94b9c4ea69fd13008ce36220b59610360f686c59e3379022057ba19d08bc29ac09b3a5359e033aecfe6a43995ec706d8bb2815ec769b49453",
                },
                {
                    "id": "29b0dc8d310b0403db818db2da8d187127cdce14a825d3176534b03537e8573eaff0f95366617074996322d64f52b7a62e2737d77f7fbc888accbf4b98f7f651",
                    "signature": "3045022100f72f47bcf77d507c36ff87387ad7d404047d58b76ec774b02b29d8fb299ba6fe02201f4d70c0201163c8280b11b691f00a0d176a765b0450f1c67b7875f2100666dc",
                },
                {
                    "id": "2a402a5c7b09e0910450f4af48fd9fa45e6cf3f44c88047e74154a37aa53f14111cfee0d7899d98a11fee16c828a203a7aedce5963ab93b0f227c51ac50d8fee",
                    "signature": "304402201989dbee848960c2820b46b1596d416da35754c6fe713e75d892a4590947048e02206c2915a82ad1de978dac8bdb257d08e3e8c3b4ee3d16173f943aeb64db74e6d8",
                },
                {
                    "id": "2a7d0ec4b5fe6bac4cd7ebd39bc725c2c797fd1b29bbd246a4f3a3887de6482b3b774af7352903df1c9c32f559459f7f1fa7e116fb509e0499f8bf39e4e47b0e",
                    "signature": "30440220325e10d0fd068a37d980c901dc0d4e747af8d888941e9fd370e925d48f10f8d10220403d81092370c8ff21bd8964f96fdd8be671371c4462a5581f9560d36f219829",
                },
                {
                    "id": "2a9e74a9c79e2044915c1ab5b7f18d461ccb0060473a1ac9d5a3f9b19b8611ba92518dad1364e6f4f3221c993a0132bdcb24ba8910df54dd730a6a6608951a15",
                    "signature": "3046022100b27a7b2ed37c692758085928e2e3e0551ab4115c038f5201e4c6bad57c388e05022100b3361e9a91fdc916cb08daf62f9285e7a6e6a3f813d2d0e99b296711717ffa9d",
                },
                {
                    "id": "2b788d55f19d80d481f9e1eee6a198309ca10920129552c3d8f9762699a0cb7c18c1f235aae9092c0d74f6896016abbfbc347a75a72e0c05851920d14531295b",
                    "signature": "3045022100dee5be54a42f5069ba9a07144388b2f704df91f5ffeb979499c78e71ec6064bb022066cfb39cec5b22e7be905dba4feb80c0a677e9ae90e56ff7389e3b463e69af6e",
                },
                {
                    "id": "2bc31a889107ad8ad7405cdd9b1f0621244bd574c8b85e9f8000e494d5020e7370ce24a7ea641543148d75bb7b11d336d66c43bf763885854b87acd28d1124de",
                    "signature": "3046022100e19baf13b11a52a07b757afaf36d06ea13f83123743556574481ab494901a403022100912476d4bf3a05c363b93f6ef5b7069f7050ddea992baed0fdedd272d1a92ed7",
                },
                {
                    "id": "2c03a2651b41b73f0ce3901646d23ad90799b21c8b02232f41177a4e6d918bec0bdaa260c705aa6979c3797d0884b89760bc15bcdc11de27e42f621da32e22b8",
                    "signature": "304402201f3c567680560d77bf4f807b445c74b8da58c77966e787dbc74a65ccb69ed99802205a9095447c15b1f4fbe8fa6cedd581e1e899b976740e97df5db9f7ca6e6708c3",
                },
                {
                    "id": "2f799c55d8e273db207a528e22187cb6458d79175414fedd9b87fc648b80f253d2ce532d021ad3a43fdfc960ad7212f03ffed1eadf011eddb4532463d67ca5f0",
                    "signature": "30440220484e8888a83ba3771c5dd287b6c143bb46da40ade8a0b4ae25280d8f5a2bad74022019d3be9ec3b2739ba56bb9aa0fdbe27e4275ff5540cc25655ead502a70d8ea27",
                },
                {
                    "id": "3122ec0a9107223e55e06205499f9e50e76d14edccd3fd219bbd48b42a43e55dc2c883837b5a17e1e71d21ece2ddb3c1bfe19d9edc259c4a416de3e2e7332380",
                    "signature": "30450220180d27b2661ad2f54bf488189ad069842bac91dc438403d9d5b09318d677776402210095b91f1ab224407e997fb752bb0f4029661f753edf8e10bf3a4f9ae2c3abf0c4",
                },
                {
                    "id": "31a705b94d4f93bb22dad0696c398f5e057dcdd6c225fc7906ddd2f2893e884bb2951298ffc28d849fd53ebdd964eb58df695dedac35e01292982bdb0e4849ee",
                    "signature": "304502205165816438828e79b64bda86b52d7cfdb2281769b40f8bdc7e8fdf85f6d263290221009009279453911a478c67ed4fcfe8dc14c79f7f4f3db7fb2b0e0568452169766d",
                },
                {
                    "id": "31fd863de2e34cb7d4383b012d665d60efeb6cbf40ae1b5088708f2a9a5f5a34b405b0acac136588a90d576d8d97a2bc3a5b91365d5fe28a3d8fb60abd314e25",
                    "signature": "304502204b964bbd444b98f3fae63648eb8c38f353968d54e45401265c56a7e48cac7592022100cecad277c16ab5534e112b024a416bba49e221deb75af7e8ac803332942701c4",
                },
                {
                    "id": "320907ff40bd50373fa8bbf81c4f4aec953f723b74d2dffccfd23c4b3f1c1612f1b1c3ba1f8a6f8d79b22f5329d23dd976e8e4f467182b78639f1d25f3e05ccf",
                    "signature": "304502201c3715837ad52ea8de3554c27943be17de72fe8ae618a1f8e1006b62217907070221008f7e2f2c2f52f585b5e961151b1b32698516cf9be9087743da9803421e2b1b18",
                },
                {
                    "id": "3458a688925a4bd89f2ac2c695362e44d2e0c2903bdbb41b341a4d39283b22d8c85b487bd33cc5d36dbe5e31b5b00a10a6eab802718ead4ed7192ade5a5d1941",
                    "signature": "3045022100fbab2be9180717ef4ccb96d01c294b77fa094e82e4f7a632ce7174c29c1132730220332b2196ca5f24ddec445603681023846e9e4dcd882038b30931fddb9bb702a0",
                },
                {
                    "id": "34a127d8c68cc342b5212d676a9ae77e968c73545a5c679dc6e19684f0690352921fa6b812882e420f89c68e7c891e556c94989d0eebf3315ebad03944662a5c",
                    "signature": "304502206d6e87ca37d49a9e651611a43e252ab7b240a4ebbb489782465829b0cb198807022100c4c031612ff7feea217d743baff491b5d719b8b1e335f2369557354b9a45c6b3",
                },
                {
                    "id": "38a139e0c3e4606ffb3328a9a9d9490cb0948a8bad538a7ba4e63e5d067fb404640d1646878a808b42d29e5485924bb3ccab84d922196e599c37ba9a9be0da77",
                    "signature": "30460221009c97ab32e4d1cac442b5c13444915baca0ec91cdda6126e38c2592ebb57826f20221009e6bb03feecfb5f0839234a2cc5acce00db55f535a62a7470acfcf522e1e75d4",
                },
                {
                    "id": "39427dca334a801eebd06fff8bfae59a1d1bebdef78eda1ef7baff0ced24e4149de7963d70e1827ba2e53e37f639ef294a6c903969b19fa37b6799ee3dfe262a",
                    "signature": "3044022071de52608d476cee9d0ec4cbed9dbccb9668a94555d2a3d5e309bb6db54dacfd022026a6aca91789efbf0ff5c4762e58dcf528663ab16b07f911d1c6b6358bb64a8a",
                },
                {
                    "id": "3a3c367eaffa5bb3c4b35404e95983c73c522d166010784cc8d69ee9cc9c415c11534d331b70fceb7d445c96dcf62021bb8d7762a81506a0b616c45a32f00055",
                    "signature": "3045022100acf6858fb1a2b291c7abcbf0f8d107885bbb976ace014582a0ea7991bc841e76022004d17a8ac8ece9429c9e6bb2ce327f3f19e7d24bb3ba4d31c93cb6a821fa2bc6",
                },
                {
                    "id": "3d865050cea8295b1d2b6666281f99298ebbbfd2400015bcea222787ce6f4ce44aa49b3a42970fcfbe73b4a00f198c89174bae3adc384449c41d716a8db04faa",
                    "signature": "30450221008a3948b9832203a8df33682cfe0b3dc1b00d266bd75fd6322b9fab38e35c515b02206f89964ac8eff069a1bf5d2bcb5640804137a2f7dd33938a71d4ee3baf65379f",
                },
                {
                    "id": "3f4644fab775bdc5f4a54ac4f2e6ec0cdc0318f462565e518b04d8f121f20f6d053d8966ba8093d638a0c036c87d4d03bf44c81783eb9c092790892d36c59314",
                    "signature": "30440220497c9c52175f29cd8fcb0f44996a2f82b6f05d23a0480838b53b25a47963ce0002202ba0501e5fa6cefa7f2b6d86a0cf89d015110a2aaeb263564e3676ca139da7cc",
                },
                {
                    "id": "405515d8640468233ea736be6a554b4de3fea5cc47d1129a4f8124d31a9653a6787a0797dd1c6d4102f8d8af079c4978ea330a0e0c8df78d87d93d4d28ba9590",
                    "signature": "304502205f151bbc47281979f9b6c5355ffa19407b9abe130c9c3ff1a12031f6bcc65e5f022100baeeafb1afc2148a11f81ffb2e544b21c87397f2bf30b02fa2b2a052270a6174",
                },
                {
                    "id": "4084bb8217a0ba4a07a3e30a78b1dd3d7a3aedfff0911cdff21a94b825512e647e12cb5c517b36870ed455a84c70df1b36b0cbfd7e2d21d1fb3dfcc616d03668",
                    "signature": "3046022100d71359961ad57adab4d247264a1367375dfb346080fab60c00fb022dec5f3eaa022100e6ef582a785c509c2689fe8d157a23d6e0efa1cc05719c1f8a0aefe76b2a7c92",
                },
                {
                    "id": "4292915521facd7d3116ba367a4e1817160d7b19bb3290ab5710d24f33cf73fc959359feaf99099657a16fea6d5c330e7a1638f67d5f2029ba23cf7f23fe0e68",
                    "signature": "3045022100f3bb6197ff2cf10c4f9b93a579f7eecec401cc84acc501bb05b411e5275c1cd702204cce38ec19977ac034572497ec077132c3131cd40ec1586f4bc4ffcecf9eb7bd",
                },
                {
                    "id": "43fb1fea413473b1e462135ab9cd8154e6fbd3b41aeb5293c2dd738f116795c8fcb6abf6efad8802e88c413cc1a3da371e5412a1d30a80a1b1a4e2eb7af261ae",
                    "signature": "3045022100de239b02f3864ca14a13198ed998866546011f4907a88bc892aaa2c9a4659e9b022071220549e9fd23fee0d3cfeb76a96a0c5e52e6b4fbf891c979f350394ef9403f",
                },
                {
                    "id": "45fd9f7489f0b135f5f65c03a67fa181db67357818b0fece3b253f7934df7fe86e5111383d156a64e031eda064e1d5bb021bdf7c2e82489adc67120e172de7a0",
                    "signature": "3046022100fda8c5f9cae9b8fe7cf1862020d393ccd2fc098568a6e0a71cf2b0b7c26f8d89022100c0fa02811838e30155edead096eb4c288975ceb38a7dab20fb91144fce830c41",
                },
                {
                    "id": "46daea11ca239cb8c0c8cdeb27db9dbe9c03744908a8a389a60d14df2ddde409260a93334d74957331eec1af323f458b12b3a6c3b8e05885608aae7e3a77eac7",
                    "signature": "3045022100fd76296162fe57540ff70a08ea6c114ee910bfec4091e493adfd74df521c24140220281113d2a3c92a05b95638984c0a2209e616e0df35b698498dc2a377493f6e12",
                },
                {
                    "id": "49967c5753f24a43dc039ff4eb02dfc1000afafaed49799103fc524db064b9f3f62018b2ad504471ccee0b14cf67df4201a753a0034ac4d42c87a97571b51691",
                    "signature": "304502210095bd38a9eff93aa47d94c9993ef3c1f9ff65523a02fad8de6b716ce1bd8fa22102202d2f41bd4ea31067948f883b982f2d4da26420c912bb66f9a20184bb7eca47f2",
                },
                {
                    "id": "4a3953b25132e1abf795b8aa380cdaa1f9f3188637b4ee3e2ca3b4ba093a91f417a12e164d79193dc25758ca0d81d2911f9c554e94998d3db7cedb905e35a90d",
                    "signature": "3046022100cfdbc7acf597d54bb2b66b73f2b21c44b17b7fc5d2fe219afb739ae49988352b022100ec8a6d6a90d327801f30f9303f398e938432bdb942264408acbe1877e8997c78",
                },
                {
                    "id": "4ea70b615a9ad00a57acb4290159dd11f3c86bbcd0fd1ab203b6b448c5baff06a9a80dcffcdb94749f6c4013f00576dbd009e6be5a34621ebb483203295777d9",
                    "signature": "3044022043d7e79d1c51c592797515838537dbef37d1b609736fb371ee31c9b81006052d022076342b158c790f0bc009391267f9c6c4519c5c3b6cf5a3fd4a13480f8eadcbab",
                },
                {
                    "id": "4ecdce5572ac576fd5db520f5f8af4f824617931066b10360513f6d93a9016228f4279c74b794ece02f895c05dbef795c6f0538c9a46c3441285b4e4e34c7485",
                    "signature": "304402203470e985f4547e48d6b90ce91dd629dc0331fe3fc5a14857d9961c5eb5771811022078378fea2b3205b97d2ced3bdd6ee03dd34d4d187507d88dfbf4cf62df2536af",
                },
                {
                    "id": "54385490ae3137341b584a14862004b59bab380425ac1d3b96ac9220a7f34abd53c315c463ea065a1987aa75102ed79d1a77129844772b6bf033437b7c106700",
                    "signature": "304402202969fec8fbc43efdcb1c0e47b5027a5b6dc228d3f6c45142b4aded993bb22345022009b8939f1b0f1152c1d9a290fffee2e03a336d930356af066c1dba8444687d10",
                },
                {
                    "id": "557ff8226c4ae609d024a3bd5373b273ed3e4b439aacdb131917b69e913a0c6e2935089814b0235ed14106df85a317aaab4af520032c842ff71d6093ef271de9",
                    "signature": "304402203bd59fe20da2b22215e8bffa428e14c27ddd9cdf338c9866dd2eb26684f3141d02206f71388f7036e011ad70780f3d9833d50a8bd6400a83b2679ec55026742aa8b4",
                },
                {
                    "id": "57517cd838a7ea52f2b58c70fb812049f200c6a2c071a2d4860af8b873b8376221ee29e9b1a020b79bc26ce6028f5597be568610abbfa6fce2c5dfa06527ab35",
                    "signature": "304402202d9527ae901906ed8d40a3c1da422c4b13c4ba9c837d11dcb41c48a288f6168b022069be39377658692d225d4a3ff53635f922d937083ad71b749cfc7da6f445ef83",
                },
                {
                    "id": "586dde326faa3dfc4eb9a2387784d0a72e9737fa24af145fad138a01d2482a08f534bb9ec431e32ba2c9d9bc0bad48af6d18ae720226a97ce82c7334671009d4",
                    "signature": "304502207ce59dd45ea1a0ba562abbd0162c2283902ee78a47597afa0173511b5515874c022100faefbf1435045f89900a282e8ebfb95358b0bab47c73616de92cf8ee83ef238f",
                },
                {
                    "id": "5c1a00e94eb632a734b444835ff125e73d80a29dac170dd9a0963b1ea0c5223681d4fa7ce489dd64963064298cd80e91f6f0540fa56e042fe1926720c578daf1",
                    "signature": "304502203d2badc65cd5f179d4b949ea65bd77ac569b9136640ea0d7a28e42cbc1dac9ae02210094b3b50617911f669464730cec49737a61bb12070253292ddef0bd25d079b501",
                },
                {
                    "id": "5c3f3d810d6d1d3152695799403536a2f6290de81838615a49f229251fafedd19ee60f478cac8260b5a1ae8cfbf73cf1f296c25f080ccf7c9d352c2cfd94f892",
                    "signature": "3045022100f20f91de8b2505d2a75f676c1940c154f0d13424f7a15d8c2194938e31fd400802200fee06e55966c8958286308c6ec0d424e2c640d6d7b058bbbbabd44b3eb8b368",
                },
                {
                    "id": "5e128ec2e99219deb9613f7871bcafc83ca1352d94f804ed3b1305a200b3ead31a48747d02e3faaa50e389bc97fdb77b3b80a2adcf802656aa17b43fbcbfa0b5",
                    "signature": "3046022100fce2235c6e7907cf8a2477e12af1115e721c8bffee6147ab97b04a5ef9323313022100a116eabcb5d91bfb23eeeb5d5bddc25c26776c98aaff6c381e65ff8604d39682",
                },
                {
                    "id": "5fc1c52b33fb93aded7997ddbbd64b285959845ea529faab55a0bd1184d48b0a4bc778d9ef2487a552e82638320b8853810b6bcc3fc5b2257e34d5ceb4b80e19",
                    "signature": "3046022100b8dceae6e91207c4313821690119062b339ee30e13634f6fbf8888badf95b164022100915d17e0192cef9fa4adcaa2a19f8afb7aea637272abb413b9d5b8db7e69f5f4",
                },
                {
                    "id": "60869d732d411be072898b7abd062fd7a04413b6ffedbd14f34edc7b3498d6d5b11c77f8d9a79f7ce03a614d3ace1351dcc990bd195e24ddd0f74d37d9bdf78c",
                    "signature": "3046022100c68a82afa8cddd76eb90afd8a793e0c3d1e1d87e78616cadc857c640da0fb3df02210099e43074c74261feeb3ffc9769e2a4544f01489c9ce1a37300a49fa55ef39c99",
                },
                {
                    "id": "615b72d69facdbd915b234771cd4ffe49692a573f7ac05fd212701afe9b703eb8ab2ab117961f819e6d5eaf5ad073456cf56a0422c67b17606d95f50460a919d",
                    "signature": "30450220730cd20dc5051343e1c90f5acc1fe948cc45a9e58bb53727cf7d655b90cb1398022100b35ce03342cc9ca8e0556a0a98377843b9da0863b8923d2bc737ac0bad781d05",
                },
                {
                    "id": "6239f0666c12017445ab0e3402f108cf1467b83d828c3cc841f210679d23b2b3a5064169daac0f81d03841663e7e0b524e16fd99095e726b216bc3469420b7ef",
                    "signature": "3045022015cba8c8b8acdcd0dc03cff60715dcb5b123a196024f2c01088b5f8a2504f600022100cf22782c8101f901e40427f0d29d0cbf1f224ef3fdc7c109248ee198232659a2",
                },
                {
                    "id": "625f9a6b791f953006633cf82341c2bf68cfbbe5c52a2b80449a383174ecffbbc071e662c51391629d71b2e8720095315c43a524009a07afc225c2c21b134467",
                    "signature": "30440220235c8c0f1bcca148c468241824d9ee31b9eca12aa75f03064270bae93b61aec5022006b72884a1f7f9f6a459a56b376428655f8cb265a273f201afb9e654eee0f475",
                },
                {
                    "id": "6262354d82b8b4280037df5d01e2f63f044da18389aa069f6168342b53d9d1eae0134fd28c2d02d313d0a3aba6fc9312d6c35038b5ee6ebc8d48f7ade751ad4c",
                    "signature": "3045022100baa31a8b5e7509bd943dc6813ee32972427041cd5e3df38a5d01304e9fff4abe022036d7477b0c3ba08c683700000785cdf6df94b3acd5985d15c27e823e85848c26",
                },
                {
                    "id": "6382a24ebb80a8202fe77baad81cc9f09002fbf229d9bf0189e12de7a9d6c446a9be551ccee30398fe898f83e1373004c3de7f092af1f46d4dbdafb8cc34e752",
                    "signature": "3045022100aae393025b36ba8e9fa712778cdca8e37138de542534a79065814caf378e8e600220426bebd69326a8418aaf4d9c9f6031e7c8c64ea3cabad771938861d8fdf50dd3",
                },
                {
                    "id": "643ce6cefb737fcf6131830e2275bf65e65b830aa2287e61032644119f72c4a88220ebe35e30ed1a383432abeb3554985b7358ef6f44f9107fcb5c0212ba273a",
                    "signature": "304402204dfdb3882a4365ac18ff98dde5cfba6022faa23a07e100c0f5e31f74bfbd8dbf0220672b40b6697422940441d8529944c42bf505ae176655b1341ecc302b8c9d3c68",
                },
                {
                    "id": "643ea6c216727cf509204e231f97182897032fc4b5037605a83dabc07f56879b743a5118e61fa6a74d5b7a64e504c076a363e5e2787b3e4bf5c9ed4777dc4795",
                    "signature": "3045022041339c2654ee8766c0b857cafdf726987c00089fb7a117cdbd954c20a565055f022100ebb70e078a142e8a1b314e487aca6f5ae160efa01cd693ceccabbd3f59cf5cad",
                },
                {
                    "id": "65f85a29308c80ff2a353c1d514f698d0566fb514fba6f48631b419ac2ff495e70988035a58a755087116d1a9654b41478261b8ce36bbced65e12456de213166",
                    "signature": "3045022020675710698e2ee8604810bb1178c48970459312d24c54b4c4ed97d0344ddfe6022100b7be745aa13c3d7e8b91256b3bf15c7370c2f313a36330273d88c992e8d7ed4e",
                },
                {
                    "id": "6c6cad88ec86ad06e2c19ec3e4926a6a2e2928525fb4c49de0ae6a304ed0da52d74e2c56325616a1175bc607113f9f934b5844c3e7d9179a99b2f7681f26327b",
                    "signature": "3046022100fd6a87ebfe7a74efdfca86d9e55ac62590c14cd33b0770a1a352e7f76f74ab95022100af22a73c49c186d298542a3d63bb43cbca95354dd4e0cffc90059e6772c7dde8",
                },
                {
                    "id": "70b9c9c0c2cefdd3fa7131de0f2a7dd7858bce0613cc52bc30ff7648f6bcb3842b8383228346fb2eb7cf7565ea8617e1a8c6283b6708888a29d57aef0748216d",
                    "signature": "3046022100d019546cb9f87e85d05493112ecb54b428aad3c7ee5119f71c5087f0ee6b1f7a022100aa8c2dec3d56468d84d9e86fbc7690c723b6cff3770e587558bfd604a001e7e6",
                },
                {
                    "id": "71c79a1c5259a6269d15a06a465941288331052fff1b7b95c0aee5c46305bf12dd539dc2731a852a78de63eb42a541cea463a0dd90684977f1a25ce8f26236cf",
                    "signature": "3046022100e69dd9aa9302e72c25b5aee6ed7df24ca573e7b563d15cf8aaebfc34c377a59d022100eb3a4914c68e38759ae47b573cdab292c6134e2505a2c158b0c6cc70d12285a3",
                },
                {
                    "id": "71d9411dc4297b810a2b3997e991bbcde3391dde2165aa0b90c9938da0ef5fb4acc22672adf0030b71f6bd89401fad656d7ee707161e1c759cb7ddd12e8a02ea",
                    "signature": "30440220347d67417783a4a94555a8a2649140f2d8cbcfb677bb6b695b9e3c90636f1711022034ed9d45a3ccd26127e3556b8de71ec51b9f5a7438e65f41ef72978e6cfb2fed",
                },
                {
                    "id": "71ded58853d8940a63c7d24edc73c549a52673fa190764d2b800af1dc2c09a975257b7a8b3bd3094243c77597e0e7d42049ceb15dc4f2328c627c35926668ee7",
                    "signature": "3046022100b5a48f5fc21226cd53bb0e4c7beedd42ff0f15ef28a70da2b432b8747a80740d022100ff330041fc631b6696e8b93858d51347e4aa1e73b23179aa36b6726532b9c5ff",
                },
                {
                    "id": "73316e9cf251b0caef199fe9dadde80be837904d252bbad3b56ed9e478744edfeb36281574b8e3e9d9757861e7b1dbe0c12e3e59b849eef1099e6b5efc6c6e50",
                    "signature": "3046022100e0c1c51baa85d13fc125d55cf067bbc0de32c2b77a477ad19a1bbf904d0d3ed3022100f2dd708a984b64fbca9db6687f6a9d4f4105aaa8168c0e0cd78ce4f4e5e1a65c",
                },
                {
                    "id": "751a7706166e8dd704267c29ed4e649175b9d88efa2cb1c638b3483acdb5c9c97348bc9b8babf85850dc5cf9c90ec7336dd1b80be451fb24c2013828c49a7f6e",
                    "signature": "3045022049ce7530606d0804908d4e6522c107864b204bbd1317c32dd55f1dab879ec7b6022100b9488686bd5386fd3a2319da05a0a0037c2945f058c375c97909c235564d36f5",
                },
                {
                    "id": "76a5550f9adc0c452bc1924deeac9b519ef339d0874b9263f27f5730ba49eb361ead947d01a3e1a353c79a3bf3376f4064ad2c9850ce1cc5fb332b2004d76c7a",
                    "signature": "3045022035d34ac317a930b1206ff5d00b0a31b196d13608c7c0ff8f72919c318a302119022100a2a515ad1ec84fe4077bceb8c0ea1ac097abbaade67b32ad7988f5e05a9af665",
                },
                {
                    "id": "7776e747cc6b0f2ae1343e41327f477e9e28adc95dabf632e7bacadfacbe08acbfc12602a8e2e68a4822e257ebcd161d1278351f9ab0e3a56b81176efca3e4a9",
                    "signature": "304402203808e76c473826581c63184b606afff4f95722d6ab1d8d33049a8e539a7d3228022063f2746caa4e976d954f4c83d7c82a3c5f5e630d2fe59546a02e04a226d18890",
                },
                {
                    "id": "77bafac645580ccd8ba87b489a3b7da7f141ff0a3c684369d0701c53b9fd17bcdf3aeb4519358c3ba3cc307594a24480445ec5add230dc3809d56a942d6a3b87",
                    "signature": "3046022100b6d4206d202769f6da1242a45525093e421f33d207b518f26486bae89642bbb40221008608a1ec977505883844b4d8e85b199c6be3801b558ada16456f7083b0c028ca",
                },
                {
                    "id": "78d4b77f925e5bf3656700d24de51e02e8b1af71a8797bbc57dbc455fa8fb3a6f82c8da4d13bef99c45962873172938d04b533238e0e990479fa103e8b0bc45b",
                    "signature": "304402202090daea99f7f39bd40eaf7071ee565a040b578d455db15443f3fb596d890b2502204eb03f39cea338aa39f6dd603f1f40b5be7349e00cee30ebba2ee91218165164",
                },
                {
                    "id": "7afa05174a3fafe5d350644c584fbb3d00826dc76335ac953651bc0bd68f230ac131f8d9774bb558d2fbdc1cee66fc3acc3240683217ce638fe6e5a229271cc6",
                    "signature": "3045022100e6e43edcf1f2de4d41b741365f59640957df92843b67862ab3c3439724197339022045f4e47614d804835203859792b7e4409e74a7fa4896744099a15cec023a9db7",
                },
                {
                    "id": "7b6169dca70dadb7a7e01b32756b781744f2ba0f4f1d8116bd5061dc2c13888a004a302698988c46cf6c3f7c0255961b04ff60d03ca6a0d15034cc8b7a857fb8",
                    "signature": "3045022059c6fed8776f7c1b96cd5c429fae3c6e3ddf2f3bcad0800bd0d0a7db4abe58b5022100809ad08362cfc46656945c41cc6f6c3292e790df824c62847f5172bbc01af535",
                },
                {
                    "id": "819c2393fb6a791c7923535ab04b5110b23b935e17cd480654f586f1c8cc46eca02836ca7127bbd68d7f1d26393bd711f9ec70317e6147816f69d0c0886d74ab",
                    "signature": "30460221009b38f768d7586c144b22f41f9411fabbe622c9e9cf156248cdda774c96ca16db0221008f153e6386aba545c9e9aa53d3a03a91c51f512cfba42fba0cbd25b8e4739f21",
                },
                {
                    "id": "83b6115ec7c1a0cb7974e0dff276d704e00ca028ae1510a4d7158a51318326636ad2fb9e79839b922c4e5163f649e935f29a361b25406e5eb9485287fd891c0a",
                    "signature": "304402204a8125a5c695d1c1d2ff97c61ee5abbb1563aa2f4c4ff51ee3c1a4f296f2c53d0220633f88658dcf3fc2157763d2997402db95efec2af2865c4d4a5397ffc7cd5b71",
                },
                {
                    "id": "84f9dcb8d8ee3b3e37fbe0585653074d8ef5534ec2555af2f47253d85cade6e4a847093f9bb1d85de9c1da9b48d204d55244693a1a47c84950fab7b84d86ac16",
                    "signature": "3046022100954771b2f614d64169498de474686472d41a6941df688c95fac46185bf6f6f97022100b2844a79ae9a9a04510f636de1bd1a874fcb9071ec283930329fd2d130b043da",
                },
                {
                    "id": "85e157f1b238bf19cb42a5da8f1efaa29a5f43cf9847ce642cb87f180b19ed330d41cd359ed2033f21803aaee6e2e5be0a0ddf689e9c96c53d811e7fac72171b",
                    "signature": "3045022014a3b56714f9e23d850f0b5b3bd86ddd48283b405d23d820aa6ce203635041de022100b4450a6347760b4474427895adae7b4aa970104403b663e9b468f829d2450d81",
                },
                {
                    "id": "866f5369e16c0a2fdc6f257033c5f09f482ae780d5113c793ba0f501e09cff0e401666a94da9bb42fde5fd64a0a18229ceed6482dbe6fe10b77d9572a34c9dbe",
                    "signature": "3046022100fef7a5742fd1a46bb97367b388ff8cdd45816751c824e98da6989c735c654633022100ca1596796de41b1cf4a5d4f3213332776f928204b79610aff0d5e9495d2e8b2d",
                },
                {
                    "id": "89d6b7f7518daf163f45e8fa97509dbebb275fc9af806470592e2df487af268f96bd00d4d11caeb6877ce80bd41d20225d9a15874b087f096ec667cffc0cd238",
                    "signature": "3046022100f0797edf5456e5fb2599a2b3906eb3949feb5fa9e0935a7a699522fc854df00c022100dc87af23aa849a4ab99574c49d83c5d81d9d8720a9d392c5ffb2de81dd62d6fe",
                },
                {
                    "id": "8dd8e9b7a07f48e27508431756d2558b3f66a6ad4869a9e133a624712919a8ed4fefc1e9612c79362c52c81ca0a389214da213fe8194d56a63fd1c67f993c3f0",
                    "signature": "3044022030be3ff9da2f237cabaaf37e2c7a4e922491a333700af0a353722d1387681cd9022043c708d2c670c9fa38ceb4977669bbb9fa0e661a8e24abdaed6c10cd42c68933",
                },
                {
                    "id": "9037974d44f664a9e3beab3ac359307967d8b4db394d6729cdbbe2c6b585354df00c363c25177ca4682fdfba065ebb239e5a9cf2b7d527d45e8d22be4a78deb5",
                    "signature": "30460221008041a21d98c8adad244d08bb3b3aeeaaeef3c15260ad0910f45802efe1209bbd022100cb347e9ff3f2e4109e7ba0ea35e22370613db4746e3d2cf87e5c9cd5018d4761",
                },
                {
                    "id": "90e817a4a031c46d99eebd291530907e1d3855eda8d62f5b079c08fd9fd1e4a098ca3e7c7e9a971322c8b219aa0dd00f1bcfee8b4432e52de7fbec09f715d5b9",
                    "signature": "304402203974453c0c1cd95c7616e45bdae02bdc7a0b704822d33ed34712fc7bdbba998602200367708eb4fa25e6c6e53359c69ad71f8573e9e6d709df1c561f0081055be995",
                },
                {
                    "id": "92442cdbdf91c4244f8e1d3d20729223abb9dda398e1d73affa2af6f889e60d6f80761e97259665161fc262f93ce67f2a94cf8c4ff65a0ff5f369d3f49eb2e87",
                    "signature": "3045022100e814e7c9e01ced9f35c27aa97a435b5788378c0aec00f8d39932837a0d01182702206766d9406fd7f28f49469e69db70a4554d01b13b185399848d6239377ab840e6",
                },
                {
                    "id": "92a997d33992e405ea85be563824b43a47232c8ad011c704d5ea6db3c18c217d465a1f1be42eb94e2a59e4dd7dd16bdebcd30f2b1e422bbdc35ef17fad9eb7bd",
                    "signature": "30460221009512829d09bed5a5e8fea5c0fc8ba7ee1fb3913e73dd07ce327bb20d301f1608022100b233ac1939fec5ff613186655df9813b85786697fc5ba086b23336167daac379",
                },
                {
                    "id": "955176c033c51aa4be4e61978c4c9fb5a5b630c22eab246e4739a90818af8773eeda09c03cf9f66231975873e41e9f6233c0f76c650214f486054d7ff3c22f2f",
                    "signature": "304502206d626474b6a02ca19b40f2033ada4f9099c8ea50e536c271aeff78c4409c9f84022100fc0957849247e0c1da7e0fe4ab7c2a955776e061b703c42a0a4e442cb4f8b293",
                },
                {
                    "id": "9612bb6471baf8117fe47d60e6241a360b14bbfd679ae19e2c05fc6633461722d992134d5ffbc77015dca5488f296fb411684f3a4c8b6fc9c9f5dc867055f50f",
                    "signature": "3046022100851500fd1ec6d1b10e36d6997de0240153122c32c6f1ea7ab55d19801a8c4639022100dfe61b0ddfb39fce440a3aacb58c7b2534e603188624cba6365e9a707dfdabc4",
                },
                {
                    "id": "96bc994767591fccfcbdd89cabdab889a8f7e4c6bdef11a76096707c2aad48da83414cd7a6f14286860cb38fd38267f4b7bd843f8d4f961a003a3ccb01fc1e59",
                    "signature": "304502206dbfa7836395a66bb3137af3f7c21c11aafdc088f9b14c746a4448ba56b7ef9e022100b31c2d534e313378cca714627dbb13f63c917e186c8240a82fa082a92df40ce1",
                },
                {
                    "id": "9b061922e862f42c273b0218e17b1e92e196e93de83b497db8897895e1e8067f83cbead903e5f5656f10caa544ccdd0a3272e4308f37580d4206ea66d95b579e",
                    "signature": "3046022100ee42b8fdc197251a09ab516011874dc15c2ed86453a009cf95761e2aae036592022100d1fa82c50eb3bc1da6f4351bc03e1ed2f0daca4653f05d4612e0c434fba5be26",
                },
                {
                    "id": "9b99453d173b675850cb1b576c4cdd993f76e66c1fab473ea89b1f0668f39aa1f5cc743cacdaeb4faeaf06c34ec897ca0b5b18b7025ae319f461e24364209e18",
                    "signature": "3045022070413f8746f35734eedaff0f28295be3a717a70733b48aacd8e153810a36d660022100abc452c3283fe3227d6d5fba0af099ee297de871ec7f23b036d6b42b4e080e28",
                },
                {
                    "id": "9fd3ee46488fff38a828b8a2bc3ef6942014b82b403af4e013957dac722344fe54d4d8b8af0398ae4e756d58eb28392d29114ab44804013ef79899c9ab163ae5",
                    "signature": "3046022100e0faeb0dc539b0e12ba5dd77518c6f58b8e21164cca2b85d2c298e22ae0e6223022100cb2bfb5a36e7e949a5f53dfc1619fd6dfb8415f079a3e4552272b6998893e636",
                },
                {
                    "id": "a0967738784c738106119febc79d94a7e6ba083c570b8289b52371df8692897e45c2dd58edaffc9b0589b603c0321d72ad548dda4f68eb382dc5671dc311db44",
                    "signature": "3046022100a7b7235aef7ba988341307f069317c166d2ee11414ba42683406b4639cd32ec4022100e5acd6aa040c074141d8da1bd3a72bcb46e6a835a8bd36ec4ed551f5542bea80",
                },
                {
                    "id": "a1f3d2377e0ded8a3a89d4e15ed5f2cc846a8c4ad075607ad1b1f2a9eddac2efd813bb2bab1633d5b697a73bc6a92e8a3e8b2c387e992ec2f6402e8823f35d4a",
                    "signature": "304502206658fd142c5928864223282157479c83c6919c4edd0fe4438af0819101f4863a022100eb56d263ade1c8e4e6aca8d87cbde27abeaf148114a89e364c9a07e254900747",
                },
                {
                    "id": "a2fc3a20053a78e4bdcbbbd9b8ff84bed35c764a457172da7fbd9f0448767a61ea2f4d1be165b4653a83167789c0dc772d1484f5afd678c41ae1417353a3dcc5",
                    "signature": "3045022100bc5c56f7ede0fc575acdb8973d925b2f7b926e2a4be30ad5dbdb9707db8cb67c0220155cfdc92f9f43992665deeb6e81f3f595ce023af61a090bf9234045f2b86433",
                },
                {
                    "id": "a31b0b988ac2e964b1545598c7e7bd48c6e18650314f3b1afc991e74e835a92b7ed477144ecbf83b17310c015745acb70fcd701f1d166c081a13d338bbe20eb6",
                    "signature": "3046022100f0e9f6eb886384bce21b7e1fa33a9a37b5ed8f0bee641ba19c998d0276197b47022100b080337467ee3b1f90f51d45821e2de6a0b2d5cfd751a77bfeab381c6dfe2175",
                },
                {
                    "id": "a7565e5a3d090d747faa4cfaf5e268c1824e314d29ec5e306e0d9ce9dc6fe19cde994763a610796f2ce6af17d4183bf18fbddecdd9783441cc7c97d7a32bbd6a",
                    "signature": "3046022100d2775d53269b266bf9a590bb054dd0cd143534cc41edfa20a7209628d0dfffb3022100cdea83a71acf89d8f55a87267e82e74393baa341b5c5817fc7b2c0f92b0d0827",
                },
                {
                    "id": "a8994e11b8a84b9bfc9bf87982f18b213b6caf02355d4ad606c8dd3b164a0ef605e17f94beca2619c784b5b0f7fdcb35aed45e50ea01afdbd27a90cf1c2eb5f1",
                    "signature": "3045022100ef2c1559276c37b736a468d93b5467ab9542d7393e603daaa492dd18b2babd1d02203c42807c991380b77935ec59bcaa737eb820869313bde17b18a7431e3dc5479e",
                },
                {
                    "id": "aa5456ed7af64cbe6b2b92df8986c5d15f3a67fca0f1093c5a9505d16e8fa6aa922e9ac7d23b1e997043b5630509e1a382a2b83e4871230aa59e510264cb8ba2",
                    "signature": "3045022100d0475da2476ea13f7ab3ab4a924e6dd8d679299abcba7efec3a6790aa8b7cab102204dc16def017daf592f0451c2c31d4070a9613d66acc4292f018b61820c8e0d4f",
                },
                {
                    "id": "aa82f3f00efababda29265fae705202b7650f030ae9de3c000830424a66b3e2f136e653859813ede93168c8480f5ba0dd28cf188175f766feffc63399abceeed",
                    "signature": "304502205aedb3f44faf62dda0933133041ebade1322d782c274eb369f5b9102db363f46022100a4f2e1f9db6797aee45a59cdfd60a7e9c808fb7a5a55c3cdce180a6a337b40e9",
                },
                {
                    "id": "ac3e2f7d9d23f907666c95e419526d2a866fd5456f35224ea1e0e1dc7076f45dbf64298077b741d78b45bc9f9a361e7d31d884f907411af8f80d6a2e1b7a971a",
                    "signature": "30440220403e7aca680f84a57bd04b8109dd74e25b5afb64e0f6cb55f8aa9c2a5dda6ae302200bb0a057d1c79157c1e403436e3b182fb8bceade943a5a6252736dc3b08789bf",
                },
                {
                    "id": "ade6b9501a243a5fe53d683b23fa8ff041d0be3324a40822e03302f9d3959b587964f80164341225a621c6d8783160422e753734013f06221c3d25a85f5dc647",
                    "signature": "3045022034f054e36bffde9eb6b48b0947ec437314c866f6eee702abebd6d7519d888172022100a84c59e0ede415d6220807c98ad6cc12cb4f026fda83cd686b627f2cc47ffa25",
                },
                {
                    "id": "afb4bd0e56aa44ac9352670ba7f2f3f71d207a9be5dcd2b415c9e3af869d29db6748fd06f7c4cd97ac5bf20e41ef8c272364bbc13ea05890bff9c8fb8ef4b550",
                    "signature": "30460221008dde4cb08e639e7c6800f0738ff5144cbc00a975ade8f20dea207e0a92b13033022100c4e640c79616071d83dac180f8d638654a939d49c74ac4b4033af4989e4c79ce",
                },
                {
                    "id": "b1e2fe52930fd58d46b68f86e972d6ffed0f96bd3fd5546c54e6a8f24d6b7c1e7bc4e12852c1b36dbc04ca0817a90d433b7fb3eae2a3ceeabdd53cab6507d2ea",
                    "signature": "304402207b6f74ad9e23fcc0d68f4f9d6a16cac5687aa22ae003f5d2d35d4ff8d79de090022002cb74f0497bff5216cc801ab872341091aa0122baa07a00aa8091858a40bcbf",
                },
                {
                    "id": "b40b7bf2ae6342f4f3acc209d44d35f659f309c08e214e30ec9ee8fbae8e44e643285abe1aa760224f8de773ba5c685e668eb7b52028e8c1497131bcf4bced70",
                    "signature": "3046022100fa3e7273a88de5e8e837d9ad4ca583125bef8ec74a8532ea3c268274316a9641022100ad188f5e95ec7b49a5abbc2b8a462faf8e079cb0a15f37d92766e365fe5d10ed",
                },
                {
                    "id": "b41aac4d367044b3b2cbb3249188921207dc181eaa7bf3b30f666c2209bca22feb305db724cc8444a1a11e9a97aade5c135f2a0b829a6f651e055ccfee36f438",
                    "signature": "3045022100b91f4e5b24cfaa1b1baaf6bc20e71de40e173ad1dc81ca90837ae654abdcde9f02200fca4cd4461e2f2ae9239f3720489984f454b9f230f21dd69cb9375bac4eec39",
                },
                {
                    "id": "b42a086d1ceb030f02489e6904786cbe3e10859ed9823c79f95d0060d3532702e93cbe10014974bcdb860a818e2b0c0aeafd6702565423e374846e93bd8a6f9e",
                    "signature": "304402200ae2813454279a14f413dd6d0405ad49a4791f21ec9354c609c2b9a6f85d1e040220287faf15d018a2d56c7542046c4e846a976f1b0667511c6c7fc45ccfa617f7fc",
                },
                {
                    "id": "b59af2e299231ddcce7683f07e98f0a091b7a9a5a3b2a03ce39489fc882c040f8a341602b001334bf59d633350e55d8989c3e0a1dd47da9851813df18aefbd80",
                    "signature": "304402207f5ea86e2b4b87a17122912ddf349f87116be5481a92e2a4e68a4e11eae933ce02204865cf073804d88d5f784ff50025dc0fea79b65c7f5c17be2318126bc3805fb4",
                },
                {
                    "id": "b6229374ebedaa6f38400b84b4b3537a39c0ac8f9217b348d39e7e7a9e32fdb9e6ddb11736b4f9e126fc609287be2dab32fc31433dcc497452982da9bf442bc9",
                    "signature": "3045022100ed6860755fa47e2d683d13fdf6b1746eea27c9c99ce2adfeebba0a61e22865ca02207ce13fe3752d638e9055b677526046870c4caa191c1131e19c703a82b5b890b2",
                },
                {
                    "id": "b75aba5c7658dda12b287ba2617192913d2b0ea516f9a1b4e4888c44260ca01aa3db848a8064f94f20c51496b5c2e1b4e100064dc4976af1a1e765921300e649",
                    "signature": "3046022100f88019c375b87a8afb16ee54c3ba31e582a2934d66a1502ecfbff8a2d14c57e9022100c56d2fcfdd2e5244429acb16a28bd15d027b4e7d8336b014e0aa36943b1b5dcc",
                },
                {
                    "id": "b79572e1c1d1775401f5e531c0ea0aa875d53fb32ee325a5ee04b10e5b9aa50e69d49a4656def67d9ed5c404554655378630361b3be456f8e942d20db3654065",
                    "signature": "30440220307dd0eade5794e565796b77408ecd1e027e237df00bcc5ae13eff44e3c0133b022048fa5b629a067ace558f0953e8a1360f666b29513b1ebddfd4ceea95e8462d7e",
                },
                {
                    "id": "b7b0c4a335a7362899f8f3b33df47a8ada55834b7c1fe48b1b5af7da34518ce53cc031a9b0091ed239c89980a5b6480d6691178543153ecb40766a51bbd9356f",
                    "signature": "3046022100b4a2fd6a3bbf2af30f54d49352d3d33148fa56a955606f71d0ecb9988742f945022100f2024b2aa1862d5dc9e203a86a8baa9ea347d62150393c0963590d2b473a80bb",
                },
                {
                    "id": "ba0bbc944d1c5b866245338ed74ed8db8d2b7a7107ab40260a0e161d70a5b04ba53e4ca02b9fe4decbe0bf948e4be0688dde08a6e0e7ba523711adaaeffcf91c",
                    "signature": "304502202b86f2dca590f18e52a57ae472cd594d93ef2e58ccfd179049f748d6f4bd3b8f022100ef4f1b051cffe9a70cfa9971090e06c7e5e24a51b7c0a73dee6d968b2460bada",
                },
                {
                    "id": "bd42ea173c96962374055a9730bd08ec8c5ceee32ead718b4e9187a7012e943803af8d282559dcb47a1fb311c6c4e80f6c48a86db293175caa7c6c0d3382ec6c",
                    "signature": "3045022100cc608a18bfbc8bbfb9482c77dbda292d1c351d8c0672d51ee9331ae79f0396dd02203723f4bf6fcaf34966ea3695b818c6659ff47dd5ed4150d3f54f3ef00022d3b5",
                },
                {
                    "id": "c76cc7e27f5e8bbb3473dd068d078534baa401c6ccb5a550ed3347848d24f1358c37294bbe4cba351591edd2e4564e9bb18a8944de8d23198404448ac1f40fa0",
                    "signature": "30440220645268eb8fb50ed7571a8945090d927b242ba557fb816b77cfc53d000b288c13022078b65fa7f94405e6555c98c585ae9154e64a52d26383d1495f6ddaac99970559",
                },
                {
                    "id": "c931d89f84f3f36151d56111d70953290c036c79a6c41393c3921f7c920e866bdbdb3ca5aed0355ea6bdfd26a046a4cb8b6d17d9c581121bf91077346a69d816",
                    "signature": "3045022100f9fe441c8020164f062c9daaa222136d9fee84c326fcdc016b06207460a33e6b02203d0a1fd43c13858eca0bb294de409d91b785ae5a8baeddfed44080805fa77af5",
                },
                {
                    "id": "cb38c6321f501af4886440d5e18d6d41df19f41036b8070e0b8eeb455015d8dda9bcd9687ff88732284e357ac67810ec8ac6a574bc3ca9b4f05ab8a1ad0b2609",
                    "signature": "3046022100aea4b8b556c904a741e55379fff8e89f932d2380aee202d8e257a4549f3ead6b0221008e9ca9495713abdea9d0945288f5f3c4026da7d8b78c426d9dcf18ec708fd4c6",
                },
                {
                    "id": "ccb9b8878cc187586702a7913a86b86acf18bd45dd39b64e3320b5034d1618f9f6326590e1e035c10300d775e50606f4fc07b89dbd4cbf8ff7ce2caa61f878a7",
                    "signature": "3046022100f606b1d8a2df37d0abb1c74d9aa95e5c87c628ea7688af85fd40980b34407c59022100bbac3a7f8dbcfb5d229fc963b183187720dfb9290dd714b07aba60095312aef4",
                },
                {
                    "id": "cd6362aeee3ffb3b209bb3f1459848624e8bd9e66cb22e82a8cb7c08d02a833e74e3becc1fce804b1233773e24baf178f17e2c9611c39758abe8cb2ad4d20558",
                    "signature": "30450220176edc212546f22eec4fc612f1e26d41e1b844ddb4f6d1f86e59cb17ba51ae1802210098f773d58f2530adb32845deea0b232bd7a5385772847bbc427d639803bbe2db",
                },
                {
                    "id": "d4542ea363df2d9bb6799bcec98498404a8fd043631934ca9c3d2d1d73119f36b16e3edb38383faea6553a7088078ad358143ae71dee3ec86d17b84dcca5eb89",
                    "signature": "304502205ecabbc987a93412c95b98345d7cea09f9e3d324f79b9e71aa83660ee6bd4c5e022100f2326521c22e2d1552bb9abaa98bb563bc49d47f2a0247fba9c9eb83680bf5b5",
                },
                {
                    "id": "d62b6ee84dee9d038ae9ade057da3bfaac593ca295323b1f6b4446ed31c9660c4375b7692420373bde87e8128c6acee6de698f95058301565783fb20ba3a29f2",
                    "signature": "30450221009f93454f2bd60c594e5ce2a8b761b0fa61b6cb6e750a4d4377217d26b3172cd60220738c014edaaabde940e1f22341f12ab05672fd5b04456d1bdd61fc3950fb85b5",
                },
                {
                    "id": "d6626c32cb178edf0b2e2939afea9977300fd73b1cfb97d397be3d7952bb58ac2d6939d414df50bf21ac578e4e9b9ae8364f0580eb523d4e55ca8485598d1f8b",
                    "signature": "3045022100a9edc47a55e1650d50e763bd612c49c8c03db1b6032c4e80ec1113c52de09a910220122172ba881852edd984d02a51a0b0d95be09a0d3006ee57150b8e9e8f1c7679",
                },
                {
                    "id": "d7494f3fb3248b037f06e59c4e4d50f19be0b4f37946470724d807a4dd1f04e0878843f3a368912fafe90683052235f5082ea4b064f7e0a4cda8feb9c59c0831",
                    "signature": "3046022100e54ac44c53198e85c98f87fa4c8d822d62cd31938064d7c05487c8b1dee63754022100fd27747dd3ecf631e5c78c8426e1efec38387b9a8963afbb39b3753b6178da67",
                },
                {
                    "id": "d79af4183ba21fcfa36b479140ecf579ed1c206a32c04a40929bc30e75d9859f7d6746e43b145f5e9b12a41295146d796771a4f49f926d8b899c42499d9b9801",
                    "signature": "3044022044b9e46773adb2d69c146d01a02548dfbf93abc1b94d80643fb19b5e8abf77fd0220673ce191a41068189094ae42d71acbd1ee1a671142be6679d869c7fb67218ca2",
                },
                {
                    "id": "d88ddd9a98d097a386c712a38a4fcbc4e55c26c5cd75648e83ff1702bd820207cf581da06323a237b2b4f9cad46424b4e309801d1dc334ca328e98d058d22c3e",
                    "signature": "304402202635f0e9de7652e9325c6c767c94e1749bc6655b99dc2d81bd9b292d70e44ad202206c1b75f684c4b67fc810c3fc046ccdfe1a5bbcf202495d935b34ba5b9a706c15",
                },
                {
                    "id": "d92b944faf967310758253b4fc6aa18161cb60435019f5420dfe4af010458ed7c9b7106c6069b74cc2adf1b85aa773336e81585bea75826d0d23958e9f07c6eb",
                    "signature": "3044022015cc74116b52064db2852669b708aef76176b5b8b238300c291ff493ff7fb2f4022063af9eab015e842bbad79663a90a1a4e47828f2eabf1f96eefb136c14898e139",
                },
                {
                    "id": "d9bed1a5f05edb8902d3058237b33fc7de1976d1e340ad3a8f31e073cbac34db96b0cd0053e319a4f7fa547f954574a3d262df70cc6df8d8c8007fc6bbf82a51",
                    "signature": "30440220428cad49236ef0fbdeea819211f15a94f8475171989d51f5dbbd78122132650b02206a9361c9b632aee927019138c4a5ea84c966795b44bb9ab6b12429273c170788",
                },
                {
                    "id": "de599d4c13506d504b325d35844a19bc5de5bd2655d4f9924fa4be5be33c852404abdc1ec976165141f68bae9840e1440b9a9c48c2d91843c63579a14dbeaa95",
                    "signature": "304502206c88a46eac3e6e58f62117c6bd77e0f215a5a57a5c3f7932115964ec16f9343d0221008e6621d01d88297111a181c75adf075d72c04cce7cef98793d0a1d4db6929565",
                },
                {
                    "id": "df7194da5b728aff5410b5f8123287d10f8f02d4ab69c37768d7577499d6a134c5b1655ed0b3a2c623dd99f48dbdcb3c19cb852a3e6de3d3f2bdd6e12c51a748",
                    "signature": "3046022100c047469d7a497e30c5ad1f225b14fd39ab734c79cf6b45cf6f5eccdc3b10108d022100f819aceaaf7d5f23d74726045c715c8310677059195ff13b548b00d47b0d109e",
                },
                {
                    "id": "e2458fb3306cc7737b16d3aa96baf197b53fe95138968d58b10293fb9c6ac1b7b2d6e7593eb045258ff1b766f8c299093396b91deb3fb8874fe1636e25e76023",
                    "signature": "304602210086c5f5eab087edf540775fc6b97c1e64c0fe042b29be44eb132a1e5be9a4e228022100ea4ff860d27f4fddb6084cb786ebcc2ef4ed1995265af8371e8ece92218538a6",
                },
                {
                    "id": "e2f4496e5872682d7a55aa06e507a58e96b5d48a5286bfdff7ed780fa464d9e789b2760ecd840f4cb3ee6e1c1d81b2ee844c88dbebf149b1084b7313eb680714",
                    "signature": "3045022100ee026706f21378d07c0877eab327cc5baea024dfbc5f1a36fe664eae6628214f02206e3dc5ddbe077658de135edf04c935f4fd5c23b7c82bdf5758c5a20558f771bc",
                },
                {
                    "id": "e52a1f715a69e8d98f38575f4b9c4b2fb2c68e59e635094a13ac7c6704e9241ba540be586b310bb1179ed0f1c71961fe4586636df81e7bd4cf5045bb74ec4b27",
                    "signature": "3045022100f5c8b76b1c8173f7cbdcbfa2dad2efb9b41a9d0d91f48cdc9ab686129b76d489022050cceb51d06251dfe1133abf46b8a356f3f691e5852008987599e77f1e0319ad",
                },
                {
                    "id": "e52a6d1d30a9438d1b34dbb725c02c4f742adf3988e5f98ccf4f4ac45f630e81635a0c09676f1d0137d725013166c67314d46638adcc4962f1f5b0d8a9f2d8fc",
                    "signature": "3045022052bbdfbe19d5615d59f7e7d70b2524be9d5fedfb790bfbf284c63a7af1950e20022100f4e22eb18bf6060c9fbebf0e0b67f7ae5d95b8121b9f00b1380d7fd7b5f1fa82",
                },
                {
                    "id": "e7a0f74215bf8a96da6e7f45ff068e80dc2a0937538333cf8bc08003a202bc4cbc8ae9c60863caa6bd4c1d4d7257538b61c11df02bb8f2e48cac1c308eac6eaf",
                    "signature": "3046022100d98582615f5517c06eb5ec25a1154968dc7339a640cbde4d6ab79c47ab758608022100a85bf55a9d85f09ae5c458a01d31d5705987d66da0c5bdc7c5d0a2a275fb28e1",
                },
                {
                    "id": "e8b5e06133c6a6f20d4b69dd415047805858cf01bf3189dcee55b810bb317f513fbe25a0c2ce2b34185791797af6513cdecee569b3d83c09e5217cde87a7a110",
                    "signature": "304502205b25682ad1470e64114c31dd668f5ead94838231a652b326a9568422c4a96a8b022100e93f937308a98e30c28c415c13fa09b4698f068228052907975cc7a3d1bc4e93",
                },
                {
                    "id": "e8c7bdbd6362521a2d79cf61f0cb92ff31d4cefe7fb7d9d3f71d457dfcb4e59598436db10f8a89e7d8a1e7686d73c48c230bf93fe42931334075eb1b16c40398",
                    "signature": "304602210097a315d60663a94abff372cec4674a5b1591d0b1e133fa86c8e2c9538b1f161f022100e58d6ad891e6da9c615980c42344216be9e62d51a898a676a8c8d5742ab85896",
                },
                {
                    "id": "e95e16342c5a5c26bb8ee33e7514cf224303849793392e30e3e1c29fcad69eaae3d9a5bf11e683401c749b061e0f515198f0011c5c234795851def26c4bb382c",
                    "signature": "3045022100c1ee6751747b143a9e1f3850fb986d40cc0a3870087f2a97f35f9522d306620202203fe1eb08907b3cce99b59f6329836c214373a2d6c6eabf76a30618e1232ae032",
                },
                {
                    "id": "ea48bf28a9dabd276ee29743f4233d0dbd809131800d410f5be300e119dc2fb6388f2c5fe99ec35071048f51db0d38dd4bc2ae5e0ba20afd09a20aeebf497ef3",
                    "signature": "3044022062740c7ae36cfad94078bac1d57e0705d3fd4bdcc94370a5d0ceaafdc6ec7e1402203fb62662832b1e32cee5833c485516470f78457dbccbb0d8e2b9c019b5985217",
                },
                {
                    "id": "ec5d822e97422e0c55841e891301351e8d29b7ef237da23a18f517f56b12cec2497599cbaaa51c25623427ea46ed3c6854dafd8a52362e87ac8e502d3dcc5eb8",
                    "signature": "3045022100ba623b441497e58059a67be8d91b85f5c5f9c6c1810ffc5e0e94c7865ca22f68022045d73e34d1c9c0bbed38cd8ea450fafabe0a9a3b57faea325600ea3a8d3ca09c",
                },
                {
                    "id": "f3394033a420c80e82dd88ec54cc847d50457b85dce4b8ddfbc164f2227120c1c045c80d32de035a42bfe32ccf47eb6f81f81f2357bb3f6b0cfe93f23bb98e34",
                    "signature": "304502204f78ababa68f3aa1b3e92639f7bc277fdf630ab73e4fa31d7cdc26e41859a850022100ccb9be41d901339561caa9cfa4982118903352cb6251d4bdb5a4737f29aca040",
                },
                {
                    "id": "f449f9ddc44f26c627f50206d4ee84240db005d40c8e7f1e8012d8f6bb170ffd898f9cae855f02d46a6111d8e330635ac1bbc545ec21c318ad3810350027d1aa",
                    "signature": "3045022066b71f29c9604f89e85e3b886d052d26cf06491a1da39ba8fcc7ea3eaed66a10022100bbe81be1c99aa30cbe91deafc8baf672ea2a45aa87fc67159194e859c6ed700f",
                },
                {
                    "id": "f64159a4cbc53efe82345176f3255112e193453ba2ffea171936b496b74b8751ce54d1e847707e5e641be1631bb8a2d74aa92f43ff3491c8caac43a1e5df6343",
                    "signature": "3045022100bc9d8d2323fccdd0631894432dd32a2a5e643c13df39cb8c2f3cdacb0a710b1f02206288a9d71659572a85b19ea5befe32e80d3b9dde5210f0a2879a8bc5237fec94",
                },
                {
                    "id": "f68671f189746a8403be273e9c291d6653efb980e41d68def96fc59cb2b0af69e4d017c77123913d47168bb70860b2d152991afaa12f7d4c91636eedfcec877c",
                    "signature": "3046022100e1d4e33249985c64ccdfb034f809a0b0e0cfa83146b72f6648013cf9660b6d330221009e18f3e5d289fa321837e408e8832f7d2a84a92f284902e6cdc5e6bab8215dba",
                },
                {
                    "id": "f6f76ca647393d431cc33036922a86b890525894b94ee5d17929456d324b2430c157d72a3856690e210230a37fb8d38701d7981d4eea4720713a32e03193b6d8",
                    "signature": "304602210089a358c38fa3dc9a9a787368664d7144bcc0f989aaa98782eacde8f03a88b9cf022100bb0f3ec94c6514e4c0b233287751055fbeba2a2f4dc5b7cb7a6bf42f364f4ec2",
                },
                {
                    "id": "fbf5a151d5665120dfd2afe5c23aed3e9b3161091ed971cc9787c70e73878a478439923f855a3892e928f7b454b084faa10593f01bc24811f3f836e12a779ede",
                    "signature": "3045022100e68ccca63d38d0012d683a31887ec4309ae020da8c1475ca8afcd4c4a9f2ad95022035534ec2ab2f043a6c1729b1f6ec23d0491da472b6299ab404a556e22a4f0a46",
                },
                {
                    "id": "fc10de13c0aa4f01d48263e38a289bbfe419b921593b7d37b583b0a31452f27a92de4aacbde92e79f8c71721e22a20a6e04252c61879821d613edd94d22a0f15",
                    "signature": "3046022100f879374f8cdb9aedfa4ce274b277448940b0b7ee7b4e9c14fdd9873ca9a2bbea022100e8b60a64078af5ae6d626faf1a7beda39bf2bf939981f79e9a90e56959f02e84",
                },
                {
                    "id": "fc71ed4d9b40835b831dd1380a22cf8a0884ab042502dda77c3432b6cc134436b2df85ba1a9705d50fd95debb50184e7a7999e7f70416b24e99cbfabf32fdf01",
                    "signature": "30450221009c0c9e115dc1fd5c8426e1d0683f19042bb53918ecba677875f05cb08d4fbffe02204b77f23d958f27c201b9f4a0869df6eb22d858933b37581ec14effe9e68e4122",
                },
                {
                    "id": "fced2fdb7a089ff569fe34bd46c672cdc9bd9fb647f1d6db804402b60d4186f4367003c008636f1b9f14cbf025ba10513e6c673c50459f0706d7f33f7b875664",
                    "signature": "304602210092e07458bd4e431f58417034ee571514271cf2f71908db15ecdf8ab936c3ed6a022100fe344c35f1e83ec6ab57f30d4547039684c7ca9d27550ce24dc0bfee0dc6b596",
                },
                {
                    "id": "fe56dcf085dcabf63219a467f2f18ebcf86db9fe7179f61e0a707888783061ca2cab802c88f654f032d9dea40370145701a7880039c5a796a616d8bb59c47287",
                    "signature": "30460221008718c253c5da586a2cf59d3ebcd0b7f8b126cb72889b82f9d0ed12c10c7d1671022100cc2d2ecb7404993e75a36101b5c836d5ef40a5eadefef63ba71eb2738bc7263e",
                },
                {
                    "id": "ff7284b0d0cf801f173c37a4b73de28d1f9d49a72c8d35ca4828e02a8e8787b6ea74f730cbc41d849e17cda8052cf81c7520d9cf56022f9342c29958876afdac",
                    "signature": "304402200b2b8c5f6539e1fc6e5ee6574ceb7c301c4651af250438569f9a149ec354c3c9022049de35c47ef4cc2c86d20326a9e680e844ab6f75e389a5e67ba532ee78edddc1",
                },
            ],
        }

    @pytest.mark.asyncio
    async def test_get_latest_snapshot_ordinal(
        self, network, httpx_mock: HTTPXMock, mock_l0_api_responses
    ):
        network.config("integrationnet")
        httpx_mock.add_response(
            url="https://l0-lb-integrationnet.constellationnetwork.io/global-snapshots/latest/ordinal",
            json=mock_l0_api_responses["latest_snapshot_ordinal"],
        )
        result = await network.l0_api.get_latest_snapshot_ordinal()
        assert result.model_dump() == {"ordinal": 3382271}


@pytest.mark.integration
class TestIntegrationL0API:
    """Test L0 API endpoints integration responses"""

    @pytest.mark.asyncio
    async def test_get_address_balance(self, network):
        network.config("integrationnet")
        address = "DAG0zJW14beJtZX2BY2KA9gLbpaZ8x6vgX4KVPVX"
        result = await network.get_address_balance(address)
        result = result.model_dump()
        assert bool(result.get("ordinal") and result.get("balance"))

    @pytest.mark.asyncio
    async def test_get_total_supply(self, network):
        network.config("integrationnet")
        result = await network.l0_api.get_total_supply()
        result = result.model_dump()
        assert bool(result.get("ordinal") and result.get("total_supply"))

    @pytest.mark.asyncio
    async def test_get_cluster_info(self, network):
        network.config("integrationnet")
        results = await network.l0_api.get_cluster_info()
        assert isinstance(results, list)

    @pytest.mark.asyncio
    async def test_get_latest_l0_snapshot(self, network):
        result = await network.l0_api.get_latest_snapshot()
        result = result.model_dump()
        assert isinstance(result.get("value"), dict)

    @pytest.mark.asyncio
    async def test_get_latest_snapshot_ordinal(self, network):
        result = await network.l0_api.get_latest_snapshot_ordinal()
        result = result.model_dump()
        assert result.get("ordinal")

    @pytest.mark.asyncio
    async def test_get_metrics(self, network):
        try:
            r = await network.l0_api.get_metrics()
            for x in r:
                print(x)
            assert isinstance(r, list)
        except httpx.ReadTimeout:
            pytest.skip("Timeout")
