# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['jurigged', 'jurigged.loop']

package_data = \
{'': ['*']}

install_requires = \
['blessed>=1.17.12,<2.0.0',
 'codefind>=0.1.3,<0.2.0',
 'ovld>=0.3.1,<0.4.0',
 'watchdog>=1.0.2,<2.0.0']

extras_require = \
{'develoop': ['giving>=0.3.6,<0.4.0',
              'rich>=10.13.0,<11.0.0',
              'hrepr>=0.4.0,<0.5.0']}

entry_points = \
{'console_scripts': ['jurigged = jurigged.live:cli']}

setup_kwargs = {
    'name': 'jurigged',
    'version': '0.4.1',
    'description': 'Live update of Python functions',
    'long_description': '\n# jurigged\n\nJurigged lets you update your code while it runs. Using it is trivial:\n\n1. `jurigged your_script.py`\n2. Change some function or method with your favorite editor and save the file\n3. Jurigged will hot patch the new function into the running script\n\nJurigged updates live code smartly: changing a function or method will fudge code pointers so that all existing instances are simultaneously modified to implement the new behavior. When modifying a module, only changed lines will be re-run.\n\n![demo](https://user-images.githubusercontent.com/599820/113785377-ffab1e80-9704-11eb-9c82-6d866c9087a6.gif)\n\nYou can also optionally install the [develoop](#develoop), a terminal-based live development environment:\n\n![develoop](https://user-images.githubusercontent.com/599820/141145793-1526b45d-c99b-420e-8b06-d64e14e690a2.gif)\n\nIn the demo above, the function decorated with `__.loop` is re-run every time the source code is modified, with changes hot-patched into the running process. The rest of the program is *not* re-run, so preprocessing is preserved and heavy modules do not have to be reloaded!\n\n## Install\n\nJurigged requires Python version >= 3.8.\n\n```bash\npip install jurigged\n```\n\nTo also install the develoop feature, which lets you interactively develop functions:\n\n```bash\npip install jurigged[develoop]\n```\n\n## Command line\n\nThe simplest way to use jurigged is to add `-m jurigged` to your script invocation, or to use `jurigged` instead of `python`. You can use `-v` to get feedback about what files are watched and what happens when you change a file.\n\n```bash\npython -m jurigged -v script.py\n\nOR\n\njurigged -v script.py\n```\n\nWith no arguments given, it will start a live REPL:\n\n```bash\npython -m jurigged\n\nOR\n\njurigged\n```\n\nFull help:\n\n```\nusage: jurigged [-h] [--interactive] [--watch PATH] [--debounce DEBOUNCE] [--poll POLL] [-m MODULE] [--dev] [--verbose] [--version]\n                [SCRIPT] ...\n\nRun a Python script so that it is live-editable.\n\npositional arguments:\n  SCRIPT                Path to the script to run\n  ...                   Script arguments\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --interactive, -i     Run an interactive session after the program ends\n  --watch PATH, -w PATH\n                        Wildcard path/directory for which files to watch\n  --debounce DEBOUNCE, -d DEBOUNCE\n                        Interval to wait for to refresh a modified file, in seconds\n  --poll POLL           Poll for changes using the given interval\n  -m MODULE             Module or module:function to run\n  --dev                 Inject jurigged.loop.__ in builtins\n  --verbose, -v         Show watched files and changes as they happen\n  --version             Print version\n```\n\n## Develoop\n\nThe "develoop" is an optional feature of Jurigged that provides a sort of live development environment for a function. If you run `jurigged --dev script.py`, a variable called `__` is injected into the builtins (you may also import it as `from jurigged.loop import __`). If a function is decorated with `__.loop`, then when it is entered, it will be run, its output will be captured and displayed, and the program will wait for input. If the source code is changed, the function will run again.\n\nAlternatively, `__.xloop` will only enter the loop if the call raises an exception. That way, you get to debug it.\n\nHere is the current functionality offered through `__`:\n\n* `__.loop`: Loop over a function call.\n* `__.xloop`: Loop over a function call only if it raises an exception.\n* `__.give`: Displays a value in a table. See [giving](https://github.com/breuleux/giving).\n\nThe default interface allows a few commands:\n\n* `r` to manually re-run the loop. This can be done in the middle of a run.\n* `a` to abort the current run (e.g. if you get stuck in an infinite loop).\n* `c` to exit the loop and continue the program normally.\n* `q` to quit the program altogether.\n\n### Using with stdin\n\nThe default develoop interface does not play well with stdin. If you want to read from stdin or set a `breakpoint()`, use the decorator `@__.loop(interface="basic")`. The interface will be cruder, but stdin/pdb will work.\n\n## Troubleshooting\n\nFirst, if there\'s a problem, use the verbose flag (`jurigged -v`) to get more information. It will output a `Watch <file>` statement for every file that it watches and `Update/Add/Delete <function>` statements when you update, add or delete a function in the original file and then save it.\n\n**The file is not being watched.**\n\nBy default, scripts are watched in the current working directory. Try `jurigged -w <file>` to watch a specific file, or `jurigged -w /` to watch all files.\n\n**The file is watched, but nothing happens when I change the function.**\n\nYou can try using the `--poll <INTERVAL>` flag to use polling instead of the OS\'s native mechanisms. If that doesn\'t work, try and see if it works with a different editor: it might have to do with the way the editor saves. For example, some editors such as vi save into a temporary swap file and moves it into place, which used to cause issues (this should be fixed starting with `v0.3.5`).\n\n**Jurigged said it updated the function but it\'s still running the old code.**\n\nIf you are editing the body of a for loop inside a function that\'s currently running, the changes will only be in effect the next time that function is called. A workaround is to extract the body of the for loop into its own helper function, which you can then edit. Alternatively, you can use [reloading](https://github.com/julvo/reloading) alongside Jurigged.\n\nSimilarly, updating a generator or async function will not change the behavior of generators or async functions that are already running.\n\n**I can update some functions but not others.**\n\nThere may be issues updating some functions when they are decorated or stashed in some data structure that Jurigged does not understand. Jurigged does have to find them to update them, unfortunately.\n\n\n## API\n\nYou can call `jurigged.watch()` to programmatically start watching for changes. This should also work within IPython or Jupyter as an alternative to the `%autoreload` magic.\n\n```python\nimport jurigged\njurigged.watch()\n```\n\nBy default all files in the current directory will be watched, but you can use `jurigged.watch("script.py")` to only watch a single file, or `jurigged.watch("/")` to watch all modules.\n\n\n### Recoders\n\nFunctions can be programmatically changed using a Recoder. Make one with `jurigged.make_recoder`. This can be used to implement hot patching or mocking. The changes can also be written back to the filesystem.\n\n```python\nfrom jurigged import make_recoder\n\ndef f(x):\n    return x * x\n\nassert f(2) == 4\n\n# Change the behavior of the function, but not in the original file\nrecoder = make_recoder(f)\nrecoder.patch("def f(x): return x * x * x")\nassert f(2) == 8\n\n# Revert changes\nrecoder.revert()\nassert f(2) == 4\n\n# OR: write the patch to the original file itself\nrecoder.commit()\n```\n\n`revert` will only revert up to the last `commit`, or to the original contents if there was no commit.\n\nA recoder also allows you to add imports, helper functions and the like to a patch, but you have to use `recoder.patch_module(...)` in that case.\n\n\n## Caveats\n\nJurigged works in a surprisingly large number of situations, but there are several cases where it won\'t work, or where problems may arise:\n\n* **Functions that are already running will keep running with the existing code.** Only the next invocations will use the new code.\n  * When debugging with a breakpoint, functions currently on the stack can\'t be changed.\n  * A running generator or async function won\'t change.\n  * You can use [reloading](https://github.com/julvo/reloading) in addition to Jurigged if you want to be able to modify a running for loop.\n* **Changing initializers or attribute names may cause errors on existing instances.**\n  * Jurigged modifies all existing instances of a class, but it will not re-run `__init__` or rename attributes on existing instances, so you can easily end up with broken objects (new methods, but old data).\n* **Updating the code of a decorator or a closure may or may not work.** Jurigged will do its best, but it is possible that some closures will be updated but not others.\n* **Decorators that look at/tweak function code will probably not update properly.**\n  * Wrappers that try to compile/JIT Python code won\'t know about jurigged and won\'t be able to redo their work for the new code.\n  * They can be made to work using a `__conform__` method (see below).\n\n\n## Customizing behavior\n\nIn order to update a transform of a Python function, for example a transform that generates a new code object based on the original source code, you need to do something like this:\n\n```python\nclass Custom:\n    __slots__ = ("code",)\n\n    def __init__(self, transformed_fn, code):\n        self.code = code\n        self.transformed_fn = transformed_fn\n\n    def __conform__(self, new_code):\n        if new_code is None:\n            # Function is being deleted\n            ...\n\n        if isinstance(new_code, types.FunctionType):\n            new_code = new_code.__code__\n\n        do_something(new_code)\n        self.code = new_code\n\n...\ntransformed_fn.somefield = Custom(transformed_fn, orig_fn.__code__)\n```\n\nBasically, when the original code is changed, jurigged will use the `gc` module to find objects that point to it, and if they have a `__conform__` method it will be called with the new code so that the transformed function can be updated. The original code must be in a slot on that object (it is important that it is in `__slots__`, otherwise the referrer is a dictionary). Multiple transformed functions may exist.\n\n\n## How it works\n\nIn a nutshell, jurigged works as follows:\n\n1. Inventory existing modules and functions:\n  a. Insert an import hook that collects and watches source files.\n  b. Look at all existing functions using `gc.get_objects()`.\n  c. Add an audit hook that watches calls to `exec` in order to inventory any new functions.\n2. Parse source files into sets of definitions.\n3. When a file is modified, re-parse it into a set of definitions and match them against the original, yielding a set of changes, additions and deletions.\n4. When a function\'s code has changed:\n  a. Strip out the decorators\n  b. Execute the new code\n  c. Use `gc.get_referrers()` to find all functions that use the old code\n  d. Replace their internal `__code__` pointers\n5. If the replacement fails or if brand new code is added, execute the new code in the module namespace.\n\n\n## Comparison\n\nThe two most comparable implementations of Jurigged\'s feature set that I could find (but it can be a bit difficult to find everything comparable) are **%autoreload** in IPython and **[limeade](https://github.com/CFSworks/limeade)**. Here are the key differences:\n\n* They both re-execute the entire module when its code is changed. Jurigged, by contrast, surgically extracts changed functions from the parse tree and only replaces these. It only executes new or changed statements in a module.\n  \n  Which is better is somewhat situation-dependent: on one hand, re-executing the module will pick up more changes. On the other hand, it will reinitialize module variables and state, so certain things might break. Jurigged\'s approach is more conservative and will only pick up on modified functions, but it will not touch anything else, so I believe it is less likely to have unintended side effects. It also tells you what it is doing :)\n\n* They will re-execute decorators, whereas Jurigged will instead dig into them and update the functions it finds inside.\n  \n  Again, there\'s no objectively superior approach. `%autoreload` will properly re-execute changed decorators, but these decorators will return new objects, so if a module imports an already decorated function, it won\'t update to the new version. If you only modify the function\'s code and not the decorators, however, Jurigged will usually be able to change it inside the decorator, so all the old instances will use the new behavior.\n\n* They rely on synchronization points, whereas Jurigged can be run in its own thread.\n\n  This is a double-edged sword, because even though Jurigged can add live updates to existing scripts with zero lines of additional code, it is not thread safe at all (code could be executed in the middle of an update, which is possibly an inconsistent state).\n\nOther similar efforts:\n\n* [reloading](https://github.com/julvo/reloading) can wrap an iterator to make modifiable for loops. Jurigged cannot do that, but you can use both packages at the same time.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/jurigged',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
