class _WrappedException(object):
    """Allow wrapping exceptions in a new class while preserving the original
    as an attribute.

    Note that while Python 2 and 3 both have reasonable ways to handle this,
    they're mutually incompatible. This is a simple, portable approach that
    should be sufficient for most use cases.
    """
    def __init__(self, *args, **kwargs):
        if args:
            orig = args[0]
            if isinstance(orig, Exception):
                args[0] = str(orig)
                self.original_exception = getattr(orig, 'original_exception',
                                                  orig)
        super(_WrappedException, self).__init__(*args, **kwargs)


class _WithStatus(object):
    """Allow an optional status_code attribute on wrapped exceptions.

    This should allow inspecting HTTP-related errors without having to know
    details about the HTTP client library.
    """
    def __init__(self, *args, **kwargs):
        status_code = kwargs.pop('status_code', None)
        super(_WithStatus, self).__init__(*args, **kwargs)
        self.status_code = status_code


class ASAPAuthenticationException(_WrappedException, ValueError):
    """Base class for exceptions raised by this library

    Inherits from ValueError to maintain backward compatibility
    with clients that caught ValueError previously.
    """


class PublicKeyRetrieverException(_WithStatus, ASAPAuthenticationException):
    """Raise when there are issues retrieving the public key"""


class PrivateKeyRetrieverException(_WithStatus, ASAPAuthenticationException):
    """Raise when there are issues retrieving the private key"""


class KeyIdentifierException(ASAPAuthenticationException):
    """Raise when there are issues validating the key identifier"""
