"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImagePipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("node:path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const CryptoJS = require("crypto-js");
const yaml = require("js-yaml");
const IamPolicyBuilderHelper_1 = require("../IamPolicyBuilderHelper");
const CheckStateMachineStatus_function_1 = require("../Lambdas/CheckStateMachineStatus/CheckStateMachineStatus-function");
const StartStateMachine_function_1 = require("../Lambdas/StartStateMachine/StartStateMachine-function");
const StateMachine_1 = require("../StateMachine");
class ImagePipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let infrastructureConfig;
        let imageRecipe;
        this.imageRecipeComponents = [];
        this.imageId = '';
        this.imagePipelineArn = '';
        const account = aws_cdk_lib_1.Stack.of(this).account;
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const stackName = aws_cdk_lib_1.Stack.of(this).stackName;
        const topicKmsKey = new aws_kms_1.Key(this, 'TopicKmsKey', {
            description: 'KMS Key used to encrypt the SNS topic',
            alias: `/ami-builder/${stackName}-TopicKmsKey`,
        });
        // Construct code below
        const topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'ImageBuilderTopic', {
            displayName: 'Image Builder Notify',
            masterKey: topicKmsKey,
        });
        if (props.email != null) {
            topic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.EmailSubscription(props.email));
        }
        const role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'IAM role used as part of an Image Builder pipeline',
        });
        role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'));
        role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'));
        role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        if (typeof props.additionalPolicies !== 'undefined' && props.additionalPolicies.length >= 1) {
            for (const policy of props.additionalPolicies) {
                role.addManagedPolicy(policy);
            }
        }
        const instanceProfileName = props.profileName ?? `${stackName}-InstanceProfile`;
        const infraConfigName = `${stackName}-InfrastructureConfiguration`;
        const imageRecipeName = `${stackName}-ImageRecipe`;
        const imageRecipeVersion = props.imageRecipeVersion ?? '0.0.1';
        const pipelineBaseName = `${stackName}-ImagePipeline`;
        const pipelineName = `${pipelineBaseName}-${imageRecipeVersion.replace(/\./g, '-')}`;
        // Use the first volume configuration's KMS key ID as the effective KMS key ID
        // If the first volume configuration does not have a KMS key ID, use the default AMI encryption key
        // Keep in mind though that default AMI encryption key is not compatible with cross-account AMI sharing
        const distributionKeyAlias = props.distributionKmsKeyAlias ?? 'alias/aws/ebs';
        const ebsVolumeEffectiveKmsAlias = props.ebsVolumeConfigurations?.[0]?.ebs?.kmsKeyId
            ? props.ebsVolumeConfigurations[0].ebs.kmsKeyId
            : (props.distributionKmsKeyAlias ?? 'alias/aws/ebs');
        const distributionAccountIds = props.distributionAccountIds ?? [];
        const distributionRegions = props.distributionRegions ?? [];
        if ((props.distributionRegions?.length === 0 && props.distributionAccountIds?.length) ||
            (props.distributionRegions?.length && props.distributionAccountIds?.length === 0)) {
            throw new Error('distributionRegions must be specified if distributionAccountIds is specified');
        }
        if ((props.distributionRegions?.length && props.distributionAccountIds?.length) && props.distributionKmsKeyAlias === 'alias/aws/ebs') {
            throw new Error('If distributionRegions and distributionAccountIds are specified, distributionKmsKeyAlias must be specified');
        }
        const profile = new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: instanceProfileName,
        });
        const defaultGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: true,
        });
        if (props.securityGroups == null || props.subnetId == null) {
            infrastructureConfig = new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: instanceProfileName,
                name: infraConfigName,
                description: 'Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
                securityGroupIds: [defaultGroup.securityGroupId],
                subnetId: props.vpc.publicSubnets[0].subnetId,
            });
        }
        else {
            infrastructureConfig = new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: instanceProfileName,
                name: infraConfigName,
                description: 'Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
                securityGroupIds: props.securityGroups,
                subnetId: props.subnetId,
            });
        }
        infrastructureConfig.addDependency(profile);
        /**
             * Image recipe configuration
             */
        let imageRecipeProps;
        imageRecipeProps = {
            components: [],
            name: imageRecipeName,
            parentImage: props.parentImage,
            version: imageRecipeVersion ?? '0.0.1',
        };
        if (props.userDataScript) {
            imageRecipeProps = {
                ...imageRecipeProps,
                additionalInstanceConfiguration: {
                    userDataOverride: props.userDataScript,
                },
            };
        }
        if (props.ebsVolumeConfigurations?.[0].ebs) {
            // Go over each configuration and set the default
            for (const config of props.ebsVolumeConfigurations) {
                // If encryption is not set, enable it
                if (config.ebs.encrypted === undefined) {
                    config.ebs.encrypted = true;
                }
                // If KMS Key ID is not set, use the default
                if (!config.ebs.kmsKeyId) {
                    config.ebs.kmsKeyId = ebsVolumeEffectiveKmsAlias;
                }
                // If deleteOnTermination is not set, enable it
                if (!config.ebs.deleteOnTermination) {
                    config.ebs.deleteOnTermination = true;
                }
            }
            imageRecipeProps = {
                ...imageRecipeProps,
                blockDeviceMappings: props.ebsVolumeConfigurations,
            };
        }
        else {
            imageRecipeProps = {
                ...imageRecipeProps,
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/sda1',
                        ebs: {
                            kmsKeyId: 'alias/aws/ebs',
                            encrypted: true,
                            volumeSize: 30,
                            volumeType: 'gp3',
                            deleteOnTermination: true,
                        },
                    },
                ],
            };
        }
        imageRecipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'ImageRecipe', imageRecipeProps);
        props.components.forEach((component, index) => {
            const componentName = component.name?.replace(/ /g, '-');
            component.name = componentName;
            const componentHash = this.stringToMD5(JSON.stringify(component));
            const componentDocumentHash = this.stringToMD5(JSON.stringify(component.componentDocument));
            const componentDocument = {
                name: componentName ?? `${stackName}-${componentDocumentHash}-${index}`,
                description: component.description,
                schemaVersion: component.schemaVersion ?? '1.0',
                phases: component.componentDocument.phases,
            };
            const data = yaml.dump(componentDocument);
            let newComponent = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, component.name ?? `Component-${index}`, {
                name: component.name ?? `${stackName.toLowerCase()}-${componentHash}-${index}`,
                description: component.description,
                platform: component.platform ?? 'Linux',
                version: component.componentVersion ?? props.imageRecipeVersion ?? '0.0.1',
                data: data,
            });
            // add the component to the Image Recipe
            this.imageRecipeComponents.push({ componentArn: newComponent.attrArn });
            imageRecipe.components = this.imageRecipeComponents;
        });
        const imageRecipeHash = this.stringToMD5(JSON.stringify(this.imageRecipeComponents));
        imageRecipe.name = `${imageRecipeName}-${imageRecipeHash}-${imageRecipeVersion.replace(/\./g, '-')}`;
        let imagePipelineProps;
        imagePipelineProps = {
            infrastructureConfigurationArn: infrastructureConfig.attrArn,
            name: pipelineName,
            description: 'Image pipeline',
            imageRecipeArn: imageRecipe.attrArn,
        };
        if (props.enableVulnScans) {
            imagePipelineProps = {
                ...imagePipelineProps,
                imageScanningConfiguration: {
                    imageScanningEnabled: props.enableVulnScans,
                    ecrConfiguration: {
                        repositoryName: props.vulnScansRepoName,
                        containerTags: props.vulnScansRepoTags,
                    },
                },
            };
        }
        if (distributionAccountIds.length > 0 && distributionRegions.length > 0) {
            const distributionsList = [];
            distributionRegions.forEach(distributionRegion => {
                const distributionConfig = {
                    region: distributionRegion,
                    amiDistributionConfiguration: {
                        //Capital case here because it's an object of type any, but capital case is what is expected in CloudFormation
                        //https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-imagebuilder-distributionconfiguration-amidistributionconfiguration.html
                        Name: `${imageRecipeName}-${distributionRegion}-{{imagebuilder:buildDate}}`,
                        Description: `copy AMI ${imageRecipeName} to ${distributionRegion}`,
                        TargetAccountIds: distributionAccountIds,
                        LaunchPermissionConfiguration: {
                            UserIds: distributionAccountIds,
                        },
                        KmsKeyId: distributionKeyAlias, //using default AWS-managed key if one isn't given is prohibited
                    },
                };
                distributionsList.push(distributionConfig);
            });
            const amiDistributionConfiguration = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'amiDistributionConfiguration', {
                name: `${imageRecipeName}-distribution-config-${imageRecipeVersion.replace(/\./g, '-')}`,
                description: `Cross account distribution settings for ${imageRecipeName}`,
                distributions: distributionsList,
            });
            imagePipelineProps = {
                ...imagePipelineProps,
                distributionConfigurationArn: amiDistributionConfiguration.attrArn,
            };
        }
        const imageBuildPipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'ImagePipeline', imagePipelineProps);
        this.imagePipelineArn = imageBuildPipeline.attrArn;
        const startStateMachineFunction = new StartStateMachine_function_1.StartStateMachineFunction(this, 'StartStateMachineFunction', {
            description: 'StartStateMachineFunction',
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.minutes(12),
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        IamPolicyBuilderHelper_1.Actions.states.StartExecution,
                        IamPolicyBuilderHelper_1.Actions.states.GetExecutionHistory,
                        IamPolicyBuilderHelper_1.Actions.states.ListExecutions,
                        IamPolicyBuilderHelper_1.Actions.states.DescribeExecution,
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        IamPolicyBuilderHelper_1.Actions.imagebuilder.ListImagePipelines,
                        IamPolicyBuilderHelper_1.Actions.imagebuilder.StartImagePipelineExecution,
                        IamPolicyBuilderHelper_1.Actions.imagebuilder.GetImage,
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'kms:*',
                    ],
                    // arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}/
                    resources: [aws_cdk_lib_1.Arn.format({
                            service: 'kms',
                            resource: 'alias/aws/ebs',
                            account: '*',
                            region: '*',
                            partition: 'aws',
                        })],
                }),
            ],
        });
        const startStateMachineFunctionLogGroup = new aws_logs_1.LogGroup(this, 'StartStateMachineFunctionLogGroup', {
            logGroupName: `/aws/lambda/${stackName}-StartStateMachineFunction`,
            retention: aws_logs_1.RetentionDays.ONE_DAY,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const cfnStartStateMachineFunction = startStateMachineFunction.node.defaultChild;
        cfnStartStateMachineFunction.addPropertyOverride('LoggingConfig', {
            LogGroup: startStateMachineFunctionLogGroup.logGroupName,
            LogFormat: 'Text',
        });
        const checkStateMachineStatusFunction = new CheckStateMachineStatus_function_1.CheckStateMachineStatusFunction(this, 'CheckStateMachineStatusFunction', {
            description: 'CheckStateMachineStatusFunction',
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.minutes(12),
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        IamPolicyBuilderHelper_1.Actions.states.DescribeExecution,
                        IamPolicyBuilderHelper_1.Actions.states.ListExecutions,
                        IamPolicyBuilderHelper_1.Actions.states.GetExecutionHistory,
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        IamPolicyBuilderHelper_1.Actions.imagebuilder.ListImagePipelines,
                        IamPolicyBuilderHelper_1.Actions.imagebuilder.StartImagePipelineExecution,
                        IamPolicyBuilderHelper_1.Actions.imagebuilder.GetImage,
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        IamPolicyBuilderHelper_1.Actions.ec2.ModifyImageAttribute,
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'kms:*',
                    ],
                    resources: [aws_cdk_lib_1.Arn.format({
                            service: 'kms',
                            resource: 'alias/aws/ebs',
                            account: '*',
                            region: '*',
                            partition: 'aws',
                        })],
                }),
            ],
        });
        const checkStateMachineFunctionLogGroup = new aws_logs_1.LogGroup(this, 'CheckStateMachineFunctionLogGroup', {
            logGroupName: `/aws/lambda/${stackName}-CheckStateMachineStatusFunction`,
            retention: aws_logs_1.RetentionDays.ONE_DAY,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const cfnCheckStateMachineStatusFunction = checkStateMachineStatusFunction.node.defaultChild;
        cfnCheckStateMachineStatusFunction.addPropertyOverride('LoggingConfig', {
            LogGroup: checkStateMachineFunctionLogGroup.logGroupName,
            LogFormat: 'Text',
        });
        const provider = new custom_resources_1.Provider(this, 'ResourceProvider', {
            onEventHandler: startStateMachineFunction,
            isCompleteHandler: checkStateMachineStatusFunction,
            totalTimeout: aws_cdk_lib_1.Duration.minutes(60),
            queryInterval: aws_cdk_lib_1.Duration.minutes(2),
        });
        const stateMachineRole = new aws_iam_1.Role(this, 'StateMachineRole', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('states.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaRole'),
            ],
            inlinePolicies: {
                StateMachinePolicy: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                    statements: [
                        new aws_cdk_lib_1.aws_iam.PolicyStatement({
                            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                            actions: [
                                IamPolicyBuilderHelper_1.Actions.xray.PutTraceSegments,
                                IamPolicyBuilderHelper_1.Actions.xray.PutTelemetryRecords,
                                IamPolicyBuilderHelper_1.Actions.xray.GetSamplingRules,
                                IamPolicyBuilderHelper_1.Actions.xray.GetSamplingTargets,
                            ],
                            resources: ['*'],
                        }),
                        new aws_cdk_lib_1.aws_iam.PolicyStatement({
                            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                            actions: [
                                'imagebuilder:*',
                            ],
                            resources: [`arn:aws:imagebuilder:${region ?? '*'}:${account ?? '*'}:image-pipeline/${pipelineBaseName.toLowerCase()}-*`],
                        }),
                        new aws_cdk_lib_1.aws_iam.PolicyStatement({
                            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                            actions: [
                                IamPolicyBuilderHelper_1.Actions.imagebuilder.GetImage,
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        const stateMachine = new StateMachine_1.StateMachine(this, 'StateMachine', {
            definition: JSON.parse(fs.readFileSync(path.join(__dirname, '..', 'StateMachineFiles', 'ImageBuildingStateMachine.json'), 'utf8').toString()),
            role: stateMachineRole,
        });
        const effectiveRemovalPolicy = props.debugImagePipeline === true ? aws_cdk_lib_1.RemovalPolicy.RETAIN : aws_cdk_lib_1.RemovalPolicy.DESTROY;
        const componentsHash = this.stringToMD5(JSON.stringify(props.components));
        const imagePipeline = new aws_cdk_lib_1.CustomResource(this, 'CustomResource', {
            serviceToken: provider.serviceToken,
            properties: {
                ImagePipelineArn: imageBuildPipeline.attrArn,
                StateMachineArn: stateMachine.stateMachineArn,
                Version: props.imageRecipeVersion ?? '0.0.1',
                ComponentsHash: componentsHash,
                PhysicalResourceId: `${stackName}-ImagePipeline-${imageRecipeVersion.replace(/\./g, '-')}`,
                AmiKmsId: distributionKeyAlias,
                SharedAccountIds: distributionAccountIds.join(','),
                DebugImagePipeline: props.debugImagePipeline === true ? 'true' : 'false',
            },
            removalPolicy: effectiveRemovalPolicy,
        });
        this.imageId = imagePipeline.getAttString('AmiId');
    }
    stringToMD5(inputString) {
        return CryptoJS.MD5(inputString).toString();
    }
}
exports.ImagePipeline = ImagePipeline;
_a = JSII_RTTI_SYMBOL_1;
ImagePipeline[_a] = { fqn: "@jjrawlins/cdk-ami-builder.ImagePipeline", version: "0.0.41" };
//# sourceMappingURL=data:application/json;base64,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