import ctypes 
import os
import sys
import traceback    

import matplotlib
from matplotlib.backends.backend_agg import FigureCanvasAgg
from matplotlib.backend_bases import cursors
from matplotlib.figure import Figure
from matplotlib.backends.backend_qt5 import TimerQT
from contextlib import contextmanager
import time
import numpy as np
import six

from PyQt5 import QtCore, QtGui, QtQuick, QtWidgets

DEBUG = False

to_qt_button = {
    1: QtCore.Qt.LeftButton
  , 2: QtCore.Qt.MidButton
  , 3: QtCore.Qt.RightButton
}

class MPLImageHelper(object):
    def __init__(self, img, ax, offset = 0.5):
        self.img = img
        self.ax = ax
        self.ax.format_coord = self.format_coord
        self.offset = offset
    
    def to_indices(self, x, y):
        numrows, numcols = self.img.shape[:2]
        col, row = (int(v+self.offset) for v in (x,y))
        if col>=0 and col<numcols and row>=0 and row<numrows:
            return col, row
        raise RuntimeError("out of bounds")

    def format_coord(self, x, y):
        try:
            col, row = self.to_indices(x,y)
            z = self.img[row,col]
            return 'x=%1.4f, y=%1.4f, value=%1.4f'%(x, y, z)
        except:
            return 'x=%1.4f, y=%1.4f'%(x, y)


class MatplotlibIconProvider(QtQuick.QQuickImageProvider):
    """ This class provide the matplotlib icons for the navigation toolbar.
    """

    def __init__(self, img_type = QtQuick.QQuickImageProvider.Pixmap):
        self.basedir = os.path.join(matplotlib.rcParams['datapath'], 'images')
        QtQuick.QQuickImageProvider.__init__(self, img_type)

    def requestImage(self, id, size):
        img = QtGui.QImage(os.path.join(self.basedir, id + '.png'))
        size = img.size()
        return img, size
        
    def requestPixmap(self, id, size):    
        img, size = self.requestImage(id, size)
        pixmap = QtGui.QPixmap.fromImage(img)
        
        return pixmap, size


class FigureCanvasQtQuickAgg(QtQuick.QQuickPaintedItem, FigureCanvasAgg):
    """ This class creates a QtQuick Item encapsulating a Matplotlib
        Figure and all the functions to interact with the 'standard'
        Matplotlib navigation toolbar.
    """

    # map Qt button codes to MouseEvent's ones:
    buttond = {
        QtCore.Qt.LeftButton: 1,
        QtCore.Qt.MidButton: 2,
        QtCore.Qt.RightButton: 3,
        # QtCore.Qt.XButton1: None,
        # QtCore.Qt.XButton2: None,
    }
    
    cursord = {
        cursors.MOVE: QtCore.Qt.SizeAllCursor,
        cursors.HAND: QtCore.Qt.PointingHandCursor,
        cursors.POINTER: QtCore.Qt.ArrowCursor,
        cursors.SELECT_REGION: QtCore.Qt.CrossCursor,
        cursors.WAIT: QtCore.Qt.WaitCursor,
    }
               
    messageChanged = QtCore.pyqtSignal(str)
    
    leftChanged = QtCore.pyqtSignal()
    rightChanged = QtCore.pyqtSignal()
    topChanged = QtCore.pyqtSignal()
    bottomChanged = QtCore.pyqtSignal()
    wspaceChanged = QtCore.pyqtSignal()
    hspaceChanged = QtCore.pyqtSignal()
    figureProviderChanged = QtCore.pyqtSignal()

    def __init__(self, figure, parent=None, coordinates=True):
        if DEBUG:
            print('FigureCanvasQtQuickAgg qtquick5: ', figure)
        # _create_qApp()
        if figure is None:
            figure = Figure((6.0, 4.0))

        QtQuick.QQuickPaintedItem.__init__(self, parent=parent)
        FigureCanvasAgg.__init__(self, figure=figure)

        self._drawRect = None
        self._figureProvider = None
        self.blitbox = None
        
        # Activate hover events and mouse press events
        self.setAcceptHoverEvents(True)
        self.setAcceptedMouseButtons(QtCore.Qt.AllButtons)
        
        self._agg_draw_pending = False
        
    def getFigure(self):
        return self.figure
    


    @QtCore.pyqtProperty(QtCore.QObject, notify=figureProviderChanged)
    def figureProvider(self):
        return self._figureProvider
    
    @figureProvider.setter
    def figureProvider(self, p):
        if p != self._figureProvider:
            if self._figureProvider is not None:
                self._figureProvider.setFigure(None) #detach old provider
            self._figureProvider = p
            self._figureProvider.setCanvas(self)
            self.figureProviderChanged.emit()

    def drawRectangle(self, rect):
        self._drawRect = rect
        self.update()

    def paint(self, p):
        """
        Copy the image from the Agg canvas to the qt.drawable.
        In Qt, all drawing should be done inside of here when a widget is
        shown onscreen.
        """
        # if the canvas does not have a renderer, then give up and wait for
        # FigureCanvasAgg.draw(self) to be called
        if not hasattr(self, 'renderer'):
            return

        if DEBUG:
            print('FigureCanvasQtQuickAgg.paint: ', self,
                  self.get_width_height())

        if self.blitbox is None:
            # matplotlib is in rgba byte order.  QImage wants to put the bytes
            # into argb format and is in a 4 byte unsigned int.  Little endian
            # system is LSB first and expects the bytes in reverse order
            # (bgra).

            if QtCore.QSysInfo.ByteOrder == QtCore.QSysInfo.LittleEndian:
                stringBuffer = np.asarray(self.renderer._renderer).take([2,1,0,3], axis=2).tobytes()
            else:
                stringBuffer = self.renderer.tostring_argb()

            refcnt = sys.getrefcount(stringBuffer)

            # convert the Agg rendered image -> qImage
            qImage = QtGui.QImage(stringBuffer, self.renderer.width,
                                  self.renderer.height,
                                  QtGui.QImage.Format_ARGB32)
            # get the rectangle for the image
            rect = qImage.rect()
            # p = QtGui.QPainter(self)
            # reset the image area of the canvas to be the back-ground color
            p.eraseRect(rect)
            # draw the rendered image on to the canvas
            p.drawPixmap(QtCore.QPoint(0, 0), QtGui.QPixmap.fromImage(qImage))

            # draw the zoom rectangle to the QPainter
            if self._drawRect is not None:
                p.setPen(QtGui.QPen(QtCore.Qt.black, 1, QtCore.Qt.DotLine))
                x, y, w, h = self._drawRect
                p.drawRect(x, y, w, h)

        else:
            bbox = self.blitbox
            l, b, r, t = bbox.extents
            w = int(r) - int(l)
            h = int(t) - int(b)
            t = int(b) + h
            reg = self.copy_from_bbox(bbox)
            stringBuffer = reg.to_string_argb()
            qImage = QtGui.QImage(stringBuffer, w, h,
                                  QtGui.QImage.Format_ARGB32)

            pixmap = QtGui.QPixmap.fromImage(qImage)
            p.drawPixmap(QtCore.QPoint(l, self.renderer.height-t), pixmap)

            # draw the zoom rectangle to the QPainter
            if self._drawRect is not None:
                p.setPen(QtGui.QPen(QtCore.Qt.black, 1, QtCore.Qt.DotLine))
                x, y, w, h = self._drawRect
                p.drawRect(x, y, w, h)
            
            self.blitbox = None

    def draw(self):
        """
        Draw the figure with Agg, and queue a request for a Qt draw.
        """
        # The Agg draw is done here; delaying causes problems with code that
        # uses the result of the draw() to update plot elements.

        FigureCanvasAgg.draw(self)
        self.update()

    def draw_idle(self):
        """
        Queue redraw of the Agg buffer and request Qt paintEvent.
        """
        # The Agg draw needs to be handled by the same thread matplotlib
        # modifies the scene graph from. Post Agg draw request to the
        # current event loop in order to ensure thread affinity and to
        # accumulate multiple draw requests from event handling.
        # TODO: queued signal connection might be safer than singleShot
        if not self._agg_draw_pending:
            self._agg_draw_pending = True
            QtCore.QTimer.singleShot(0, self.__draw_idle_agg)

    def __draw_idle_agg(self, *args):
        if self.height() < 0 or self.width() < 0:
            self._agg_draw_pending = False
            return
        try:
            FigureCanvasAgg.draw(self)
            self.update()
        except Exception:
            # Uncaught exceptions are fatal for PyQt5, so catch them instead.
            traceback.print_exc()
        finally:
            self._agg_draw_pending = False

    def blit(self, bbox=None):
        """
        Blit the region in bbox
        """
        # If bbox is None, blit the entire canvas. Otherwise
        # blit only the area defined by the bbox.
        if bbox is None and self.figure:
            bbox = self.figure.bbox

        self.blitbox = bbox
        l, b, w, h = bbox.bounds
        t = b + h
        self.repaint(l, self.renderer.height-t, w, h)       

    def geometryChanged(self, new_geometry, old_geometry):
        w = new_geometry.width()
        h = new_geometry.height()
        
        if (w <= 0.0) and (h <= 0.0):
            return
            
        if DEBUG:
            print('resize (%d x %d)' % (w, h))
            print("FigureCanvasQtQuickAgg.geometryChanged(%d, %d)" % (w, h))
        dpival = self.figure.dpi
        winch = w / dpival
        hinch = h / dpival
        self.figure.set_size_inches(winch, hinch)
        FigureCanvasAgg.resize_event(self)
        self.draw_idle()
        QtQuick.QQuickPaintedItem.geometryChanged(self, new_geometry, old_geometry)
        
    def hoverEnterEvent(self, event):
        FigureCanvasAgg.enter_notify_event(self, guiEvent=event, xy=(event.pos().x(), event.pos().x()))

    def hoverLeaveEvent(self, event):
        QtWidgets.QApplication.restoreOverrideCursor()
        FigureCanvasAgg.leave_notify_event(self, guiEvent=event)

    def hoverMoveEvent(self, event):
        x = event.pos().x()
        # flipy so y=0 is bottom of canvas
        y = self.figure.bbox.height - event.pos().y()
        FigureCanvasAgg.motion_notify_event(self, x, y, guiEvent=event)
        
        # if DEBUG: 
            # print('hover move')

    # hoverMoveEvent kicks in when no mouse buttons are pressed
    # otherwise mouseMoveEvent are emitted
    def mouseMoveEvent(self, event):
        x = event.x()
        # flipy so y=0 is bottom of canvas
        y = self.figure.bbox.height - event.y()
        FigureCanvasAgg.motion_notify_event(self, x, y, guiEvent=event)
        # if DEBUG: 
            # print('mouse move')

    def mousePressEvent(self, event):
        x = event.pos().x()
        # flipy so y=0 is bottom of canvas
        y = self.figure.bbox.height - event.pos().y()
        button = self.buttond.get(event.button())
        if button is not None:
            FigureCanvasAgg.button_press_event(self, x, y, button,
                                                guiEvent=event)
        if DEBUG:
            print('button pressed:', event.button())

    def mouseReleaseEvent(self, event):
        x = event.x()
        # flipy so y=0 is bottom of canvas
        y = self.figure.bbox.height - event.y()
        button = self.buttond.get(event.button())
        if button is not None:
            FigureCanvasAgg.button_release_event(self, x, y, button,
                                                  guiEvent=event)
        if DEBUG:
            print('button released')

    def mouseDoubleClickEvent(self, event):
        x = event.pos().x()
        # flipy so y=0 is bottom of canvas
        y = self.figure.bbox.height - event.pos().y()
        button = self.buttond.get(event.button())
        if button is not None:
            FigureCanvasAgg.button_press_event(self, x, y,
                                                button, dblclick=True,
                                                guiEvent=event)
        if DEBUG:
            print('button doubleclicked:', event.button())

    def wheelEvent(self, event):
        x = event.x()
        # flipy so y=0 is bottom of canvas
        y = self.figure.bbox.height - event.y()
        # from QWheelEvent::delta doc
        if event.pixelDelta().x() == 0 and event.pixelDelta().y() == 0:
            steps = event.angleDelta().y() / 120
        else:
            steps = event.pixelDelta().y()

        if steps != 0:
            FigureCanvasAgg.scroll_event(self, x, y, steps, guiEvent=event)
            if DEBUG:
                print('scroll event: '
                      'steps = %i ' % (steps))

    def keyPressEvent(self, event):
        key = self._get_key(event)
        if key is None:
            return
        FigureCanvasAgg.key_press_event(self, key, guiEvent=event)
        if DEBUG:
            print('key press', key)

    def keyReleaseEvent(self, event):
        key = self._get_key(event)
        if key is None:
            return
        FigureCanvasAgg.key_release_event(self, key, guiEvent=event)
        if DEBUG:
            print('key release', key)

    def _get_key(self, event):
        if event.isAutoRepeat():
            return None

        event_key = event.key()
        event_mods = int(event.modifiers())  # actually a bitmask

        # get names of the pressed modifier keys
        # bit twiddling to pick out modifier keys from event_mods bitmask,
        # if event_key is a MODIFIER, it should not be duplicated in mods
        mods = [name for name, mod_key, qt_key in MODIFIER_KEYS
                if event_key != qt_key and (event_mods & mod_key) == mod_key]
        try:
            # for certain keys (enter, left, backspace, etc) use a word for the
            # key, rather than unicode
            key = SPECIAL_KEYS[event_key]
        except KeyError:
            # unicode defines code points up to 0x0010ffff
            # QT will use Key_Codes larger than that for keyboard keys that are
            # are not unicode characters (like multimedia keys)
            # skip these
            # if you really want them, you should add them to SPECIAL_KEYS
            MAX_UNICODE = 0x10ffff
            if event_key > MAX_UNICODE:
                return None

            key = six.unichr(event_key)
            # qt delivers capitalized letters.  fix capitalization
            # note that capslock is ignored
            if 'shift' in mods:
                mods.remove('shift')
            else:
                key = key.lower()

        mods.reverse()
        return '+'.join(mods + [key])

    def new_timer(self, *args, **kwargs):
        """
        Creates a new backend-specific subclass of
        :class:`backend_bases.Timer`.  This is useful for getting
        periodic events through the backend's native event
        loop. Implemented only for backends with GUIs.

        optional arguments:

        *interval*
            Timer interval in milliseconds

        *callbacks*
            Sequence of (func, args, kwargs) where func(*args, **kwargs)
            will be executed by the timer every *interval*.
        """
        return TimerQT(*args, **kwargs)

    def flush_events(self):
        global qApp
        qApp.processEvents()

    def start_event_loop(self, timeout):
        FigureCanvasAgg.start_event_loop(self, timeout)

    start_event_loop.__doc__ = \
                             FigureCanvasAgg.start_event_loop.__doc__

    def stop_event_loop(self):
        FigureCanvasAgg.stop_event_loop(self)

    stop_event_loop.__doc__ = FigureCanvasAgg.stop_event_loop.__doc__

     
class FigureQtQuickAggToolbar(FigureCanvasQtQuickAgg):
    """ This class creates a QtQuick Item encapsulating a Matplotlib
        Figure and all the functions to interact with the 'standard'
        Matplotlib navigation toolbar.
    """
    
    cursord = {
        None: QtCore.Qt.ArrowCursor,
        cursors.MOVE: QtCore.Qt.SizeAllCursor,
        cursors.HAND: QtCore.Qt.PointingHandCursor,
        cursors.POINTER: QtCore.Qt.ArrowCursor,
        cursors.SELECT_REGION: QtCore.Qt.CrossCursor,
        cursors.WAIT: QtCore.Qt.WaitCursor,
    }
               
    messageChanged = QtCore.pyqtSignal(str)
    
    leftChanged = QtCore.pyqtSignal()
    rightChanged = QtCore.pyqtSignal()
    topChanged = QtCore.pyqtSignal()
    bottomChanged = QtCore.pyqtSignal()
    wspaceChanged = QtCore.pyqtSignal()
    hspaceChanged = QtCore.pyqtSignal()

    def __init__(self, figure, parent=None, coordinates=True):
        if DEBUG:
            print('FigureQtQuickAggToolbar qtquick5: ', figure)

        FigureCanvasQtQuickAgg.__init__(self, figure=figure, parent=parent)
        
        self._message = ""
        #
        # Attributes from NavigationToolbar2QT
        #
        self.coordinates = coordinates
        self._actions = {}
        
        # reference holder for subplots_adjust window
        self.adj_window = None
        #
        # Attributes from NavigationToolbar2
        #
        self.canvas = self.figure.canvas
        self.toolbar = self
        self._lastCursor = cursors.POINTER
        
        # a dict from axes index to a list of view limits
        self._views = matplotlib.cbook.Stack()
        self._positions = matplotlib.cbook.Stack()  # stack of subplot positions
        self._xypress = None  # the location and axis info at the time
                              # of the press
        self._idPress = None
        self._idRelease = None
        self._active = None
        self._lastCursor = None
        
        self._idDrag = self.canvas.mpl_connect(
            'motion_notify_event', self.mouse_move)

        self._ids_zoom = []
        self._zoom_mode = None

        self._button_pressed = None  # determined by the button pressed
                                     # at start

        self.mode = ''  # a mode string for the status bar
        self.set_history_buttons()
        
        #
        # Store margin
        #
        self._defaults = {}
        for attr in ('left', 'bottom', 'right', 'top', 'wspace', 'hspace', ):
            val = getattr(self.figure.subplotpars, attr)
            self._defaults[attr] = val
            setattr(self, attr, val)
    
    @QtCore.pyqtProperty('QString', notify=messageChanged)
    def message(self):
        return self._message
    
    @message.setter
    def message(self, msg):
        if msg != self._message:
            self._message = msg
            self.messageChanged.emit(msg)
    
    @QtCore.pyqtProperty('QString', constant=True)
    def defaultDirectory(self):
        startpath = matplotlib.rcParams.get('savefig.directory', '')
        return os.path.expanduser(startpath)
    
    @QtCore.pyqtProperty('QStringList', constant=True)
    def fileFilters(self):
        filetypes = self.canvas.get_supported_filetypes_grouped()
        sorted_filetypes = list(six.iteritems(filetypes))
        sorted_filetypes.sort()
        
        filters = []
        for name, exts in sorted_filetypes:
            exts_list = " ".join(['*.%s' % ext for ext in exts])
            filter = '%s (%s)' % (name, exts_list)
            filters.append(filter)
        
        return filters

    @QtCore.pyqtProperty('QString', constant=True)
    def defaultFileFilter(self):        
        default_filetype = self.canvas.get_default_filetype()
        
        selectedFilter = None
        for filter in self.fileFilters:
            exts = filter.split('(', maxsplit=1)[1]
            exts = exts[:-1].split()
            if default_filetype in exts:
                selectedFilter = filter
                break
        
        if selectedFilter is None:
            selectedFilter = self.fileFilters[0]
                
        return selectedFilter
    
    @QtCore.pyqtProperty(float, notify=leftChanged)
    def left(self):
        return self.figure.subplotpars.left
        
    @left.setter
    def left(self, value):
        if value != self.figure.subplotpars.left:
            self.figure.subplots_adjust(left=value)
            self.leftChanged.emit()
            
            self.figure.canvas.draw_idle()
    
    @QtCore.pyqtProperty(float, notify=rightChanged)
    def right(self):
        return self.figure.subplotpars.right
        
    @right.setter
    def right(self, value):
        if value != self.figure.subplotpars.right:
            self.figure.subplots_adjust(right=value)
            self.rightChanged.emit()
            
            self.figure.canvas.draw_idle()
    
    @QtCore.pyqtProperty(float, notify=topChanged)
    def top(self):
        return self.figure.subplotpars.top
        
    @top.setter
    def top(self, value):
        if value != self.figure.subplotpars.top:
            self.figure.subplots_adjust(top=value)
            self.topChanged.emit()
            
            self.figure.canvas.draw_idle()
    
    @QtCore.pyqtProperty(float, notify=bottomChanged)
    def bottom(self):
        return self.figure.subplotpars.bottom
        
    @bottom.setter
    def bottom(self, value):
        if value != self.figure.subplotpars.bottom:
            self.figure.subplots_adjust(bottom=value)
            self.bottomChanged.emit()
            
            self.figure.canvas.draw_idle()
    
    @QtCore.pyqtProperty(float, notify=hspaceChanged)
    def hspace(self):
        return self.figure.subplotpars.hspace
        
    @hspace.setter
    def hspace(self, value):
        if value != self.figure.subplotpars.hspace:
            self.figure.subplots_adjust(hspace=value)
            self.hspaceChanged.emit()
            
            self.figure.canvas.draw_idle()
    
    @QtCore.pyqtProperty(float, notify=wspaceChanged)
    def wspace(self):
        return self.figure.subplotpars.wspace
        
    @wspace.setter
    def wspace(self, value):
        if value != self.figure.subplotpars.wspace:
            self.figure.subplots_adjust(wspace=value)
            self.wspaceChanged.emit()
            
            self.figure.canvas.draw_idle()

    @contextmanager
    def _wait_cursor_for_draw_cm(self):
        """
        Set the cursor to a wait cursor when drawing the canvas.
        In order to avoid constantly changing the cursor when the canvas
        changes frequently, do nothing if this context was triggered during the
        last second.  (Optimally we'd prefer only setting the wait cursor if
        the *current* draw takes too long, but the current draw blocks the GUI
        thread).
        """
        self._draw_time, last_draw_time = (
            time.time(), getattr(self, "_draw_time", -np.inf))
        if self._draw_time - last_draw_time > 1:
            try:
                self.set_cursor(cursors.WAIT)
                yield
            finally:
                self.set_cursor(self._lastCursor)
        else:
            yield

    def mouse_move(self, event):
        self._set_cursor(event)

        if event.inaxes and event.inaxes.get_navigate():

            try:
                s = event.inaxes.format_coord(event.xdata, event.ydata)
            except (ValueError, OverflowError):
                pass
            else:
                artists = [a for a in event.inaxes.artists
                           if a.mouseover]

                if artists:

                    a = max(enumerate(artists), key=lambda x: x[1].zorder)[1]
                    if a is not event.inaxes.patch:
                        data = a.get_cursor_data(event)
                        if data is not None:
                            s += ' [{:s}]'.format(a.format_cursor_data(data))

                if len(self.mode):
                    self.message = '{:s}, {:s}'.format(self.mode, s)
                else:
                    self.message = s
        else:
            self.message = self.mode

    def dynamic_update(self):
        self.canvas.draw_idle()

    def push_current(self):
        """push the current view limits and position onto the stack"""
        views = []
        pos = []
        for a in self.canvas.figure.get_axes():
            views.append(a._get_view())
            # Store both the original and modified positions
            pos.append((
                a.get_position(True).frozen(),
                a.get_position().frozen()))
        self._views.push(views)
        self._positions.push(pos)
        self.set_history_buttons()

    def set_history_buttons(self):
        """Enable or disable back/forward button"""
        pass

    def _update_view(self):
        """Update the viewlim and position from the view and
        position stack for each axes
        """

        views = self._views()
        if views is None:
            return
        pos = self._positions()
        if pos is None:
            return
        for i, a in enumerate(self.canvas.figure.get_axes()):
            a._set_view(views[i])
            # Restore both the original and modified positions
            a.set_position(pos[i][0], 'original')
            a.set_position(pos[i][1], 'active')

        self.canvas.draw_idle()

    @QtCore.pyqtSlot()
    def home(self, *args):
        """Restore the original view"""
        self._views.home()
        self._positions.home()
        self.set_history_buttons()
        self._update_view()

    @QtCore.pyqtSlot()
    def forward(self, *args):
        """Move forward in the view lim stack"""
        self._views.forward()
        self._positions.forward()
        self.set_history_buttons()
        self._update_view()

    @QtCore.pyqtSlot()
    def back(self, *args):
        """move back up the view lim stack"""
        self._views.back()
        self._positions.back()
        self.set_history_buttons()
        self._update_view()

    def _set_cursor(self, event):
        if not event.inaxes or not self._active:
            if self._lastCursor != cursors.POINTER:
                self.set_cursor(cursors.POINTER)
                self._lastCursor = cursors.POINTER
        else:
            if self._active == 'ZOOM':
                if self._lastCursor != cursors.SELECT_REGION:
                    self.set_cursor(cursors.SELECT_REGION)
                    self._lastCursor = cursors.SELECT_REGION
            elif (self._active == 'PAN' and
                  self._lastCursor != cursors.MOVE):
                self.set_cursor(cursors.MOVE)

                self._lastCursor = cursors.MOVE

    def set_cursor(self, cursor):
        """
        Set the current cursor to one of the :class:`Cursors`
        enums values
        """
        if DEBUG:
            print('Set cursor', cursor)
        self.canvas.setCursor(self.cursord[cursor])

    def draw_with_locators_update(self):
        """Redraw the canvases, update the locators"""
        for a in self.canvas.figure.get_axes():
            xaxis = getattr(a, 'xaxis', None)
            yaxis = getattr(a, 'yaxis', None)
            locators = []
            if xaxis is not None:
                locators.append(xaxis.get_major_locator())
                locators.append(xaxis.get_minor_locator())
            if yaxis is not None:
                locators.append(yaxis.get_major_locator())
                locators.append(yaxis.get_minor_locator())

            for loc in locators:
                loc.refresh()
        self.canvas.draw_idle()

    def press(self, event):
        """Called whenever a mouse button is pressed."""
        pass

    def press_pan(self, event):
        """the press mouse button in pan/zoom mode callback"""

        if event.button == 1:
            self._button_pressed = 1
        elif event.button == 3:
            self._button_pressed = 3
        else:
            self._button_pressed = None
            return

        x, y = event.x, event.y

        # push the current view to define home if stack is empty
        if self._views.empty():
            self.push_current()

        self._xypress = []
        for i, a in enumerate(self.canvas.figure.get_axes()):
            if (x is not None and y is not None and a.in_axes(event) and
                    a.get_navigate() and a.can_pan()):
                a.start_pan(x, y, event.button)
                self._xypress.append((a, i))
                self.canvas.mpl_disconnect(self._idDrag)
                self._idDrag = self.canvas.mpl_connect('motion_notify_event',
                                                       self.drag_pan)

        self.press(event)

    def release(self, event):
        """this will be called whenever mouse button is released"""
        pass

    def release_pan(self, event):
        """the release mouse button callback in pan/zoom mode"""

        if self._button_pressed is None:
            return
        self.canvas.mpl_disconnect(self._idDrag)
        self._idDrag = self.canvas.mpl_connect(
            'motion_notify_event', self.mouse_move)
        for a, ind in self._xypress:
            a.end_pan()
        if not self._xypress:
            return
        self._xypress = []
        self._button_pressed = None
        self.push_current()
        self.release(event)
        self.draw_with_locators_update()

    def drag_pan(self, event):
        """the drag callback in pan/zoom mode"""

        for a, ind in self._xypress:
            #safer to use the recorded button at the press than current button:
            #multiple button can get pressed during motion...
            a.drag_pan(self._button_pressed, event.key, event.x, event.y)
        self.dynamic_update()

    @QtCore.pyqtSlot()
    def pan(self, *args):
        """Activate the pan/zoom tool. pan with left button, zoom with right"""
        # set the pointer icon and button press funcs to the
        # appropriate callbacks

        if self._active == 'PAN':
            self._active = None
        else:
            self._active = 'PAN'
        if self._idPress is not None:
            self._idPress = self.canvas.mpl_disconnect(self._idPress)
            self.mode = ''

        if self._idRelease is not None:
            self._idRelease = self.canvas.mpl_disconnect(self._idRelease)
            self.mode = ''

        if self._active:
            self._idPress = self.canvas.mpl_connect(
                'button_press_event', self.press_pan)
            self._idRelease = self.canvas.mpl_connect(
                'button_release_event', self.release_pan)
            self.mode = 'pan/zoom'
            self.canvas.widgetlock(self)
        else:
            self.canvas.widgetlock.release(self)

        for a in self.canvas.figure.get_axes():
            a.set_navigate_mode(self._active)

        self.message = self.mode

    def draw_rubberband(self, event, x0, y0, x1, y1):
        """Draw a rectangle rubberband to indicate zoom limits"""
        height = self.canvas.figure.bbox.height
        y1 = height - y1
        y0 = height - y0

        w = abs(x1 - x0)
        h = abs(y1 - y0)

        rect = [int(val)for val in (min(x0, x1), min(y0, y1), w, h)]
        self.canvas.drawRectangle(rect)

    def remove_rubberband(self):
        """Remove the rubberband"""
        self.canvas.drawRectangle(None)

    def _switch_on_zoom_mode(self, event):
        self._zoom_mode = event.key
        self.mouse_move(event)

    def _switch_off_zoom_mode(self, event):
        self._zoom_mode = None
        self.mouse_move(event)

    def drag_zoom(self, event):
        """the drag callback in zoom mode"""

        if self._xypress:
            x, y = event.x, event.y
            lastx, lasty, a, ind, view = self._xypress[0]

            # adjust x, last, y, last
            x1, y1, x2, y2 = a.bbox.extents
            x, lastx = max(min(x, lastx), x1), min(max(x, lastx), x2)
            y, lasty = max(min(y, lasty), y1), min(max(y, lasty), y2)

            if self._zoom_mode == "x":
                x1, y1, x2, y2 = a.bbox.extents
                y, lasty = y1, y2
            elif self._zoom_mode == "y":
                x1, y1, x2, y2 = a.bbox.extents
                x, lastx = x1, x2

            self.draw_rubberband(event, x, y, lastx, lasty)

    def press_zoom(self, event):
        """the press mouse button in zoom to rect mode callback"""
        # If we're already in the middle of a zoom, pressing another
        # button works to "cancel"
        if self._ids_zoom != []:
            for zoom_id in self._ids_zoom:
                self.canvas.mpl_disconnect(zoom_id)
            self.release(event)
            self.draw_with_locators_update()
            self._xypress = None
            self._button_pressed = None
            self._ids_zoom = []
            return

        if event.button == 1:
            self._button_pressed = 1
        elif event.button == 3:
            self._button_pressed = 3
        else:
            self._button_pressed = None
            return

        x, y = event.x, event.y

        # push the current view to define home if stack is empty
        if self._views.empty():
            self.push_current()

        self._xypress = []
        for i, a in enumerate(self.canvas.figure.get_axes()):
            if (x is not None and y is not None and a.in_axes(event) and
                    a.get_navigate() and a.can_zoom()):
                self._xypress.append((x, y, a, i, a._get_view()))

        id1 = self.canvas.mpl_connect('motion_notify_event', self.drag_zoom)
        id2 = self.canvas.mpl_connect('key_press_event',
                                      self._switch_on_zoom_mode)
        id3 = self.canvas.mpl_connect('key_release_event',
                                      self._switch_off_zoom_mode)

        self._ids_zoom = id1, id2, id3
        self._zoom_mode = event.key

        self.press(event)

    def release_zoom(self, event):
        """the release mouse button callback in zoom to rect mode"""
        for zoom_id in self._ids_zoom:
            self.canvas.mpl_disconnect(zoom_id)
        self._ids_zoom = []

        self.remove_rubberband()

        if not self._xypress:
            return

        last_a = []

        for cur_xypress in self._xypress:
            x, y = event.x, event.y
            lastx, lasty, a, ind, view = cur_xypress
            # ignore singular clicks - 5 pixels is a threshold
            # allows the user to "cancel" a zoom action
            # by zooming by less than 5 pixels
            if ((abs(x - lastx) < 5 and self._zoom_mode!="y") or
                    (abs(y - lasty) < 5 and self._zoom_mode!="x")):
                self._xypress = None
                self.release(event)
                self.draw_with_locators_update()
                return

            # detect twinx,y axes and avoid double zooming
            twinx, twiny = False, False
            if last_a:
                for la in last_a:
                    if a.get_shared_x_axes().joined(a, la):
                        twinx = True
                    if a.get_shared_y_axes().joined(a, la):
                        twiny = True
            last_a.append(a)

            if self._button_pressed == 1:
                direction = 'in'
            elif self._button_pressed == 3:
                direction = 'out'
            else:
                continue

            a._set_view_from_bbox((lastx, lasty, x, y), direction,
                                  self._zoom_mode, twinx, twiny)

        self.draw_with_locators_update()
        self._xypress = None
        self._button_pressed = None

        self._zoom_mode = None

        self.push_current()
        self.release(event)

    @QtCore.pyqtSlot()
    def zoom(self, *args):
        """Activate zoom to rect mode"""
        if self._active == 'ZOOM':
            self._active = None
        else:
            self._active = 'ZOOM'

        if self._idPress is not None:
            self._idPress = self.canvas.mpl_disconnect(self._idPress)
            self.mode = ''

        if self._idRelease is not None:
            self._idRelease = self.canvas.mpl_disconnect(self._idRelease)
            self.mode = ''

        if self._active:
            self._idPress = self.canvas.mpl_connect('button_press_event',
                                                    self.press_zoom)
            self._idRelease = self.canvas.mpl_connect('button_release_event',
                                                      self.release_zoom)
            self.mode = 'zoom rect'
            self.canvas.widgetlock(self)
        else:
            self.canvas.widgetlock.release(self)

        for a in self.canvas.figure.get_axes():
            a.set_navigate_mode(self._active)

        self.message = self.mode

    @QtCore.pyqtSlot()
    def tight_layout(self):
        self.figure.tight_layout()
        # self._setSliderPositions()
        self.draw_idle()

    @QtCore.pyqtSlot()
    def reset_margin(self):
        self.figure.subplots_adjust(**self._defaults)
        # self._setSliderPositions()
        self.draw_idle()
        
    @QtCore.pyqtSlot(str)
    def print_figure(self, fname, *args, **kwargs):
        if fname:

            fname = QtCore.QUrl(fname).url()
            # save dir for next time
            savefig_dir = os.path.dirname(six.text_type(fname))
            matplotlib.rcParams['savefig.directory'] = savefig_dir
            fname = six.text_type(fname)
        FigureCanvasAgg.print_figure(self, fname, *args, **kwargs)
        self.draw()
     
FigureCanvasQTAgg = FigureCanvasQtQuickAgg
FigureCanvasQTAggToolbar = FigureQtQuickAggToolbar