import numpy as np
from scipy import interpolate

# Μονός οπλισμός
array_μsd = np.array([0.000, 0.010, 0.020, 0.030, 0.040, 0.050, 0.060, 0.070, 0.080, 0.090, 0.100,
                      0.110, 0.120, 0.130, 0.140, 0.150, 0.160, 0.170, 0.180, 0.190, 0.200, 0.210,
                      0.220, 0.230, 0.240, 0.250, 0.260, 0.270, 0.280, 0.290, 0.300, 0.310, 0.320,
                      0.330, 0.340, 0.350, 0.360, 0.370])

array_ω = np.array([0.0000, 0.0101, 0.0203, 0.0306, 0.0410, 0.0515, 0.0621, 0.0728, 0.0836, 0.0946, 0.1058,
                    0.1170, 0.1285, 0.1401, 0.1519, 0.1638, 0.1759, 0.1882, 0.2007, 0.2134, 0.2263, 0.2395,
                    0.2529, 0.2665, 0.2804, 0.2946, 0.3091, 0.3239, 0.3391, 0.3546, 0.3706, 0.3869, 0.4038,
                    0.4211, 0.4391, 0.4576, 0.4768, 0.4968])

# Διπλός οπλισμός
array_μsd_double = np.linspace(0.30, 0.70, 21)

array_ω1_double = np.array([0.367, 0.387, 0.408, 0.428, 0.449, 0.469, 0.490,
                            0.510, 0.531, 0.551, 0.572, 0.592, 0.613, 0.633,
                            0.654, 0.674, 0.695, 0.716, 0.736, 0.757, 0.777])

array_ω2_double = np.array([0.004, 0.025, 0.045, 0.066, 0.086, 0.107, 0.128,
                            0.148, 0.169, 0.190, 0.210, 0.231, 0.251, 0.272,
                            0.293, 0.313, 0.334, 0.355, 0.375, 0.396, 0.416])

array_μsd_tbeam = np.array([0.000, 0.02, 0.04, 0.06, 0.08, 0.10, 0.12])

array_ω_tbeam = np.array([0.0000, 0.021, 0.042, 0.063, 0.085, 0.107, 0.131])


def ω_for_μsd(μsd):
    """ Υπολογισμός ω για μsd από πίνακα ορθογωνικών διατομών της Καραβεζύρογλου

    Args:
        μsd (float): Ανηγμένη ροπή

    Returns:
        float: ω

    """
    if μsd > array_μsd[-1]:
        res = ω_for_μsd_poly2(μsd)
    else:
        res = np.interp(μsd, array_μsd, array_ω)

    return res

def ω_for_μsd_poly2(μsd):
    """ Υπολογισμός ω για μsd από πίνακα ορθογωνικών διατομών της Καραβεζύρογλου

    Args:
        μsd (float): Ανηγμένη ροπή

    Returns:
        float: ω

    """

    return 1.1905*μsd**2 + 0.8731*μsd + 0.0043

def ω_for_μsd_tbeam(μsd):
    """ Υπολογισμός ω για μsd από πίνακα ορθογωνικών διατομών της Καραβεζύρογλου

    Args:
        μsd (float): Ανηγμένη ροπή

    Returns:
        float: ω

    """

    return np.interp(μsd, array_μsd_tbeam, array_ω_tbeam)


def ω1ω2_for_μsd(μsd):
    """ Υπολογισμός ω1 και ω2 για μsd από πίνακα ορθογωνικών διατομών της Καραβεζύρογλου για μsd >= 0.30

    Args:
        μsd (float): Ανηγμένη ροπή

    Returns:
        tuple: ω1, ω2

    """

    return np.interp(μsd, array_μsd_double, array_ω1_double), np.interp(μsd, array_μsd_double, array_ω2_double)


def ρ1ρ2_for_d2d(d2d, ω1):
    """ Υπολογισμός ρ1 και ρ2 για ω1 και d2/d από πίνακα ορθογωνικών διατομών της Καραβεζύρογλου για μsd >= 0.30

    Args:
        d2d (float): Ο λόγος d2/d
        ω1 (float):

    Returns:
        tuple: ρ1, ρ2

    """

    _array_d2d = np.linspace(0.03, 0.17, 8)

    _array_ω1 = np.array([0.363, 0.469, 0.572, 0.674, 0.777])

    _array_ρ1 = np.array([[1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000],
            [1.000, 1.005, 1.010, 1.015, 1.020, 1.026, 1.032, 1.038],
            [1.000, 1.008, 1.016, 1.024, 1.033, 1.042, 1.052, 1.062],
            [1.000, 1.01, 1.02, 1.03, 1.042, 1.053, 1.065, 1.078],
            [1.000, 1.011, 1.023, 1.035, 1.048, 1.061, 1.075, 1.09]])

    _array_ρ2 = np.array([1, 1.021, 1.043, 1.066, 1.09, 1.115, 1.141, 1.169])

    ρ1 = interpolate.interp2d(_array_d2d, _array_ω1, _array_ρ1, kind='linear')(d2d, ω1)[0]
    ρ2 = np.interp(d2d, _array_d2d, _array_ρ2)

    return ρ1, ρ2


