"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expiration = void 0;
/**
 * Represents a date of expiration.
 *
 * The amount can be specified either as a Date object, timestamp, Duration or string.
 */
class Expiration {
    constructor(date) {
        this.date = date;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expiration(d); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expiration.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expiration.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Expire at specified date, represented as a string
     *
     * @param s the string that represents date to expire at
     */
    static fromString(s) { return new Expiration(new Date(s)); }
    /**
     * Exipration Value in a formatted Unix Epoch Time in seconds
     */
    toEpoch() {
        return Math.round(this.date.getTime() / 1000);
    }
    /**
     * Check if Exipiration expires before input
     * @param t the duration to check against
     */
    isBefore(t) {
        return this.date < new Date(Date.now() + t.toMilliseconds());
    }
    /**
     * Check if Exipiration expires after input
     * @param t the duration to check against
     */
    isAfter(t) {
        return this.date > new Date(Date.now() + t.toMilliseconds());
    }
}
exports.Expiration = Expiration;
//# sourceMappingURL=data:application/json;base64,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