"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        this.env = {
            account: (_a = props.account) !== null && _a !== void 0 ? _a : this.stack.account,
            region: (_b = props.region) !== null && _b !== void 0 ? _b : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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