"use strict";
const lib_1 = require("../lib");
const util_1 = require("./util");
module.exports = {
    'simple case provider identified by service token'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            serviceToken: 'MyServiceToken',
            properties: {
                Prop1: 'boo',
                Prop2: 'bar',
            },
        });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyCustomResource: {
                    Type: 'AWS::CloudFormation::CustomResource',
                    Properties: {
                        ServiceToken: 'MyServiceToken',
                        Prop1: 'boo',
                        Prop2: 'bar',
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'resource type can be specified'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            serviceToken: 'MyServiceToken',
            resourceType: 'Custom::MyResourceType',
        });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyCustomResource: {
                    Type: 'Custom::MyResourceType',
                    Properties: {
                        ServiceToken: 'MyServiceToken',
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'removal policy'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            serviceToken: 'MyServiceToken',
            removalPolicy: lib_1.RemovalPolicy.RETAIN,
        });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyCustomResource: {
                    Type: 'AWS::CloudFormation::CustomResource',
                    Properties: {
                        ServiceToken: 'MyServiceToken',
                    },
                    UpdateReplacePolicy: 'Retain',
                    DeletionPolicy: 'Retain',
                },
            },
        });
        test.done();
    },
    'resource type must begin with "Custom::"'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // THEN
        test.throws(() => new lib_1.CustomResource(stack, 'MyCustomResource', {
            resourceType: 'MyResourceType',
            serviceToken: 'FooBar',
        }), /Custom resource type must begin with "Custom::"/);
        test.done();
    },
    'properties can be pascal-cased'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            serviceToken: 'MyServiceToken',
            pascalCaseProperties: true,
            properties: {
                prop1: 'boo',
                boom: {
                    onlyFirstLevel: 1234,
                },
            },
        });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyCustomResource: {
                    Type: 'AWS::CloudFormation::CustomResource',
                    Properties: {
                        ServiceToken: 'MyServiceToken',
                        Prop1: 'boo',
                        Boom: {
                            onlyFirstLevel: 1234,
                        },
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'pascal-casing of props is disabled by default'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            serviceToken: 'MyServiceToken',
            properties: {
                prop1: 'boo',
                boom: {
                    onlyFirstLevel: 1234,
                },
            },
        });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), {
            Resources: {
                MyCustomResource: {
                    Type: 'AWS::CloudFormation::CustomResource',
                    Properties: {
                        ServiceToken: 'MyServiceToken',
                        prop1: 'boo',
                        boom: {
                            onlyFirstLevel: 1234,
                        },
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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