"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
class TouchingAspect {
    constructor() {
        this.visits = new Array();
    }
    visit(node) {
        this.visits.push(node);
    }
}
class BogusStack extends lib_1.Stack {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'Resource', {
            type: 'CDK::Test::Resource',
        });
    }
}
function acctRegion(s) {
    return [s.account, s.region];
}
module.exports = {
    'Stack inherits unspecified part of the env from Stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'Stage', {
            env: { account: 'account', region: 'region' },
        });
        // WHEN
        const stack1 = new lib_1.Stack(stage, 'Stack1', { env: { region: 'elsewhere' } });
        const stack2 = new lib_1.Stack(stage, 'Stack2', { env: { account: 'tnuocca' } });
        // THEN
        test.deepEqual(acctRegion(stack1), ['account', 'elsewhere']);
        test.deepEqual(acctRegion(stack2), ['tnuocca', 'region']);
        test.done();
    },
    'envs are inherited deeply'(test) {
        // GIVEN
        const app = new lib_1.App();
        const outer = new lib_1.Stage(app, 'Stage', {
            env: { account: 'account', region: 'region' },
        });
        // WHEN
        const innerAcct = new lib_1.Stage(outer, 'Acct', { env: { account: 'tnuocca' } });
        const innerRegion = new lib_1.Stage(outer, 'Rgn', { env: { region: 'elsewhere' } });
        const innerNeither = new lib_1.Stage(outer, 'Neither');
        // THEN
        test.deepEqual(acctRegion(new lib_1.Stack(innerAcct, 'Stack')), ['tnuocca', 'region']);
        test.deepEqual(acctRegion(new lib_1.Stack(innerRegion, 'Stack')), ['account', 'elsewhere']);
        test.deepEqual(acctRegion(new lib_1.Stack(innerNeither, 'Stack')), ['account', 'region']);
        test.done();
    },
    'The Stage Assembly is in the app Assembly\'s manifest'(test) {
        // WHEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'Stage');
        new BogusStack(stage, 'Stack2');
        // THEN -- app manifest contains a nested cloud assembly
        const appAsm = app.synth();
        const artifact = appAsm.artifacts.find(x => x instanceof cxapi.NestedCloudAssemblyArtifact);
        test.ok(artifact);
        test.done();
    },
    'Stacks in Stage are in a different cxasm than Stacks in App'(test) {
        // WHEN
        const app = new lib_1.App();
        const stack1 = new BogusStack(app, 'Stack1');
        const stage = new lib_1.Stage(app, 'Stage');
        const stack2 = new BogusStack(stage, 'Stack2');
        // THEN
        const stageAsm = stage.synth();
        test.deepEqual(stageAsm.stacks.map(s => s.stackName), [stack2.stackName]);
        const appAsm = app.synth();
        test.deepEqual(appAsm.stacks.map(s => s.stackName), [stack1.stackName]);
        test.done();
    },
    'Can nest Stages inside other Stages'(test) {
        // WHEN
        const app = new lib_1.App();
        const outer = new lib_1.Stage(app, 'Outer');
        const inner = new lib_1.Stage(outer, 'Inner');
        const stack = new BogusStack(inner, 'Stack');
        // WHEN
        const appAsm = app.synth();
        const outerAsm = appAsm.getNestedAssembly(outer.artifactId);
        const innerAsm = outerAsm.getNestedAssembly(inner.artifactId);
        test.ok(innerAsm.tryGetArtifact(stack.artifactId));
        test.done();
    },
    'Default stack name in Stage objects incorporates the Stage name and no hash'(test) {
        // WHEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'MyStack');
        // THEN
        test.equal(stage.stageName, 'MyStage');
        test.equal(stack.stackName, 'MyStage-MyStack');
        test.done();
    },
    'Can not have dependencies to stacks outside the nested asm'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new BogusStack(app, 'Stack1');
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack2 = new BogusStack(stage, 'Stack2');
        // WHEN
        test.throws(() => {
            stack2.addDependency(stack1);
        }, /dependency cannot cross stage boundaries/);
        test.done();
    },
    'When we synth() a stage, prepare must be called on constructs in the stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        let prepared = false;
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'Stack');
        class HazPrepare extends lib_1.Construct {
            prepare() {
                prepared = true;
            }
        }
        new HazPrepare(stack, 'Preparable');
        // WHEN
        stage.synth();
        // THEN
        test.equals(prepared, true);
        test.done();
    },
    'When we synth() a stage, aspects inside it must have been applied'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'Stack');
        // WHEN
        const aspect = new TouchingAspect();
        lib_1.Aspects.of(stack).add(aspect);
        // THEN
        app.synth();
        test.deepEqual(aspect.visits.map(c => c.node.path), [
            'MyStage/Stack',
            'MyStage/Stack/Resource',
        ]);
        test.done();
    },
    'Aspects do not apply inside a Stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        new BogusStack(stage, 'Stack');
        // WHEN
        const aspect = new TouchingAspect();
        lib_1.Aspects.of(app).add(aspect);
        // THEN
        app.synth();
        test.deepEqual(aspect.visits.map(c => c.node.path), [
            '',
            'Tree',
        ]);
        test.done();
    },
    'Automatic dependencies inside a stage are available immediately after synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack1 = new lib_1.Stack(stage, 'Stack1');
        const stack2 = new lib_1.Stack(stage, 'Stack2');
        // WHEN
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', {
            type: 'CDK::Test::Resource',
        });
        new lib_1.CfnResource(stack2, 'Resource', {
            type: 'CDK::Test::Resource',
            properties: {
                OtherThing: resource1.ref,
            },
        });
        const asm = stage.synth();
        // THEN
        test.deepEqual(asm.getStackArtifact(stack2.artifactId).dependencies.map(d => d.id), [stack1.artifactId]);
        test.done();
    },
    'Assemblies can be deeply nested'(test) {
        // GIVEN
        const app = new lib_1.App({ runtimeInfo: false, treeMetadata: false });
        const level1 = new lib_1.Stage(app, 'StageLevel1');
        const level2 = new lib_1.Stage(level1, 'StageLevel2');
        new lib_1.Stage(level2, 'StageLevel3');
        // WHEN
        const rootAssembly = app.synth();
        // THEN
        test.deepEqual(rootAssembly.manifest.artifacts, {
            'assembly-StageLevel1': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1',
                    displayName: 'StageLevel1',
                },
            },
        });
        const assemblyLevel1 = rootAssembly.getNestedAssembly('assembly-StageLevel1');
        test.deepEqual(assemblyLevel1.manifest.artifacts, {
            'assembly-StageLevel1-StageLevel2': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1-StageLevel2',
                    displayName: 'StageLevel1/StageLevel2',
                },
            },
        });
        const assemblyLevel2 = assemblyLevel1.getNestedAssembly('assembly-StageLevel1-StageLevel2');
        test.deepEqual(assemblyLevel2.manifest.artifacts, {
            'assembly-StageLevel1-StageLevel2-StageLevel3': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1-StageLevel2-StageLevel3',
                    displayName: 'StageLevel1/StageLevel2/StageLevel3',
                },
            },
        });
        test.done();
    },
    'stage name validation'(test) {
        const app = new lib_1.App();
        new lib_1.Stage(app, 'abcd');
        new lib_1.Stage(app, 'abcd123');
        new lib_1.Stage(app, 'abcd123-588dfjjk');
        new lib_1.Stage(app, 'abcd123-588dfjjk.sss');
        new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid');
        test.throws(() => new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid '), /invalid stage name "abcd123-588dfjjk.sss_ajsid "/);
        test.throws(() => new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid/dfo'), /invalid stage name "abcd123-588dfjjk.sss_ajsid\/dfo"/);
        test.throws(() => new lib_1.Stage(app, '&'), /invalid stage name "&"/);
        test.throws(() => new lib_1.Stage(app, '45hello'), /invalid stage name "45hello"/);
        test.throws(() => new lib_1.Stage(app, 'f'), /invalid stage name "f"/);
        test.done();
    },
    'outdir cannot be specified for nested stages'(test) {
        // WHEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stage(app, 'mystage', { outdir: '/tmp/foo/bar' }), /"outdir" cannot be specified for nested stages/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zdGFnZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3Quc3RhZ2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLHlDQUF5QztBQUV6QyxnQ0FBaUc7QUEwUmpHLE1BQU0sY0FBYztJQUFwQjtRQUNrQixXQUFNLEdBQUcsSUFBSSxLQUFLLEVBQWMsQ0FBQztJQUluRCxDQUFDO0lBSFEsS0FBSyxDQUFDLElBQWdCO1FBQzNCLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3pCLENBQUM7Q0FDRjtBQUVELE1BQU0sVUFBVyxTQUFRLFdBQUs7SUFDNUIsWUFBWSxLQUFnQixFQUFFLEVBQVU7UUFDdEMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLGlCQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNoQyxJQUFJLEVBQUUscUJBQXFCO1NBQzVCLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQUVELFNBQVMsVUFBVSxDQUFDLENBQVE7SUFDMUIsT0FBTyxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQy9CLENBQUM7QUEzU0QsaUJBQVM7SUFDUCx1REFBdUQsQ0FBQyxJQUFVO1FBQ2hFLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7WUFDcEMsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFO1NBQzlDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM1RSxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUUzRSxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRTFELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyQkFBMkIsQ0FBQyxJQUFVO1FBQ3BDLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7WUFDcEMsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFO1NBQzlDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLFNBQVMsR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM1RSxNQUFNLFdBQVcsR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM5RSxNQUFNLFlBQVksR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFakQsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFDakYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxXQUFLLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUN0RixJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRXBGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1REFBdUQsQ0FBQyxJQUFVO1FBQ2hFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN0QyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFaEMsd0RBQXdEO1FBQ3hELE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUUzQixNQUFNLFFBQVEsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsWUFBWSxLQUFLLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUM1RixJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRWxCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2REFBNkQsQ0FBQyxJQUFVO1FBQ3RFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksVUFBVSxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDdEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxVQUFVLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRS9DLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDL0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBRTFFLE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMzQixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7UUFFeEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFDQUFxQyxDQUFDLElBQVU7UUFDOUMsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFN0MsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMzQixNQUFNLFFBQVEsR0FBRyxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzVELE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFOUQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBRW5ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2RUFBNkUsQ0FBQyxJQUFVO1FBQ3RGLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFL0MsT0FBTztRQUNQLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUUvQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNERBQTRELENBQUMsSUFBVTtRQUNyRSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLE1BQU0sR0FBRyxJQUFJLFVBQVUsQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDN0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUUvQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixNQUFNLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLENBQUMsRUFBRSwwQ0FBMEMsQ0FBQyxDQUFDO1FBRS9DLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0RUFBNEUsQ0FBQyxJQUFVO1FBQ3JGLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLElBQUksUUFBUSxHQUFHLEtBQUssQ0FBQztRQUNyQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzdDLE1BQU0sVUFBVyxTQUFRLGVBQVM7WUFDdEIsT0FBTztnQkFDZixRQUFRLEdBQUcsSUFBSSxDQUFDO1lBQ2xCLENBQUM7U0FDRjtRQUNELElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztRQUVwQyxPQUFPO1FBQ1AsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWQsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBRTVCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRUFBbUUsQ0FBQyxJQUFVO1FBQzVFLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFN0MsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxFQUFFLENBQUM7UUFDcEMsYUFBTyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFOUIsT0FBTztRQUNQLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUNaLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2xELGVBQWU7WUFDZix3QkFBd0I7U0FDekIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFDQUFxQyxDQUFDLElBQVU7UUFDOUMsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUUvQixPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxjQUFjLEVBQUUsQ0FBQztRQUNwQyxhQUFPLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUU1QixPQUFPO1FBQ1AsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ1osSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDbEQsRUFBRTtZQUNGLE1BQU07U0FDUCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkVBQTZFLENBQUMsSUFBVTtRQUN0RixRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUUxQyxPQUFPO1FBQ1AsTUFBTSxTQUFTLEdBQUcsSUFBSSxpQkFBVyxDQUFDLE1BQU0sRUFBRSxVQUFVLEVBQUU7WUFDcEQsSUFBSSxFQUFFLHFCQUFxQjtTQUM1QixDQUFDLENBQUM7UUFDSCxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRTtZQUNsQyxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsU0FBUyxDQUFDLEdBQUc7YUFDMUI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFMUIsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQ1osR0FBRyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUNuRSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBRXZCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpQ0FBaUMsQ0FBQyxJQUFVO1FBQzFDLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUUsWUFBWSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFakUsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNoRCxJQUFJLFdBQUssQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFakMsT0FBTztRQUNQLE1BQU0sWUFBWSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVqQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUM5QyxzQkFBc0IsRUFBRTtnQkFDdEIsSUFBSSxFQUFFLG9CQUFvQjtnQkFDMUIsVUFBVSxFQUFFO29CQUNWLGFBQWEsRUFBRSxzQkFBc0I7b0JBQ3JDLFdBQVcsRUFBRSxhQUFhO2lCQUMzQjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDLGlCQUFpQixDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDOUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUNoRCxrQ0FBa0MsRUFBRTtnQkFDbEMsSUFBSSxFQUFFLG9CQUFvQjtnQkFDMUIsVUFBVSxFQUFFO29CQUNWLGFBQWEsRUFBRSxrQ0FBa0M7b0JBQ2pELFdBQVcsRUFBRSx5QkFBeUI7aUJBQ3ZDO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxjQUFjLENBQUMsaUJBQWlCLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM1RixJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFO1lBQ2hELDhDQUE4QyxFQUFFO2dCQUM5QyxJQUFJLEVBQUUsb0JBQW9CO2dCQUMxQixVQUFVLEVBQUU7b0JBQ1YsYUFBYSxFQUFFLDhDQUE4QztvQkFDN0QsV0FBVyxFQUFFLHFDQUFxQztpQkFDbkQ7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1QkFBdUIsQ0FBQyxJQUFVO1FBQ2hDLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFFdEIsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZCLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUMxQixJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUNuQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztRQUN2QyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsNEJBQTRCLENBQUMsQ0FBQztRQUU3QyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSw2QkFBNkIsQ0FBQyxFQUFFLGtEQUFrRCxDQUFDLENBQUM7UUFDckgsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsZ0NBQWdDLENBQUMsRUFBRSxzREFBc0QsQ0FBQyxDQUFDO1FBQzVILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLHdCQUF3QixDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLEVBQUUsOEJBQThCLENBQUMsQ0FBQztRQUM3RSxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO1FBRWpFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO1FBQ3ZELE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBRXRCLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUUsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFLENBQUMsRUFBRSxnREFBZ0QsQ0FBQyxDQUFDO1FBQzNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgeyBBcHAsIENmblJlc291cmNlLCBDb25zdHJ1Y3QsIElBc3BlY3QsIElDb25zdHJ1Y3QsIFN0YWNrLCBTdGFnZSwgQXNwZWN0cyB9IGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ1N0YWNrIGluaGVyaXRzIHVuc3BlY2lmaWVkIHBhcnQgb2YgdGhlIGVudiBmcm9tIFN0YWdlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnU3RhZ2UnLCB7XG4gICAgICBlbnY6IHsgYWNjb3VudDogJ2FjY291bnQnLCByZWdpb246ICdyZWdpb24nIH0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKHN0YWdlLCAnU3RhY2sxJywgeyBlbnY6IHsgcmVnaW9uOiAnZWxzZXdoZXJlJyB9IH0pO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhzdGFnZSwgJ1N0YWNrMicsIHsgZW52OiB7IGFjY291bnQ6ICd0bnVvY2NhJyB9IH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKGFjY3RSZWdpb24oc3RhY2sxKSwgWydhY2NvdW50JywgJ2Vsc2V3aGVyZSddKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhY2N0UmVnaW9uKHN0YWNrMiksIFsndG51b2NjYScsICdyZWdpb24nXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZW52cyBhcmUgaW5oZXJpdGVkIGRlZXBseScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IG91dGVyID0gbmV3IFN0YWdlKGFwcCwgJ1N0YWdlJywge1xuICAgICAgZW52OiB7IGFjY291bnQ6ICdhY2NvdW50JywgcmVnaW9uOiAncmVnaW9uJyB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGlubmVyQWNjdCA9IG5ldyBTdGFnZShvdXRlciwgJ0FjY3QnLCB7IGVudjogeyBhY2NvdW50OiAndG51b2NjYScgfSB9KTtcbiAgICBjb25zdCBpbm5lclJlZ2lvbiA9IG5ldyBTdGFnZShvdXRlciwgJ1JnbicsIHsgZW52OiB7IHJlZ2lvbjogJ2Vsc2V3aGVyZScgfSB9KTtcbiAgICBjb25zdCBpbm5lck5laXRoZXIgPSBuZXcgU3RhZ2Uob3V0ZXIsICdOZWl0aGVyJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoYWNjdFJlZ2lvbihuZXcgU3RhY2soaW5uZXJBY2N0LCAnU3RhY2snKSksIFsndG51b2NjYScsICdyZWdpb24nXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYWNjdFJlZ2lvbihuZXcgU3RhY2soaW5uZXJSZWdpb24sICdTdGFjaycpKSwgWydhY2NvdW50JywgJ2Vsc2V3aGVyZSddKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhY2N0UmVnaW9uKG5ldyBTdGFjayhpbm5lck5laXRoZXIsICdTdGFjaycpKSwgWydhY2NvdW50JywgJ3JlZ2lvbiddKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdUaGUgU3RhZ2UgQXNzZW1ibHkgaXMgaW4gdGhlIGFwcCBBc3NlbWJseVxcJ3MgbWFuaWZlc3QnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ1N0YWdlJyk7XG4gICAgbmV3IEJvZ3VzU3RhY2soc3RhZ2UsICdTdGFjazInKTtcblxuICAgIC8vIFRIRU4gLS0gYXBwIG1hbmlmZXN0IGNvbnRhaW5zIGEgbmVzdGVkIGNsb3VkIGFzc2VtYmx5XG4gICAgY29uc3QgYXBwQXNtID0gYXBwLnN5bnRoKCk7XG5cbiAgICBjb25zdCBhcnRpZmFjdCA9IGFwcEFzbS5hcnRpZmFjdHMuZmluZCh4ID0+IHggaW5zdGFuY2VvZiBjeGFwaS5OZXN0ZWRDbG91ZEFzc2VtYmx5QXJ0aWZhY3QpO1xuICAgIHRlc3Qub2soYXJ0aWZhY3QpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrcyBpbiBTdGFnZSBhcmUgaW4gYSBkaWZmZXJlbnQgY3hhc20gdGhhbiBTdGFja3MgaW4gQXBwJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgQm9ndXNTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBzdGFnZSA9IG5ldyBTdGFnZShhcHAsICdTdGFnZScpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBCb2d1c1N0YWNrKHN0YWdlLCAnU3RhY2syJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3Qgc3RhZ2VBc20gPSBzdGFnZS5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWdlQXNtLnN0YWNrcy5tYXAocyA9PiBzLnN0YWNrTmFtZSksIFtzdGFjazIuc3RhY2tOYW1lXSk7XG5cbiAgICBjb25zdCBhcHBBc20gPSBhcHAuc3ludGgoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhcHBBc20uc3RhY2tzLm1hcChzID0+IHMuc3RhY2tOYW1lKSwgW3N0YWNrMS5zdGFja05hbWVdKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdDYW4gbmVzdCBTdGFnZXMgaW5zaWRlIG90aGVyIFN0YWdlcycodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgb3V0ZXIgPSBuZXcgU3RhZ2UoYXBwLCAnT3V0ZXInKTtcbiAgICBjb25zdCBpbm5lciA9IG5ldyBTdGFnZShvdXRlciwgJ0lubmVyJyk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgQm9ndXNTdGFjayhpbm5lciwgJ1N0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXBwQXNtID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3Qgb3V0ZXJBc20gPSBhcHBBc20uZ2V0TmVzdGVkQXNzZW1ibHkob3V0ZXIuYXJ0aWZhY3RJZCk7XG4gICAgY29uc3QgaW5uZXJBc20gPSBvdXRlckFzbS5nZXROZXN0ZWRBc3NlbWJseShpbm5lci5hcnRpZmFjdElkKTtcblxuICAgIHRlc3Qub2soaW5uZXJBc20udHJ5R2V0QXJ0aWZhY3Qoc3RhY2suYXJ0aWZhY3RJZCkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0RlZmF1bHQgc3RhY2sgbmFtZSBpbiBTdGFnZSBvYmplY3RzIGluY29ycG9yYXRlcyB0aGUgU3RhZ2UgbmFtZSBhbmQgbm8gaGFzaCcodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnTXlTdGFnZScpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IEJvZ3VzU3RhY2soc3RhZ2UsICdNeVN0YWNrJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5lcXVhbChzdGFnZS5zdGFnZU5hbWUsICdNeVN0YWdlJyk7XG4gICAgdGVzdC5lcXVhbChzdGFjay5zdGFja05hbWUsICdNeVN0YWdlLU15U3RhY2snKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdDYW4gbm90IGhhdmUgZGVwZW5kZW5jaWVzIHRvIHN0YWNrcyBvdXRzaWRlIHRoZSBuZXN0ZWQgYXNtJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IEJvZ3VzU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnTXlTdGFnZScpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBCb2d1c1N0YWNrKHN0YWdlLCAnU3RhY2syJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgc3RhY2syLmFkZERlcGVuZGVuY3koc3RhY2sxKTtcbiAgICB9LCAvZGVwZW5kZW5jeSBjYW5ub3QgY3Jvc3Mgc3RhZ2UgYm91bmRhcmllcy8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1doZW4gd2Ugc3ludGgoKSBhIHN0YWdlLCBwcmVwYXJlIG11c3QgYmUgY2FsbGVkIG9uIGNvbnN0cnVjdHMgaW4gdGhlIHN0YWdlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgbGV0IHByZXBhcmVkID0gZmFsc2U7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnTXlTdGFnZScpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IEJvZ3VzU3RhY2soc3RhZ2UsICdTdGFjaycpO1xuICAgIGNsYXNzIEhhelByZXBhcmUgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgICAgcHJvdGVjdGVkIHByZXBhcmUoKSB7XG4gICAgICAgIHByZXBhcmVkID0gdHJ1ZTtcbiAgICAgIH1cbiAgICB9XG4gICAgbmV3IEhhelByZXBhcmUoc3RhY2ssICdQcmVwYXJhYmxlJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhZ2Uuc3ludGgoKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmVxdWFscyhwcmVwYXJlZCwgdHJ1ZSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnV2hlbiB3ZSBzeW50aCgpIGEgc3RhZ2UsIGFzcGVjdHMgaW5zaWRlIGl0IG11c3QgaGF2ZSBiZWVuIGFwcGxpZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFnZSA9IG5ldyBTdGFnZShhcHAsICdNeVN0YWdlJyk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgQm9ndXNTdGFjayhzdGFnZSwgJ1N0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXNwZWN0ID0gbmV3IFRvdWNoaW5nQXNwZWN0KCk7XG4gICAgQXNwZWN0cy5vZihzdGFjaykuYWRkKGFzcGVjdCk7XG5cbiAgICAvLyBUSEVOXG4gICAgYXBwLnN5bnRoKCk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNwZWN0LnZpc2l0cy5tYXAoYyA9PiBjLm5vZGUucGF0aCksIFtcbiAgICAgICdNeVN0YWdlL1N0YWNrJyxcbiAgICAgICdNeVN0YWdlL1N0YWNrL1Jlc291cmNlJyxcbiAgICBdKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdBc3BlY3RzIGRvIG5vdCBhcHBseSBpbnNpZGUgYSBTdGFnZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlKGFwcCwgJ015U3RhZ2UnKTtcbiAgICBuZXcgQm9ndXNTdGFjayhzdGFnZSwgJ1N0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXNwZWN0ID0gbmV3IFRvdWNoaW5nQXNwZWN0KCk7XG4gICAgQXNwZWN0cy5vZihhcHApLmFkZChhc3BlY3QpO1xuXG4gICAgLy8gVEhFTlxuICAgIGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzcGVjdC52aXNpdHMubWFwKGMgPT4gYy5ub2RlLnBhdGgpLCBbXG4gICAgICAnJyxcbiAgICAgICdUcmVlJyxcbiAgICBdKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQXV0b21hdGljIGRlcGVuZGVuY2llcyBpbnNpZGUgYSBzdGFnZSBhcmUgYXZhaWxhYmxlIGltbWVkaWF0ZWx5IGFmdGVyIHN5bnRoJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2UoYXBwLCAnTXlTdGFnZScpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhzdGFnZSwgJ1N0YWNrMScpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhzdGFnZSwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHJlc291cmNlMSA9IG5ldyBDZm5SZXNvdXJjZShzdGFjazEsICdSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdDREs6OlRlc3Q6OlJlc291cmNlJyxcbiAgICB9KTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2syLCAnUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQ0RLOjpUZXN0OjpSZXNvdXJjZScsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIE90aGVyVGhpbmc6IHJlc291cmNlMS5yZWYsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgYXNtID0gc3RhZ2Uuc3ludGgoKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChcbiAgICAgIGFzbS5nZXRTdGFja0FydGlmYWN0KHN0YWNrMi5hcnRpZmFjdElkKS5kZXBlbmRlbmNpZXMubWFwKGQgPT4gZC5pZCksXG4gICAgICBbc3RhY2sxLmFydGlmYWN0SWRdKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdBc3NlbWJsaWVzIGNhbiBiZSBkZWVwbHkgbmVzdGVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKHsgcnVudGltZUluZm86IGZhbHNlLCB0cmVlTWV0YWRhdGE6IGZhbHNlIH0pO1xuXG4gICAgY29uc3QgbGV2ZWwxID0gbmV3IFN0YWdlKGFwcCwgJ1N0YWdlTGV2ZWwxJyk7XG4gICAgY29uc3QgbGV2ZWwyID0gbmV3IFN0YWdlKGxldmVsMSwgJ1N0YWdlTGV2ZWwyJyk7XG4gICAgbmV3IFN0YWdlKGxldmVsMiwgJ1N0YWdlTGV2ZWwzJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgcm9vdEFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwocm9vdEFzc2VtYmx5Lm1hbmlmZXN0LmFydGlmYWN0cywge1xuICAgICAgJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxJzoge1xuICAgICAgICB0eXBlOiAnY2RrOmNsb3VkLWFzc2VtYmx5JyxcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGRpcmVjdG9yeU5hbWU6ICdhc3NlbWJseS1TdGFnZUxldmVsMScsXG4gICAgICAgICAgZGlzcGxheU5hbWU6ICdTdGFnZUxldmVsMScsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgYXNzZW1ibHlMZXZlbDEgPSByb290QXNzZW1ibHkuZ2V0TmVzdGVkQXNzZW1ibHkoJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHlMZXZlbDEubWFuaWZlc3QuYXJ0aWZhY3RzLCB7XG4gICAgICAnYXNzZW1ibHktU3RhZ2VMZXZlbDEtU3RhZ2VMZXZlbDInOiB7XG4gICAgICAgIHR5cGU6ICdjZGs6Y2xvdWQtYXNzZW1ibHknLFxuICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgZGlyZWN0b3J5TmFtZTogJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxLVN0YWdlTGV2ZWwyJyxcbiAgICAgICAgICBkaXNwbGF5TmFtZTogJ1N0YWdlTGV2ZWwxL1N0YWdlTGV2ZWwyJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBhc3NlbWJseUxldmVsMiA9IGFzc2VtYmx5TGV2ZWwxLmdldE5lc3RlZEFzc2VtYmx5KCdhc3NlbWJseS1TdGFnZUxldmVsMS1TdGFnZUxldmVsMicpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzc2VtYmx5TGV2ZWwyLm1hbmlmZXN0LmFydGlmYWN0cywge1xuICAgICAgJ2Fzc2VtYmx5LVN0YWdlTGV2ZWwxLVN0YWdlTGV2ZWwyLVN0YWdlTGV2ZWwzJzoge1xuICAgICAgICB0eXBlOiAnY2RrOmNsb3VkLWFzc2VtYmx5JyxcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGRpcmVjdG9yeU5hbWU6ICdhc3NlbWJseS1TdGFnZUxldmVsMS1TdGFnZUxldmVsMi1TdGFnZUxldmVsMycsXG4gICAgICAgICAgZGlzcGxheU5hbWU6ICdTdGFnZUxldmVsMS9TdGFnZUxldmVsMi9TdGFnZUxldmVsMycsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWdlIG5hbWUgdmFsaWRhdGlvbicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcblxuICAgIG5ldyBTdGFnZShhcHAsICdhYmNkJyk7XG4gICAgbmV3IFN0YWdlKGFwcCwgJ2FiY2QxMjMnKTtcbiAgICBuZXcgU3RhZ2UoYXBwLCAnYWJjZDEyMy01ODhkZmpqaycpO1xuICAgIG5ldyBTdGFnZShhcHAsICdhYmNkMTIzLTU4OGRmamprLnNzcycpO1xuICAgIG5ldyBTdGFnZShhcHAsICdhYmNkMTIzLTU4OGRmamprLnNzc19hanNpZCcpO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFN0YWdlKGFwcCwgJ2FiY2QxMjMtNTg4ZGZqamsuc3NzX2Fqc2lkICcpLCAvaW52YWxpZCBzdGFnZSBuYW1lIFwiYWJjZDEyMy01ODhkZmpqay5zc3NfYWpzaWQgXCIvKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgU3RhZ2UoYXBwLCAnYWJjZDEyMy01ODhkZmpqay5zc3NfYWpzaWQvZGZvJyksIC9pbnZhbGlkIHN0YWdlIG5hbWUgXCJhYmNkMTIzLTU4OGRmamprLnNzc19hanNpZFxcL2Rmb1wiLyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFN0YWdlKGFwcCwgJyYnKSwgL2ludmFsaWQgc3RhZ2UgbmFtZSBcIiZcIi8pO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBTdGFnZShhcHAsICc0NWhlbGxvJyksIC9pbnZhbGlkIHN0YWdlIG5hbWUgXCI0NWhlbGxvXCIvKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgU3RhZ2UoYXBwLCAnZicpLCAvaW52YWxpZCBzdGFnZSBuYW1lIFwiZlwiLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnb3V0ZGlyIGNhbm5vdCBiZSBzcGVjaWZpZWQgZm9yIG5lc3RlZCBzdGFnZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBTdGFnZShhcHAsICdteXN0YWdlJywgeyBvdXRkaXI6ICcvdG1wL2Zvby9iYXInIH0pLCAvXCJvdXRkaXJcIiBjYW5ub3QgYmUgc3BlY2lmaWVkIGZvciBuZXN0ZWQgc3RhZ2VzLyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuXG5jbGFzcyBUb3VjaGluZ0FzcGVjdCBpbXBsZW1lbnRzIElBc3BlY3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgdmlzaXRzID0gbmV3IEFycmF5PElDb25zdHJ1Y3Q+KCk7XG4gIHB1YmxpYyB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgdGhpcy52aXNpdHMucHVzaChub2RlKTtcbiAgfVxufVxuXG5jbGFzcyBCb2d1c1N0YWNrIGV4dGVuZHMgU3RhY2sge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIG5ldyBDZm5SZXNvdXJjZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQ0RLOjpUZXN0OjpSZXNvdXJjZScsXG4gICAgfSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gYWNjdFJlZ2lvbihzOiBTdGFjaykge1xuICByZXR1cm4gW3MuYWNjb3VudCwgcy5yZWdpb25dO1xufSJdfQ==