"use strict";
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const sinon = require("sinon");
const lib_1 = require("../lib");
const STUB_INPUT_FILE = '/tmp/docker-stub.input';
const STUB_INPUT_CONCAT_FILE = '/tmp/docker-stub.input.concat';
var DockerStubCommand;
(function (DockerStubCommand) {
    DockerStubCommand["SUCCESS"] = "DOCKER_STUB_SUCCESS";
    DockerStubCommand["FAIL"] = "DOCKER_STUB_FAIL";
    DockerStubCommand["SUCCESS_NO_OUTPUT"] = "DOCKER_STUB_SUCCESS_NO_OUTPUT";
})(DockerStubCommand || (DockerStubCommand = {}));
const userInfo = os.userInfo();
const USER_ARG = `-u ${userInfo.uid}:${userInfo.gid}`;
// this is a way to provide a custom "docker" command for staging.
process.env.CDK_DOCKER = `${__dirname}/docker-stub.sh`;
// Reads a docker stub and cleans the volume paths out of the stub.
function readAndCleanDockerStubInput(file) {
    return fs
        .readFileSync(file, 'utf-8')
        .trim()
        .replace(/-v ([^:]+):\/asset-input/g, '-v /input:/asset-input')
        .replace(/-v ([^:]+):\/asset-output/g, '-v /output:/asset-output');
}
// Last docker input since last teardown
function readDockerStubInput() {
    return readAndCleanDockerStubInput(STUB_INPUT_FILE);
}
// Concatenated docker inputs since last teardown
function readDockerStubInputConcat() {
    return readAndCleanDockerStubInput(STUB_INPUT_CONCAT_FILE);
}
module.exports = {
    'tearDown'(cb) {
        if (fs.existsSync(STUB_INPUT_FILE)) {
            fs.unlinkSync(STUB_INPUT_FILE);
        }
        if (fs.existsSync(STUB_INPUT_CONCAT_FILE)) {
            fs.unlinkSync(STUB_INPUT_CONCAT_FILE);
        }
        cb();
        sinon.restore();
    },
    'base case'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), 'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.done();
    },
    'staging can be disabled through context'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT, true);
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), sourcePath);
        test.done();
    },
    'files are copied to the output directory during synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const file = path.join(__dirname, 'fs', 'fixtures.tar.gz');
        // WHEN
        new lib_1.AssetStaging(stack, 's1', { sourcePath: directory });
        new lib_1.AssetStaging(stack, 'file', { sourcePath: file });
        // THEN
        const assembly = app.synth();
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'asset.af10ac04b3b607b0f8659c8f0cee8c343025ee75baf0b146f10f0e5311d2c46b.gz',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'allow specifying extra data to include in the source hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const withoutExtra = new lib_1.AssetStaging(stack, 'withoutExtra', { sourcePath: directory });
        const withExtra = new lib_1.AssetStaging(stack, 'withExtra', { sourcePath: directory, extraHash: 'boom' });
        // THEN
        test.notEqual(withoutExtra.sourceHash, withExtra.sourceHash);
        test.deepEqual(withoutExtra.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(withExtra.sourceHash, 'c95c915a5722bb9019e2c725d11868e5a619b55f36172f76bcbcaa8bb2d10c5f');
        test.done();
    },
    'with bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const processStdErrWriteSpy = sinon.spy(process.stderr, 'write');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.b1e32e86b3523f2fa512eb99180ee2975a50a4439e63e8badd153f2a68d61aa4',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        // shows a message before bundling
        test.ok(processStdErrWriteSpy.calledWith('Bundling asset stack/Asset...\n'));
        test.done();
    },
    'bundler succeeds when staging is disabled'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        stack.node.setContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT, true);
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.b1e32e86b3523f2fa512eb99180ee2975a50a4439e63e8badd153f2a68d61aa4',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'bundler reuses its output when it can'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        new lib_1.AssetStaging(stack, 'AssetDuplicate', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        // We're testing that docker was run exactly once even though there are two bundling assets.
        test.deepEqual(readDockerStubInputConcat(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.b1e32e86b3523f2fa512eb99180ee2975a50a4439e63e8badd153f2a68d61aa4',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'bundler considers its options when reusing bundle output'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        new lib_1.AssetStaging(stack, 'AssetWithDifferentBundlingOptions', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
                environment: {
                    UNIQUE_ENV_VAR: 'SOMEVALUE',
                },
            },
        });
        // THEN
        const assembly = app.synth();
        // We're testing that docker was run twice - once for each set of bundler options
        // operating on the same source asset.
        test.deepEqual(readDockerStubInputConcat(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS\n` +
            `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated --env UNIQUE_ENV_VAR=SOMEVALUE -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.b1e32e86b3523f2fa512eb99180ee2975a50a4439e63e8badd153f2a68d61aa4',
            'asset.e80bb8f931b87e84975de193f5a7ecddd7558d3caf3d35d3a536d9ae6539234f',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'bundler outputs to intermediate dir and renames to asset'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const mkdtempSyncSpy = sinon.spy(fs, 'mkdtempSync');
        const chmodSyncSpy = sinon.spy(fs, 'chmodSync');
        const renameSyncSpy = sinon.spy(fs, 'renameSync');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.BUNDLE,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        test.ok(mkdtempSyncSpy.calledWith(sinon.match(path.join(assembly.directory, 'bundling-temp-'))));
        test.ok(chmodSyncSpy.calledWith(sinon.match(path.join(assembly.directory, 'bundling-temp-')), 0o777));
        test.ok(renameSyncSpy.calledWith(sinon.match(path.join(assembly.directory, 'bundling-temp-')), sinon.match(path.join(assembly.directory, 'asset.'))));
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.33cbf2cae5432438e0f046bc45ba8c3cef7b6afcf47b59d1c183775c1918fb1f',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'bundling failure preserves the bundleDir for diagnosability'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.FAIL],
            },
        }), /Failed.*bundl.*asset.*-error/);
        // THEN
        const assembly = app.synth();
        const dir = fs.readdirSync(assembly.directory);
        test.ok(dir.some(entry => entry.match(/asset.*-error/)));
        test.done();
    },
    'bundler re-uses assets from previous synths'(test) {
        // GIVEN
        const TEST_OUTDIR = path.join(__dirname, 'cdk.out');
        if (fs.existsSync(TEST_OUTDIR)) {
            fs.removeSync(TEST_OUTDIR);
        }
        const app = new lib_1.App({ outdir: TEST_OUTDIR });
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // GIVEN
        const app2 = new lib_1.App({ outdir: TEST_OUTDIR });
        const stack2 = new lib_1.Stack(app2, 'stack');
        // WHEN
        new lib_1.AssetStaging(stack2, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const appAssembly = app.synth();
        const app2Assembly = app2.synth();
        test.deepEqual(readDockerStubInputConcat(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.equals(appAssembly.directory, app2Assembly.directory);
        test.deepEqual(fs.readdirSync(appAssembly.directory), [
            'asset.b1e32e86b3523f2fa512eb99180ee2975a50a4439e63e8badd153f2a68d61aa4',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'bundling throws when /asset-ouput is empty'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS_NO_OUTPUT],
            },
        }), /Bundling did not produce any output/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS_NO_OUTPUT`);
        test.done();
    },
    'bundling with BUNDLE asset hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHashType: lib_1.AssetHashType.BUNDLE,
        });
        // THEN
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.equal(asset.assetHash, '33cbf2cae5432438e0f046bc45ba8c3cef7b6afcf47b59d1c183775c1918fb1f');
        test.done();
    },
    'custom hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHash: 'my-custom-hash',
        });
        // THEN
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.equal(asset.assetHash, 'b9c77053f5b83bbe5ba343bc18e92db939a49017010813225fea91fa892c4823'); // hash of 'my-custom-hash'
        test.done();
    },
    'throws with assetHash and not CUSTOM hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHash: 'my-custom-hash',
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot specify `bundle` for `assetHashType`/);
        test.done();
    },
    'throws with BUNDLE hash type and no bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.done();
    },
    'throws with CUSTOM and no hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.CUSTOM,
        }), /`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false); // "docker" not executed
        test.done();
    },
    'throws when bundling fails'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('this-is-an-invalid-docker-image'),
                command: [DockerStubCommand.FAIL],
            },
        }), /Failed to bundle asset stack\/Asset/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input this-is-an-invalid-docker-image DOCKER_STUB_FAIL`);
        test.done();
    },
    'with local bundling'(test) {
        var _a;
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        let dir;
        let opts;
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
                local: {
                    tryBundle(outputDir, options) {
                        dir = outputDir;
                        opts = options;
                        fs.writeFileSync(path.join(outputDir, 'hello.txt'), 'hello'); // output cannot be empty
                        return true;
                    },
                },
            },
        });
        // THEN
        test.ok(dir && /asset.[0-9a-f]{16,}/.test(dir));
        test.equals((_a = opts === null || opts === void 0 ? void 0 : opts.command) === null || _a === void 0 ? void 0 : _a[0], DockerStubCommand.SUCCESS);
        test.throws(() => readDockerStubInput());
        if (dir) {
            fs.removeSync(path.join(dir, 'hello.txt'));
        }
        test.done();
    },
    'with local bundling returning false'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
                local: {
                    tryBundle(_bundleDir) {
                        return false;
                    },
                },
            },
        });
        // THEN
        test.ok(readDockerStubInput());
        test.done();
    },
    'bundling looks at bundling stacks in context'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        stack.node.setContext(cxapi.BUNDLING_STACKS, ['OtherStack']);
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.BUNDLE,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        test.throws(() => readDockerStubInput()); // Bundling did not run
        test.equal(asset.assetHash, '3d96e735e26b857743a7c44523c9160c285c2d3ccf273d80fa38a1e674c32cb3'); // hash of MyStack/Asset
        test.equal(asset.sourcePath, directory);
        test.equal(stack.resolve(asset.stagedPath), directory);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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