"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This module is intended to distributed as part of automatically generated code by the
peakrdl-python tool. It provides a set of classes used by the autogenerated code to represent
memories
"""
from array import array as Array
from typing import Union, TYPE_CHECKING, Optional
from abc import ABC
from collections.abc import Iterator, Iterable
import sys

from .base import NodeArray, IterationClassification
from .sections import AsyncAddressMap
from .memory import BaseMemory

from .callbacks import AsyncCallbackSet, AsyncCallbackSetLegacy

# same bit of code exists in base so flags as duplicate
# pylint: disable=duplicate-code
if sys.version_info >= (3, 10):
    # type guarding was introduced in python 3.10
    from typing import TypeGuard
else:
    from typing_extensions import TypeGuard
# pylint: enable=duplicate-code


if TYPE_CHECKING:
    from .async_register_and_field import AsyncReg, AsyncRegArray
    from .async_register_and_field import ReadableAsyncRegister, WritableAsyncRegister
    from .async_register_and_field import ReadableAsyncRegisterArray, WriteableAsyncRegisterArray
# pylint: disable=duplicate-code


class AsyncMemory(BaseMemory, Iterable[Union['AsyncReg', 'AsyncRegArray']], ABC):
    """
    base class of non_async memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    # pylint: disable=too-many-arguments
    def __init__(self, *,
                 address: int,
                 width: int,
                 accesswidth: int,
                 entries: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AsyncAddressMap, 'AsyncMemoryArray']):
        """
        Initialise the class

        Args:
            address: address of the register
            width: width of the register in bits
            logger_handle: name to be used logging messages associate with thisobject
        """
        if not isinstance(parent, (AsyncAddressMap,
                                   MemoryAsyncWriteOnlyArray,
                                   MemoryAsyncReadOnlyArray,
                                   MemoryAsyncReadWriteArray,
                                   MemoryAsyncWriteOnlyLegacyArray,
                                   MemoryAsyncReadOnlyLegacyArray,
                                   MemoryAsyncReadWriteLegacyArray)):
            raise TypeError(f'parent should be either AddressMap or Memory Array got '
                            f'{type(parent)}')

        if not isinstance(parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            raise TypeError(f'callback set type is wrong, got {type(parent._callbacks)}')

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         width=width,
                         accesswidth=accesswidth,
                         entries=entries,
                         parent=parent)

    @property
    def _callbacks(self) -> Union[AsyncCallbackSet, AsyncCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.parent is None:
            raise RuntimeError('Parent must be set')

        if isinstance(self.parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')

    def get_children(self, unroll: bool = False) -> Iterator[Union['AsyncReg', 'AsyncRegArray']]:
        """
        generator that produces all the registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        if unroll:
            for child in iter(self):
                if isinstance(child, NodeArray):
                    yield from child
                else:
                    yield child
        else:
            yield from iter(self)

    def get_registers(self, unroll: bool = False) -> \
            Iterator[Union['AsyncReg', 'AsyncRegArray']]:
        """
        generator that produces all the registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        yield from self.get_children(unroll=unroll)


class _MemoryAsyncReadOnly(AsyncMemory, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = []

    async def _read(self, start_entry: int, number_entries: int) -> list[int]:
        """
        Read from the memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            number_entries: number of entries to read

        Returns: data read from memory

        """

        if not isinstance(start_entry, int):
            raise TypeError(f'start_entry should be an int got {type(start_entry)}')

        if not isinstance(number_entries, int):
            raise TypeError(f'number_entries should be an int got {type(number_entries)}')

        if start_entry not in range(0, self.entries):
            raise ValueError(f'entry must be in range 0 to {self.entries - 1:d} '
                             f'but got {start_entry:d}')

        if number_entries not in range(0, self.entries - start_entry + 1):
            raise ValueError(f'number_entries must be in range 0 to'
                             f' {self.entries - start_entry:d} but got {number_entries:d}')

        read_block_callback = self._callbacks.read_block_callback
        read_callback = self._callbacks.read_callback

        if read_block_callback is not None:
            addr = self.address_lookup(entry=start_entry)
            data_read = \
                await read_block_callback(addr=addr,
                                          width=self.width,
                                          accesswidth=self.width,
                                          length=number_entries)

            if isinstance(self._callbacks, AsyncCallbackSet):
                if not isinstance(data_read, list):
                    raise TypeError('The read block callback is expected to return an List')
                return data_read

            if isinstance(self._callbacks, AsyncCallbackSetLegacy):
                if not isinstance(data_read, Array):
                    raise TypeError('The read block callback is expected to return an array')
                return data_read.tolist()

            raise RuntimeError(f'There is no usable callback block callback:{read_block_callback}')

        if read_callback is not None:
            # there is not read_block_callback defined so we must used individual read
            data_read = [0 for _ in range(number_entries)]

            for entry in range(number_entries):
                entry_address = self.address_lookup(entry=start_entry+entry)

                data_entry = await read_callback(addr=entry_address,
                                                 width=self.width,
                                                 accesswidth=self.width)

                data_read[entry] = data_entry

            return data_read

        raise RuntimeError(f'There is no usable callback, '
                           f'block callback:{read_block_callback}, '
                           f'normal callback:{read_callback}')

    async def _read_legacy(self, start_entry: int, number_entries: int) -> Array:
        """
        Read from the memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            number_entries: number of entries to read

        Returns: data read from memory

        """

        if not isinstance(start_entry, int):
            raise TypeError(f'start_entry should be an int got {type(start_entry)}')

        if not isinstance(number_entries, int):
            raise TypeError(f'number_entries should be an int got {type(number_entries)}')

        if start_entry not in range(0, self.entries):
            raise ValueError(f'entry must be in range 0 to {self.entries - 1:d} '
                             f'but got {start_entry:d}')

        if number_entries not in range(0, self.entries - start_entry + 1):
            raise ValueError(f'number_entries must be in range 0 to'
                             f' {self.entries - start_entry:d} but got {number_entries:d}')

        read_block_callback = self._callbacks.read_block_callback
        read_callback = self._callbacks.read_callback

        if read_block_callback is not None:
            addr = self.address_lookup(entry=start_entry)
            data_read = \
                await read_block_callback(addr=addr,
                                          width=self.width,
                                          accesswidth=self.width,
                                          length=number_entries)

            if isinstance(self._callbacks, AsyncCallbackSet):
                if not isinstance(data_read, list):
                    raise TypeError('The read block callback is expected to return an List')
                return Array(self.array_typecode, data_read)

            if isinstance(self._callbacks, AsyncCallbackSetLegacy):
                if not isinstance(data_read, Array):
                    raise TypeError('The read block callback is expected to return an array')
                return data_read

            raise RuntimeError(f'There is no usable callback block callback:{read_block_callback}')

        if read_callback is not None:
            # there is not read_block_callback defined so we must used individual read
            data_read_block = Array(self.array_typecode, [0 for _ in range(number_entries)])

            for entry in range(number_entries):
                entry_address = self.address_lookup(entry=start_entry + entry)
                data_entry = await read_callback(addr=entry_address,
                                                 width=self.width,
                                                 accesswidth=self.width)

                data_read_block[entry] = data_entry

            return data_read_block

        raise RuntimeError(f'There is no usable callback, '
                           f'block callback:{read_block_callback}, '
                           f'normal callback:{read_callback}')

    def get_readable_registers(self, unroll: bool = False) -> \
            Iterator[Union['ReadableAsyncRegister', 'ReadableAsyncRegisterArray']]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_readable(item: Union['AsyncReg', 'AsyncRegArray']) -> \
                TypeGuard[Union['ReadableAsyncRegister', 'ReadableAsyncRegisterArray']]:
            # pylint: disable-next=protected-access
            return item._is_readable

        return filter(is_readable, self.get_registers(unroll=unroll))


class MemoryAsyncReadOnly(_MemoryAsyncReadOnly, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    async def read(self, start_entry: int, number_entries: int) -> list[int]:
        """
        Read from the memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            number_entries: number of entries to read

        Returns: data read from memory

        """
        return await self._read(start_entry=start_entry, number_entries=number_entries)


class MemoryAsyncReadOnlyLegacy(_MemoryAsyncReadOnly, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    async def read(self, start_entry: int, number_entries: int) -> Array:
        """
        Read from the memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            number_entries: number of entries to read

        Returns: data read from memory

        """
        return await self._read_legacy(start_entry=start_entry, number_entries=number_entries)


class _MemoryAsyncWriteOnly(AsyncMemory, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = []

    async def _write(self, start_entry: int, data: Union[Array, list[int]]) -> None:
        """
        Asynchronously write data to memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            data: data to write

        Returns: None

        """
        # pylint:disable=too-many-branches
        if not isinstance(start_entry, int):
            raise TypeError(f'start_entry should be an int got {type(start_entry)}')

        if start_entry not in range(0, self.entries):
            raise ValueError(f'entry must be in range 0 to {self.entries - 1:d} '
                             f'but got {start_entry:d}')

        if not isinstance(data, (list, Array)):
            raise TypeError(f'data should be an array.array got {type(data)}')

        if len(data) not in range(0, self.entries - start_entry + 1):
            raise ValueError(f'data length must be in range 0 to {self.entries - start_entry:d} '
                             f'but got {len(data):d}')

        if self._callbacks.write_block_callback is not None:

            addr = self.address_lookup(entry=start_entry)
            if isinstance(self._callbacks, AsyncCallbackSet):
                if isinstance(data, Array):
                    await self._callbacks.write_block_callback(addr=addr,
                                                               width=self.width,
                                                               accesswidth=self.width,
                                                               data=data.tolist())
                else:
                    await self._callbacks.write_block_callback(addr=addr,
                                                               width=self.width,
                                                               accesswidth=self.width,
                                                               data=data)
            if isinstance(self._callbacks, AsyncCallbackSetLegacy):
                if isinstance(data, list):
                    # need to convert the data to an array before calling
                    await self._callbacks.write_block_callback(
                        addr=addr,
                        width=self.width,
                        accesswidth=self.width,
                        data=Array(self.array_typecode, data))
                else:
                    await self._callbacks.write_block_callback(addr=addr,
                                                               width=self.width,
                                                               accesswidth=self.width,
                                                               data=data)

        elif self._callbacks.write_callback is not None:
            # there is not write_block_callback defined so we must used individual write
            for entry_index, entry_data in enumerate(data):
                entry_address = self.address_lookup(entry=start_entry+entry_index)
                await self._callbacks.write_callback(addr=entry_address,
                                                     width=self.width,
                                                     accesswidth=self.width,
                                                     data=entry_data)

        else:
            raise RuntimeError('No suitable callback')

    def get_writable_registers(self, unroll: bool = False) -> \
            Iterator[Union['WritableAsyncRegister', 'WriteableAsyncRegisterArray']]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_writable(item: Union['AsyncReg', 'AsyncRegArray']) -> \
                TypeGuard[Union['WritableAsyncRegister', 'WriteableAsyncRegisterArray']]:
            # pylint: disable-next=protected-access
            return item._is_writeable

        return filter(is_writable, self.get_registers(unroll=unroll))


class MemoryAsyncWriteOnly(_MemoryAsyncWriteOnly, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    async def write(self, start_entry: int, data: list[int]) -> None:
        """
        Asynchronously write data to memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            data: data to write

        Returns: None

        """
        if not isinstance(data, list):
            raise TypeError(f'data should be an List got {type(data)}')
        return await self._write(start_entry=start_entry, data=data)


class MemoryAsyncWriteOnlyLegacy(_MemoryAsyncWriteOnly, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    async def write(self, start_entry: int, data: Array) -> None:
        """
        Asynchronously write data to memory

        Args:
            start_entry: index in the memory to start from, this is not the address
            data: data to write

        Returns: None

        """
        if not isinstance(data, Array):
            raise TypeError(f'data should be an Array {type(data)}')
        return await self._write(start_entry=start_entry, data=data)


class MemoryAsyncReadWrite(MemoryAsyncReadOnly, MemoryAsyncWriteOnly, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = []


class MemoryAsyncReadWriteLegacy(MemoryAsyncReadOnlyLegacy, MemoryAsyncWriteOnlyLegacy, ABC):
    """
    base class of memory wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = []


class MemoryAsyncReadOnlyArray(NodeArray[MemoryAsyncReadOnly], ABC):
    """
    base class for a array of asynchronous read only memories
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.MEMORY

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[MemoryAsyncReadOnly, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be either AsyncAddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions, elements=elements)

class MemoryAsyncReadOnlyLegacyArray(NodeArray[MemoryAsyncReadOnlyLegacy], ABC):
    """
    base class for a array of asynchronous read only memories
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.MEMORY

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[MemoryAsyncReadOnlyLegacy, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be either AsyncAddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions, elements=elements)


class MemoryAsyncWriteOnlyArray(NodeArray[MemoryAsyncWriteOnly], ABC):
    """
    base class for a array of asynchronous write only memories
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.MEMORY

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[MemoryAsyncWriteOnly, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be either AsyncAddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions,
                         elements=elements)

class MemoryAsyncWriteOnlyLegacyArray(NodeArray[MemoryAsyncWriteOnlyLegacy], ABC):
    """
    base class for a array of asynchronous write only memories
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.MEMORY

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[MemoryAsyncWriteOnlyLegacy, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be either AsyncAddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions,
                         elements=elements)


class MemoryAsyncReadWriteArray(NodeArray[MemoryAsyncReadWrite], ABC):
    """
    base class for a array of asynchronous read and write memories
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.MEMORY

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[MemoryAsyncReadWrite, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be either AsyncAddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions,
                         elements=elements)

class MemoryAsyncReadWriteLegacyArray(NodeArray[MemoryAsyncReadWriteLegacy], ABC):
    """
    base class for a array of asynchronous read and write memories
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.MEMORY

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[MemoryAsyncReadWriteLegacy, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be either AsyncAddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions,
                         elements=elements)


ReadableAsyncMemory = Union[MemoryAsyncReadOnly, MemoryAsyncReadWrite]
WritableAsyncMemory = Union[MemoryAsyncWriteOnly, MemoryAsyncReadWrite]
ReadableAsyncMemoryLegacy = Union[MemoryAsyncReadOnlyLegacy, MemoryAsyncReadWriteLegacy]
WritableAsyncMemoryLegacy = Union[MemoryAsyncWriteOnlyLegacy, MemoryAsyncReadWriteLegacy]
AsyncMemoryArrayNonLegacy = Union[MemoryAsyncReadOnlyArray, MemoryAsyncWriteOnlyArray,
                         MemoryAsyncReadWriteArray]
AsyncMemoryArrayLegacy = Union[MemoryAsyncReadOnlyLegacyArray, MemoryAsyncWriteOnlyLegacyArray,
                         MemoryAsyncReadWriteLegacyArray]
AsyncMemoryArray = Union[AsyncMemoryArrayNonLegacy,AsyncMemoryArrayLegacy]
