"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This package is intended to distributed as part of automatically generated code by the PeakRDL
Python tool. It provide the base class common to both the async and non-async versions
"""
import unittest
from abc import ABC
from typing import Union, Optional

from ..lib import FieldReadWrite, FieldReadOnly, FieldWriteOnly
from ..lib import FieldEnumReadWrite, FieldEnumReadOnly, FieldEnumWriteOnly
from ..lib import FieldAsyncReadOnly, FieldAsyncWriteOnly, FieldAsyncReadWrite
from ..lib import FieldEnumAsyncReadOnly, FieldEnumAsyncWriteOnly, FieldEnumAsyncReadWrite
from ..lib.base_register import BaseReg
from .utilities import get_field_bitmask_int, get_field_inv_bitmask

class CommonTestBase(unittest.TestCase, ABC):
    """
    Base Test class for the autogenerated register test to be used for for the async and
    non-async cases
    """

    # pylint:disable-next=too-many-arguments
    def _single_field_property_test(self, *,
                                    fut: Union[FieldReadWrite,
                                               FieldReadOnly,
                                               FieldWriteOnly,
                                               FieldEnumReadWrite,
                                               FieldEnumReadOnly,
                                               FieldEnumWriteOnly,
                                               FieldAsyncReadOnly,
                                               FieldAsyncWriteOnly,
                                               FieldAsyncReadWrite,
                                               FieldEnumAsyncReadOnly,
                                               FieldEnumAsyncWriteOnly,
                                               FieldEnumAsyncReadWrite],
                                    lsb: int,
                                    msb: int,
                                    low: int,
                                    high: int,
                                    is_volatile: bool,
                                    default: Optional[int]) -> None:
        self.assertEqual(fut.lsb, lsb)
        self.assertEqual(fut.msb, msb)
        self.assertEqual(fut.low, low)
        self.assertEqual(fut.high, high)
        self.assertEqual(fut.bitmask, get_field_bitmask_int(fut))
        self.assertEqual(fut.inverse_bitmask, get_field_inv_bitmask(fut))
        width = (fut.high - fut.low) + 1
        self.assertEqual(fut.width, width)
        self.assertEqual(fut.max_value, (2**width) - 1)
        self.assertEqual(fut.is_volatile, is_volatile)

        if default is None:
            self.assertIsNone(fut.default)
        else:
            if isinstance(fut, (FieldEnumReadWrite,
                                FieldEnumReadOnly,
                                FieldEnumWriteOnly,
                                FieldEnumAsyncReadOnly,
                                FieldEnumAsyncWriteOnly,
                                FieldEnumAsyncReadWrite)):
                # pylint does not realise this is a class being returned rather than an object, so
                # is unhappy with the name
                # pylint:disable-next=invalid-name
                EnumCls = fut.enum_cls
                if default in [item.value for item in fut.enum_cls]:
                    self.assertEqual(fut.default, EnumCls(default))
                else:
                    # this is a special case if the default value for the field does not map
                    # to a legal value of the encoding
                    self.assertIsNone(fut.default)
            else:
                self.assertEqual(fut.default, default)

    def _single_register_property_test(self, *,
                                       rut: BaseReg,
                                       address: int,
                                       width: int,
                                       accesswidth: Optional[int]) -> None:
        self.assertEqual(rut.address, address)
        self.assertEqual(rut.width, width)
        if accesswidth is not None:
            self.assertEqual(rut.accesswidth, accesswidth)
        else:
            self.assertEqual(rut.accesswidth, width)
