import os
from datetime import datetime
from pathlib import Path

import yaml


class CacheProvider:
    def __init__(self):
        self._cache_file = ".platform-helper-config-cache.yml"

    def read_supported_versions_from_cache(self, resource_name):

        platform_helper_config = self.__read_file_as_yaml(self._cache_file)

        return platform_helper_config.get(resource_name).get("versions")

    def update_cache(self, resource_name, supported_versions):

        platform_helper_config = {}

        if self.__cache_exists():
            platform_helper_config = self.__read_file_as_yaml(self._cache_file)

        cache_dict = {
            resource_name: {
                "versions": supported_versions,
                "date-retrieved": datetime.now().strftime("%d-%m-%y %H:%M:%S"),
            }
        }

        platform_helper_config.update(cache_dict)

        self.__write_cache(platform_helper_config)

    def cache_refresh_required(self, resource_name) -> bool:
        """
        Checks if the platform-helper should reach out to AWS to 'refresh' its
        cached values.

        An API call is needed if any of the following conditions are met:
            1. No cache file (.platform-helper-config.yml) exists.
            2. The resource name (e.g. redis, opensearch) does not exist within the cache file.
            3. The date-retrieved value of the cached data is > than a time interval. In this case 1 day.
        """

        if not self.__cache_exists():
            return True

        platform_helper_config = self.__read_file_as_yaml(self._cache_file)

        if platform_helper_config.get(resource_name):
            return self.__check_if_cached_datetime_is_greater_than_interval(
                platform_helper_config[resource_name].get("date-retrieved"), 1
            )

        return True

    @staticmethod
    def __check_if_cached_datetime_is_greater_than_interval(date_retrieved, interval_in_days):

        current_datetime = datetime.now()
        cached_datetime = datetime.strptime(date_retrieved, "%d-%m-%y %H:%M:%S")
        delta = current_datetime - cached_datetime

        return delta.days > interval_in_days

    # TODO - same applies here as below
    @staticmethod
    def __read_file_as_yaml(file_name):

        return yaml.safe_load(Path(file_name).read_text())

    # TODO - temp fix to the unit test coverage issue, plan is to seperate out any yaml interaction methods into a seperate 'yaml' provider
    # should be done under a different sub-task which will need to loop back to this provider as part of that work to use the yaml provider instead
    def __write_cache(self, contents):

        with open(self._cache_file, "w") as file:
            file.write("# [!] This file is autogenerated via the platform-helper. Do not edit.\n")
            yaml.dump(contents, file)

    def __cache_exists(self):
        return os.path.exists(self._cache_file)
