"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const construct_1 = require("./construct");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            }
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            }
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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