# pydbms/pydbms/config.py
#NOTE: If config.json file generated by the program is deleted or currupted, it will safely restore to defaults.

from .pydbms_path import pydbms_path
from .dependencies import json, os, copy

DEFAULT_SESSION_CONFIG = {
    "expand-query-result": False
}

def expand_query_session_config_mapping():
    return "fold" if SESSION_CONFIG["expand-query-result"] else "ellipsis"
    
SESSION_CONFIG = copy.deepcopy(DEFAULT_SESSION_CONFIG)

DEFAULT_CONFIG = {
    "config_version": 1,

    "ui": {
        "show_banner": True,
        "max_rows": None,
    },

    "mysql": {
        "host": "localhost",
        "port": 3306,
        "user": "root",
        "database": None,
    }
}

def create_config() -> None:
    path = pydbms_path("config.json")

    if not os.path.exists(path):
        with open(path, "w") as f:
            json.dump(DEFAULT_CONFIG, f, indent=4)

def validate_config_types() -> dict:
    cfg = load_config()
    changed=False

    # validate ui.show_banner
    if not isinstance(cfg.get("ui", {}).get("show_banner"), bool):
        cfg.setdefault("ui", {})["show_banner"] = DEFAULT_CONFIG["ui"]["show_banner"]
        changed=True

    # validate mysql.port
    if not isinstance(cfg.get("mysql", {}).get("port"), int):
        cfg.setdefault("mysql", {})["port"] = DEFAULT_CONFIG["mysql"]["port"]
        changed=True

    if changed:
        save_config(cfg)
        
    return cfg

def load_config() -> dict:
    path = pydbms_path("config.json")

    try:
        with open(path, "r") as f:
            return json.load(f)
    except (FileNotFoundError, json.JSONDecodeError):
        create_config()
        return copy.deepcopy(DEFAULT_CONFIG)

def save_config(config: dict) -> None:
    path = pydbms_path("config.json")

    with open(path, "w") as f:
        json.dump(config, f, indent=4)

def parse_query_config(path: str) -> tuple[str, str] | None:
    if "." not in path:
        return None
    section, key = path.split(".", 1)
    return section, key

def coerce_value_config(value: str) -> bool | int | float | None | object:
    v = value.lower()

    if v in {"true", "yes", "on"}:
        return True
    if v in {"false", "no", "off"}:
        return False
    if v.isdigit():
        return int(v)

    try:
        return float(v)
    except ValueError:
        pass

    if v == "null" or v == "none":
        return None

    return value

def get_default_value_config(section: str, key: str) -> bool | int | float | None:
    try:
        return DEFAULT_CONFIG[section][key]
    except KeyError:
        return None