"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const bundlers_1 = require("./bundlers");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const relativeEntryPath = path.relative(projectRoot, path.resolve(options.entry));
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_b = options.externalModules) !== null && _b !== void 0 ? _b : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_c = options.nodeModules) !== null && _c !== void 0 ? _c : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        let installer = bundlers_1.Installer.NPM;
        let lockFile;
        if (dependencies) {
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.YARN))) {
                installer = bundlers_1.Installer.YARN;
                lockFile = bundlers_1.LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.NPM))) {
                lockFile = bundlers_1.LockFile.NPM;
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_d = options.sourceMaps) !== null && _d !== void 0 ? _d : false,
                    minify: (_e = options.minify) !== null && _e !== void 0 ? _e : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Local
        let localBundler;
        if (!options.forceDockerBundling) {
            localBundler = new bundlers_1.LocalBundler({
                projectRoot,
                relativeEntryPath,
                cacheDir: options.cacheDir,
                environment: options.parcelEnvironment,
                dependencies,
                installer,
                lockFile,
            });
        }
        // Docker
        const dockerBundler = new bundlers_1.DockerBundler({
            runtime: options.runtime,
            relativeEntryPath,
            cacheDir: options.cacheDir,
            environment: options.parcelEnvironment,
            buildImage: !bundlers_1.LocalBundler.runsLocally || options.forceDockerBundling,
            buildArgs: options.buildArgs,
            parcelVersion: options.parcelVersion,
            dependencies,
            installer,
            lockFile,
        });
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                local: localBundler,
                ...dockerBundler.bundlingOptions,
            },
        });
    }
}
exports.Bundling = Bundling;
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDhDQUE4QztBQUM5QyxxQ0FBcUM7QUFDckMseUNBQThFO0FBQzlFLGlFQUE0RDtBQUM1RCxpQ0FBZ0M7QUFzR2hDOztHQUVHO0FBQ0gsTUFBYSxRQUFRO0lBQ25COztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFzQjs7UUFDekMsb0JBQW9CO1FBQ3BCLE1BQU0sV0FBVyxTQUFHLE9BQU8sQ0FBQyxXQUFXLG1DQUFJLGFBQU0sQ0FBQyxPQUFPLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1NBQ3BGO1FBQ0QsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRWxGLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSx5Q0FBa0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRS9FLHVDQUF1QztRQUN2QyxJQUFJLGtCQUEwRCxDQUFDO1FBQy9ELElBQUksWUFBbUQsQ0FBQztRQUN4RCxNQUFNLGVBQWUsU0FBRyxPQUFPLENBQUMsZUFBZSxtQ0FBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQy9ELElBQUksZUFBZSxJQUFJLE9BQU8sQ0FBQyxXQUFXLEVBQUU7WUFDMUMsTUFBTSxPQUFPLEdBQUcsQ0FBQyxHQUFHLGVBQWUsRUFBRSxTQUFHLE9BQU8sQ0FBQyxXQUFXLG1DQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ25FLGtCQUFrQixHQUFHLEVBQUUsQ0FBQztZQUN4QixLQUFLLE1BQU0sR0FBRyxJQUFJLE9BQU8sRUFBRTtnQkFDekIsa0JBQWtCLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO2FBQ2pDO1lBQ0QsSUFBSSxPQUFPLENBQUMsV0FBVyxFQUFFO2dCQUN2QixZQUFZLEdBQUcsa0JBQWtCLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQzthQUNwRTtTQUNGO1FBRUQsSUFBSSxTQUFTLEdBQUcsb0JBQVMsQ0FBQyxHQUFHLENBQUM7UUFDOUIsSUFBSSxRQUE4QixDQUFDO1FBQ25DLElBQUksWUFBWSxFQUFFO1lBQ2hCLHNDQUFzQztZQUN0QyxJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsbUJBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFO2dCQUN4RCxTQUFTLEdBQUcsb0JBQVMsQ0FBQyxJQUFJLENBQUM7Z0JBQzNCLFFBQVEsR0FBRyxtQkFBUSxDQUFDLElBQUksQ0FBQzthQUMxQjtpQkFBTSxJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsbUJBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFO2dCQUM5RCxRQUFRLEdBQUcsbUJBQVEsQ0FBQyxHQUFHLENBQUM7YUFDekI7U0FDRjtRQUVELDhDQUE4QztRQUM5QyxrQkFBa0IsQ0FBQyxNQUFNLENBQUM7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLFlBQVksRUFBRTtvQkFDWixPQUFPLEVBQUUsTUFBTTtvQkFDZixrQkFBa0IsRUFBRSxrQkFBa0IsYUFBbEIsa0JBQWtCLGNBQWxCLGtCQUFrQixHQUFJLElBQUk7b0JBQzlDLFNBQVMsUUFBRSxPQUFPLENBQUMsVUFBVSxtQ0FBSSxLQUFLO29CQUN0QyxNQUFNLFFBQUUsT0FBTyxDQUFDLE1BQU0sbUNBQUksS0FBSztvQkFDL0IsT0FBTyxFQUFFO3dCQUNQLElBQUksRUFBRSxNQUFNLGNBQWMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUU7cUJBQzlDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxRQUFRO1FBQ1IsSUFBSSxZQUE0QyxDQUFDO1FBQ2pELElBQUksQ0FBQyxPQUFPLENBQUMsbUJBQW1CLEVBQUU7WUFDaEMsWUFBWSxHQUFHLElBQUksdUJBQVksQ0FBQztnQkFDOUIsV0FBVztnQkFDWCxpQkFBaUI7Z0JBQ2pCLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUTtnQkFDMUIsV0FBVyxFQUFFLE9BQU8sQ0FBQyxpQkFBaUI7Z0JBQ3RDLFlBQVk7Z0JBQ1osU0FBUztnQkFDVCxRQUFRO2FBQ1QsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxTQUFTO1FBQ1QsTUFBTSxhQUFhLEdBQUcsSUFBSSx3QkFBYSxDQUFDO1lBQ3RDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztZQUN4QixpQkFBaUI7WUFDakIsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRO1lBQzFCLFdBQVcsRUFBRSxPQUFPLENBQUMsaUJBQWlCO1lBQ3RDLFVBQVUsRUFBRSxDQUFDLHVCQUFZLENBQUMsV0FBVyxJQUFJLE9BQU8sQ0FBQyxtQkFBbUI7WUFDcEUsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTO1lBQzVCLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYTtZQUNwQyxZQUFZO1lBQ1osU0FBUztZQUNULFFBQVE7U0FDVCxDQUFDLENBQUM7UUFFSCxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRTtZQUN4QyxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ3ZDLFFBQVEsRUFBRTtnQkFDUixLQUFLLEVBQUUsWUFBWTtnQkFDbkIsR0FBRyxhQUFhLENBQUMsZUFBZTthQUNqQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQTVGRCw0QkE0RkM7QUFFRCxTQUFTLGNBQWMsQ0FBQyxPQUF1QjtJQUM3QyxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUVoRCxJQUFJLENBQUMsS0FBSyxFQUFFO1FBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0tBQ3pEO0lBRUQsT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDbEIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBEb2NrZXJCdW5kbGVyLCBJbnN0YWxsZXIsIExvY2FsQnVuZGxlciwgTG9ja0ZpbGUgfSBmcm9tICcuL2J1bmRsZXJzJztcbmltcG9ydCB7IFBhY2thZ2VKc29uTWFuYWdlciB9IGZyb20gJy4vcGFja2FnZS1qc29uLW1hbmFnZXInO1xuaW1wb3J0IHsgZmluZFVwIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBCYXNlIG9wdGlvbnMgZm9yIFBhcmNlbCBidW5kbGluZ1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFBhcmNlbEJhc2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gbWluaWZ5IGZpbGVzIHdoZW4gYnVuZGxpbmcuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBtaW5pZnk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGluY2x1ZGUgc291cmNlIG1hcHMgd2hlbiBidW5kbGluZy5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHNvdXJjZU1hcHM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgY2FjaGUgZGlyZWN0b3J5IChyZWxhdGl2ZSB0byB0aGUgcHJvamVjdCByb290KVxuICAgKlxuICAgKiBQYXJjZWwgdXNlcyBhIGZpbGVzeXN0ZW0gY2FjaGUgZm9yIGZhc3QgcmVidWlsZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYC5wYXJjZWwtY2FjaGVgIGluIHRoZSB3b3JraW5nIGRpcmVjdG9yeVxuICAgKi9cbiAgcmVhZG9ubHkgY2FjaGVEaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByb290IG9mIHRoZSBwcm9qZWN0LiBUaGlzIHdpbGwgYmUgdXNlZCBhcyB0aGUgc291cmNlIGZvciB0aGUgdm9sdW1lXG4gICAqIG1vdW50ZWQgaW4gdGhlIERvY2tlciBjb250YWluZXIuIElmIHlvdSBzcGVjaWZ5IHRoaXMgcHJvcCwgZW5zdXJlIHRoYXRcbiAgICogdGhpcyBwYXRoIGluY2x1ZGVzIGBlbnRyeWAgYW5kIGFueSBtb2R1bGUvZGVwZW5kZW5jaWVzIHVzZWQgYnkgeW91clxuICAgKiBmdW5jdGlvbiBvdGhlcndpc2UgYnVuZGxpbmcgd2lsbCBub3QgYmUgcG9zc2libGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGNsb3Nlc3QgcGF0aCBjb250YWluaW5nIGEgLmdpdCBmb2xkZXJcbiAgICovXG4gIHJlYWRvbmx5IHByb2plY3RSb290Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBFbnZpcm9ubWVudCB2YXJpYWJsZXMgZGVmaW5lZCB3aGVuIFBhcmNlbCBydW5zLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGVudmlyb25tZW50IHZhcmlhYmxlcyBhcmUgZGVmaW5lZC5cbiAgICovXG4gIHJlYWRvbmx5IHBhcmNlbEVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmc7IH07XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBtb2R1bGVzIHRoYXQgc2hvdWxkIGJlIGNvbnNpZGVyZWQgYXMgZXh0ZXJuYWxzIChhbHJlYWR5IGF2YWlsYWJsZVxuICAgKiBpbiB0aGUgcnVudGltZSkuXG4gICAqXG4gICAqIEBkZWZhdWx0IFsnYXdzLXNkayddXG4gICAqL1xuICByZWFkb25seSBleHRlcm5hbE1vZHVsZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIG1vZHVsZXMgdGhhdCBzaG91bGQgYmUgaW5zdGFsbGVkIGluc3RlYWQgb2YgYnVuZGxlZC4gTW9kdWxlcyBhcmVcbiAgICogaW5zdGFsbGVkIGluIGEgTGFtYmRhIGNvbXBhdGlibGUgZW52aXJvbm5tZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGFsbCBtb2R1bGVzIGFyZSBidW5kbGVkXG4gICAqL1xuICByZWFkb25seSBub2RlTW9kdWxlcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBUaGUgdmVyc2lvbiBvZiBQYXJjZWwgdG8gdXNlIHdoZW4gcnVubmluZyBpbiBhIERvY2tlciBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gMi4wLjAtYmV0YS4xXG4gICAqL1xuICByZWFkb25seSBwYXJjZWxWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBCdWlsZCBhcmd1bWVudHMgdG8gcGFzcyB3aGVuIGJ1aWxkaW5nIHRoZSBidW5kbGluZyBpbWFnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBidWlsZCBhcmd1bWVudHMgYXJlIHBhc3NlZFxuICAgKi9cbiAgcmVhZG9ubHkgYnVpbGRBcmdzPzogeyBba2V5OnN0cmluZ10gOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogRm9yY2UgYnVuZGxpbmcgaW4gYSBEb2NrZXIgY29udGFpbmVyIGV2ZW4gaWYgbG9jYWwgYnVuZGxpbmcgaXNcbiAgICogcG9zc2libGUuVGhpcyAgaXMgdXNlZnVsIGlmIHlvdXIgZnVuY3Rpb24gcmVsaWVzIG9uIG5vZGUgbW9kdWxlc1xuICAgKiB0aGF0IHNob3VsZCBiZSBpbnN0YWxsZWQgKGBub2RlTW9kdWxlc2ApIGluIGEgTGFtYmRhIGNvbXBhdGlibGVcbiAgICogZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBmb3JjZURvY2tlckJ1bmRsaW5nPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBQYXJjZWwgYnVuZGxpbmdcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQYXJjZWxPcHRpb25zIGV4dGVuZHMgUGFyY2VsQmFzZU9wdGlvbnMge1xuICAvKipcbiAgICogRW50cnkgZmlsZVxuICAgKi9cbiAgcmVhZG9ubHkgZW50cnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJ1bnRpbWUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZTogbGFtYmRhLlJ1bnRpbWU7XG59XG5cbi8qKlxuICogQnVuZGxpbmdcbiAqL1xuZXhwb3J0IGNsYXNzIEJ1bmRsaW5nIHtcbiAgLyoqXG4gICAqIFBhcmNlbCBidW5kbGVkIExhbWJkYSBhc3NldCBjb2RlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHBhcmNlbChvcHRpb25zOiBQYXJjZWxPcHRpb25zKTogbGFtYmRhLkFzc2V0Q29kZSB7XG4gICAgLy8gRmluZCBwcm9qZWN0IHJvb3RcbiAgICBjb25zdCBwcm9qZWN0Um9vdCA9IG9wdGlvbnMucHJvamVjdFJvb3QgPz8gZmluZFVwKGAuZ2l0JHtwYXRoLnNlcH1gKTtcbiAgICBpZiAoIXByb2plY3RSb290KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBmaW5kIHByb2plY3Qgcm9vdC4gUGxlYXNlIHNwZWNpZnkgaXQgd2l0aCBgcHJvamVjdFJvb3RgLicpO1xuICAgIH1cbiAgICBjb25zdCByZWxhdGl2ZUVudHJ5UGF0aCA9IHBhdGgucmVsYXRpdmUocHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShvcHRpb25zLmVudHJ5KSk7XG5cbiAgICBjb25zdCBwYWNrYWdlSnNvbk1hbmFnZXIgPSBuZXcgUGFja2FnZUpzb25NYW5hZ2VyKHBhdGguZGlybmFtZShvcHRpb25zLmVudHJ5KSk7XG5cbiAgICAvLyBDb2xsZWN0IGV4dGVybmFsIGFuZCBpbnN0YWxsIG1vZHVsZXNcbiAgICBsZXQgaW5jbHVkZU5vZGVNb2R1bGVzOiB7IFtrZXk6IHN0cmluZ106IGJvb2xlYW4gfSB8IHVuZGVmaW5lZDtcbiAgICBsZXQgZGVwZW5kZW5jaWVzOiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9IHwgdW5kZWZpbmVkO1xuICAgIGNvbnN0IGV4dGVybmFsTW9kdWxlcyA9IG9wdGlvbnMuZXh0ZXJuYWxNb2R1bGVzID8/IFsnYXdzLXNkayddO1xuICAgIGlmIChleHRlcm5hbE1vZHVsZXMgfHwgb3B0aW9ucy5ub2RlTW9kdWxlcykge1xuICAgICAgY29uc3QgbW9kdWxlcyA9IFsuLi5leHRlcm5hbE1vZHVsZXMsIC4uLm9wdGlvbnMubm9kZU1vZHVsZXMgPz8gW11dO1xuICAgICAgaW5jbHVkZU5vZGVNb2R1bGVzID0ge307XG4gICAgICBmb3IgKGNvbnN0IG1vZCBvZiBtb2R1bGVzKSB7XG4gICAgICAgIGluY2x1ZGVOb2RlTW9kdWxlc1ttb2RdID0gZmFsc2U7XG4gICAgICB9XG4gICAgICBpZiAob3B0aW9ucy5ub2RlTW9kdWxlcykge1xuICAgICAgICBkZXBlbmRlbmNpZXMgPSBwYWNrYWdlSnNvbk1hbmFnZXIuZ2V0VmVyc2lvbnMob3B0aW9ucy5ub2RlTW9kdWxlcyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgbGV0IGluc3RhbGxlciA9IEluc3RhbGxlci5OUE07XG4gICAgbGV0IGxvY2tGaWxlOiBMb2NrRmlsZSB8IHVuZGVmaW5lZDtcbiAgICBpZiAoZGVwZW5kZW5jaWVzKSB7XG4gICAgICAvLyBVc2UgbnBtIHVubGVzcyB3ZSBoYXZlIGEgeWFybi5sb2NrLlxuICAgICAgaWYgKGZzLmV4aXN0c1N5bmMocGF0aC5qb2luKHByb2plY3RSb290LCBMb2NrRmlsZS5ZQVJOKSkpIHtcbiAgICAgICAgaW5zdGFsbGVyID0gSW5zdGFsbGVyLllBUk47XG4gICAgICAgIGxvY2tGaWxlID0gTG9ja0ZpbGUuWUFSTjtcbiAgICAgIH0gZWxzZSBpZiAoZnMuZXhpc3RzU3luYyhwYXRoLmpvaW4ocHJvamVjdFJvb3QsIExvY2tGaWxlLk5QTSkpKSB7XG4gICAgICAgIGxvY2tGaWxlID0gTG9ja0ZpbGUuTlBNO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIENvbmZpZ3VyZSB0YXJnZXQgaW4gcGFja2FnZS5qc29uIGZvciBQYXJjZWxcbiAgICBwYWNrYWdlSnNvbk1hbmFnZXIudXBkYXRlKHtcbiAgICAgIHRhcmdldHM6IHtcbiAgICAgICAgJ2Nkay1sYW1iZGEnOiB7XG4gICAgICAgICAgY29udGV4dDogJ25vZGUnLFxuICAgICAgICAgIGluY2x1ZGVOb2RlTW9kdWxlczogaW5jbHVkZU5vZGVNb2R1bGVzID8/IHRydWUsXG4gICAgICAgICAgc291cmNlTWFwOiBvcHRpb25zLnNvdXJjZU1hcHMgPz8gZmFsc2UsXG4gICAgICAgICAgbWluaWZ5OiBvcHRpb25zLm1pbmlmeSA/PyBmYWxzZSxcbiAgICAgICAgICBlbmdpbmVzOiB7XG4gICAgICAgICAgICBub2RlOiBgPj0gJHtydW50aW1lVmVyc2lvbihvcHRpb25zLnJ1bnRpbWUpfWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBMb2NhbFxuICAgIGxldCBsb2NhbEJ1bmRsZXI6IGNkay5JTG9jYWxCdW5kbGluZyB8IHVuZGVmaW5lZDtcbiAgICBpZiAoIW9wdGlvbnMuZm9yY2VEb2NrZXJCdW5kbGluZykge1xuICAgICAgbG9jYWxCdW5kbGVyID0gbmV3IExvY2FsQnVuZGxlcih7XG4gICAgICAgIHByb2plY3RSb290LFxuICAgICAgICByZWxhdGl2ZUVudHJ5UGF0aCxcbiAgICAgICAgY2FjaGVEaXI6IG9wdGlvbnMuY2FjaGVEaXIsXG4gICAgICAgIGVudmlyb25tZW50OiBvcHRpb25zLnBhcmNlbEVudmlyb25tZW50LFxuICAgICAgICBkZXBlbmRlbmNpZXMsXG4gICAgICAgIGluc3RhbGxlcixcbiAgICAgICAgbG9ja0ZpbGUsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBEb2NrZXJcbiAgICBjb25zdCBkb2NrZXJCdW5kbGVyID0gbmV3IERvY2tlckJ1bmRsZXIoe1xuICAgICAgcnVudGltZTogb3B0aW9ucy5ydW50aW1lLFxuICAgICAgcmVsYXRpdmVFbnRyeVBhdGgsXG4gICAgICBjYWNoZURpcjogb3B0aW9ucy5jYWNoZURpcixcbiAgICAgIGVudmlyb25tZW50OiBvcHRpb25zLnBhcmNlbEVudmlyb25tZW50LFxuICAgICAgYnVpbGRJbWFnZTogIUxvY2FsQnVuZGxlci5ydW5zTG9jYWxseSB8fCBvcHRpb25zLmZvcmNlRG9ja2VyQnVuZGxpbmcsXG4gICAgICBidWlsZEFyZ3M6IG9wdGlvbnMuYnVpbGRBcmdzLFxuICAgICAgcGFyY2VsVmVyc2lvbjogb3B0aW9ucy5wYXJjZWxWZXJzaW9uLFxuICAgICAgZGVwZW5kZW5jaWVzLFxuICAgICAgaW5zdGFsbGVyLFxuICAgICAgbG9ja0ZpbGUsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHByb2plY3RSb290LCB7XG4gICAgICBhc3NldEhhc2hUeXBlOiBjZGsuQXNzZXRIYXNoVHlwZS5CVU5ETEUsXG4gICAgICBidW5kbGluZzoge1xuICAgICAgICBsb2NhbDogbG9jYWxCdW5kbGVyLFxuICAgICAgICAuLi5kb2NrZXJCdW5kbGVyLmJ1bmRsaW5nT3B0aW9ucyxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gcnVudGltZVZlcnNpb24ocnVudGltZTogbGFtYmRhLlJ1bnRpbWUpOiBzdHJpbmcge1xuICBjb25zdCBtYXRjaCA9IHJ1bnRpbWUubmFtZS5tYXRjaCgvbm9kZWpzKFxcZCspLyk7XG5cbiAgaWYgKCFtYXRjaCkge1xuICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGV4dHJhY3QgdmVyc2lvbiBmcm9tIHJ1bnRpbWUuJyk7XG4gIH1cblxuICByZXR1cm4gbWF0Y2hbMV07XG59XG4iXX0=