"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyStackSynthesizer = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cfn_fn_1 = require("../cfn-fn");
const construct_compat_1 = require("../construct-compat");
const asset_parameters_1 = require("../private/asset-parameters");
const _shared_1 = require("./_shared");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the original deployment environment
 *
 * This deployment environment is restricted in cross-environment deployments,
 * CI/CD deployments, and will use up CloudFormation parameters in your template.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 */
class LegacyStackSynthesizer {
    constructor() {
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    bind(stack) {
        this.stack = stack;
    }
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        //
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addFileAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        // See `addFileAsset` for explanation.
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addDockerImageAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    synthesizeStackArtifacts(session) {
        _shared_1.assertBound(this.stack);
        // Just do the default stuff, nothing special
        _shared_1.addStackArtifactToAssembly(session, this.stack, {}, []);
    }
    doAddDockerImageAsset(asset) {
        var _a, _b;
        _shared_1.assertBound(this.stack);
        // check if we have an override from context
        const repositoryNameOverride = this.stack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = (_b = (_a = asset.repositoryName) !== null && _a !== void 0 ? _a : repositoryNameOverride) !== null && _b !== void 0 ? _b : ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.stack.region}.${this.stack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        _shared_1.assertBound(this.stack);
        if (!this._assetParameters) {
            this._assetParameters = new construct_compat_1.Construct(this.stack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
//# sourceMappingURL=data:application/json;base64,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