"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoveTag = exports.Tags = exports.Tag = void 0;
// import * as cxapi from '@aws-cdk/cx-api';
const aspect_1 = require("./aspect");
const tag_manager_1 = require("./tag-manager");
const annotations_1 = require("./annotations");
/**
 * The common functionality for Tag and Remove Tag Aspects
 */
class TagBase {
    constructor(key, props = {}) {
        this.key = key;
        this.props = props;
    }
    visit(construct) {
        if (tag_manager_1.TagManager.isTaggable(construct)) {
            this.applyTag(construct);
        }
    }
}
/**
 * The Tag Aspect will handle adding a tag to this node and cascading tags to children
 */
class Tag extends TagBase {
    constructor(key, value, props = {}) {
        super(key, props);
        this.defaultPriority = 100;
        if (value === undefined) {
            throw new Error('Tag must have a value');
        }
        this.value = value;
    }
    /**
     * DEPRECATED: add tags to the node of a construct and all its the taggable children
     *
     * @deprecated use `Tags.of(scope).add()`
     */
    static add(scope, key, value, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.add(scope,k,v)', 'Use "Tags.of(scope).add(k,v)" instead');
        Tags.of(scope).add(key, value, props);
    }
    /**
     * DEPRECATED: remove tags to the node of a construct and all its the taggable children
     *
     * @deprecated use `Tags.of(scope).remove()`
     */
    static remove(scope, key, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.remove(scope,k,v)', 'Use "Tags.of(scope).remove(k,v)" instead');
        Tags.of(scope).remove(key, props);
    }
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.setTag(this.key, this.value, this.props.priority !== undefined ? this.props.priority : this.defaultPriority, this.props.applyToLaunchedInstances !== false);
        }
    }
}
exports.Tag = Tag;
/**
 * Manages AWS tags for all resources within a construct scope.
 */
class Tags {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the tags API for this scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Tags(scope);
    }
    /**
     * add tags to the node of a construct and all its the taggable children
     */
    add(key, value, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new Tag(key, value, props));
    }
    /**
     * remove tags to the node of a construct and all its the taggable children
     */
    remove(key, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new RemoveTag(key, props));
    }
}
exports.Tags = Tags;
/**
 * The RemoveTag Aspect will handle removing tags from this node and children
 */
class RemoveTag extends TagBase {
    constructor(key, props = {}) {
        super(key, props);
        this.defaultPriority = 200;
    }
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.removeTag(this.key, this.props.priority !== undefined ? this.props.priority : this.defaultPriority);
        }
    }
}
exports.RemoveTag = RemoveTag;
//# sourceMappingURL=data:application/json;base64,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