"use strict";
const fs = require("fs");
const path = require("path");
const ts_mock_imports_1 = require("ts-mock-imports");
const fs_1 = require("../../lib/fs");
const util = require("../../lib/fs/utils");
module.exports = {
    shouldExclude: {
        'excludes nothing by default'(test) {
            test.ok(!util.shouldExclude([], path.join('some', 'file', 'path')));
            test.done();
        },
        'excludes requested files'(test) {
            const exclusions = ['*.ignored'];
            test.ok(util.shouldExclude(exclusions, path.join('some', 'file.ignored')));
            test.ok(!util.shouldExclude(exclusions, path.join('some', 'important', 'file')));
            test.done();
        },
        'does not exclude whitelisted files'(test) {
            const exclusions = ['*.ignored', '!important.*'];
            test.ok(!util.shouldExclude(exclusions, path.join('some', 'important.ignored')));
            test.done();
        },
    },
    shouldFollow: {
        always: {
            'follows internal'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join(sourceRoot, 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', true);
                try {
                    test.ok(util.shouldFollow(fs_1.SymlinkFollowMode.ALWAYS, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'follows external'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join('alternate', 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', true);
                try {
                    test.ok(util.shouldFollow(fs_1.SymlinkFollowMode.ALWAYS, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'does not follow internal when the referent does not exist'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join(sourceRoot, 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', false);
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.ALWAYS, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'does not follow external when the referent does not exist'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join('alternate', 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', false);
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.ALWAYS, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
        },
        external: {
            'does not follow internal'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join(sourceRoot, 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync');
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.EXTERNAL, sourceRoot, linkTarget));
                    test.ok(mockFsExists.notCalled);
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'follows external'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join('alternate', 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', true);
                try {
                    test.ok(util.shouldFollow(fs_1.SymlinkFollowMode.EXTERNAL, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'does not follow external when referent does not exist'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join('alternate', 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', false);
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.EXTERNAL, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
        },
        blockExternal: {
            'follows internal'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join(sourceRoot, 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', true);
                try {
                    test.ok(util.shouldFollow(fs_1.SymlinkFollowMode.BLOCK_EXTERNAL, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'does not follow internal when referent does not exist'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join(sourceRoot, 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync', false);
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.BLOCK_EXTERNAL, sourceRoot, linkTarget));
                    test.ok(mockFsExists.calledOnceWith(linkTarget));
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'does not follow external'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join('alternate', 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync');
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.BLOCK_EXTERNAL, sourceRoot, linkTarget));
                    test.ok(mockFsExists.notCalled);
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
        },
        never: {
            'does not follow internal'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join(sourceRoot, 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync');
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.NEVER, sourceRoot, linkTarget));
                    test.ok(mockFsExists.notCalled);
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
            'does not follow external'(test) {
                const sourceRoot = path.join('source', 'root');
                const linkTarget = path.join('alternate', 'referent');
                const mockFsExists = ts_mock_imports_1.ImportMock.mockFunction(fs, 'existsSync');
                try {
                    test.ok(!util.shouldFollow(fs_1.SymlinkFollowMode.NEVER, sourceRoot, linkTarget));
                    test.ok(mockFsExists.notCalled);
                    test.done();
                }
                finally {
                    mockFsExists.restore();
                }
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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