"use strict";
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const util_1 = require("./util");
module.exports = {
    'create from components with defaults'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'sqs',
            resource: 'myqueuename',
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:sqs:${pseudo.region}:${pseudo.accountId}:myqueuename`));
        test.done();
    },
    'create from components with specific values for the various components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'dynamodb',
            resource: 'table',
            account: '123456789012',
            region: 'us-east-1',
            partition: 'aws-cn',
            resourceName: 'mytable/stream/label',
        });
        test.deepEqual(stack.resolve(arn), 'arn:aws-cn:dynamodb:us-east-1:123456789012:table/mytable/stream/label');
        test.done();
    },
    'allow empty string in components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 's3',
            resource: 'my-bucket',
            account: '',
            region: '',
            partition: 'aws-cn',
        });
        test.deepEqual(stack.resolve(arn), 'arn:aws-cn:s3:::my-bucket');
        test.done();
    },
    'resourcePathSep can be set to ":" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'codedeploy',
            resource: 'application',
            sep: ':',
            resourceName: 'WordPress_App',
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:codedeploy:${pseudo.region}:${pseudo.accountId}:application:WordPress_App`));
        test.done();
    },
    'resourcePathSep can be set to "" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'ssm',
            resource: 'parameter',
            sep: '',
            resourceName: '/parameter-name',
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:ssm:${pseudo.region}:${pseudo.accountId}:parameter/parameter-name`));
        test.done();
    },
    'fails if resourcePathSep is neither ":" nor "/"'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => stack.formatArn({
            service: 'foo',
            resource: 'bar',
            sep: 'x',
        }));
        test.done();
    },
    'Arn.parse(s)': {
        fails: {
            'if doesn\'t start with "arn:"'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('barn:foo:x:a:1:2'), /ARNs must start with "arn:": barn:foo/);
                test.done();
            },
            'if the ARN doesnt have enough components'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:is:too:short'), /ARNs must have at least 6 components: arn:is:too:short/);
                test.done();
            },
            'if "service" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws::4:5:6'), /The `service` component \(3rd component\) is required/);
                test.done();
            },
            'if "resource" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws:service:::'), /The `resource` component \(6th component\) is required/);
                test.done();
            },
        },
        'various successful parses'(test) {
            const stack = new lib_1.Stack();
            const tests = {
                'arn:aws:a4b:region:accountid:resourcetype/resource': {
                    partition: 'aws',
                    service: 'a4b',
                    region: 'region',
                    account: 'accountid',
                    resource: 'resourcetype',
                    resourceName: 'resource',
                    sep: '/',
                },
                'arn:aws:apigateway:us-east-1::a123456789012bc3de45678901f23a45:/test/mydemoresource/*': {
                    partition: 'aws',
                    service: 'apigateway',
                    region: 'us-east-1',
                    account: '',
                    resource: 'a123456789012bc3de45678901f23a45',
                    sep: ':',
                    resourceName: '/test/mydemoresource/*',
                },
                'arn:aws-cn:cloud9::123456789012:environment:81e900317347585a0601e04c8d52eaEX': {
                    partition: 'aws-cn',
                    service: 'cloud9',
                    region: '',
                    account: '123456789012',
                    resource: 'environment',
                    resourceName: '81e900317347585a0601e04c8d52eaEX',
                    sep: ':',
                },
                'arn::cognito-sync:::identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla': {
                    service: 'cognito-sync',
                    region: '',
                    account: '',
                    resource: 'identitypool',
                    resourceName: 'us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla',
                    sep: '/',
                },
                'arn:aws:s3:::my_corporate_bucket': {
                    partition: 'aws',
                    service: 's3',
                    region: '',
                    account: '',
                    resource: 'my_corporate_bucket',
                },
            };
            Object.keys(tests).forEach(arn => {
                const expected = tests[arn];
                test.deepEqual(stack.parseArn(arn), expected, arn);
            });
            test.done();
        },
        'a Token with : separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new intrinsic_1.Intrinsic(theToken).toString(), ':');
            test.deepEqual(stack.resolve(parsed.partition), { 'Fn::Select': [1, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.service), { 'Fn::Select': [2, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.region), { 'Fn::Select': [3, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.account), { 'Fn::Select': [4, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.resource), { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.resourceName), { 'Fn::Select': [6, { 'Fn::Split': [':', theToken] }] });
            test.equal(parsed.sep, ':');
            test.done();
        },
        'a Token with / separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new intrinsic_1.Intrinsic(theToken).toString());
            test.equal(parsed.sep, '/');
            // eslint-disable-next-line max-len
            test.deepEqual(stack.resolve(parsed.resource), { 'Fn::Select': [0, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            // eslint-disable-next-line max-len
            test.deepEqual(stack.resolve(parsed.resourceName), { 'Fn::Select': [1, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            test.done();
        },
        'returns empty string ARN components'(test) {
            const stack = new lib_1.Stack();
            const arn = 'arn:aws:iam::123456789012:role/abc123';
            const expected = {
                partition: 'aws',
                service: 'iam',
                region: '',
                account: '123456789012',
                resource: 'role',
                resourceName: 'abc123',
                sep: '/',
            };
            test.deepEqual(stack.parseArn(arn), expected, arn);
            test.done();
        },
    },
    'can use a fully specified ARN from a different stack without incurring an import'(test) {
        // GIVEN
        const stack1 = new lib_1.Stack(undefined, 'Stack1', { env: { account: '12345678', region: 'us-turbo-5' } });
        const stack2 = new lib_1.Stack(undefined, 'Stack2', { env: { account: '87654321', region: 'us-turbo-1' } });
        // WHEN
        const arn = stack1.formatArn({
            // No partition specified here
            service: 'bla',
            resource: 'thing',
            resourceName: 'thong',
        });
        new lib_1.CfnOutput(stack2, 'SomeValue', { value: arn });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack2), {
            Outputs: {
                SomeValue: {
                    Value: {
                        // Look ma, no Fn::ImportValue!
                        'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':bla:us-turbo-5:12345678:thing/thong']],
                    },
                },
            },
        });
        test.done();
    },
    'parse other fields if only some are tokens'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        const parsed = stack.parseArn(`arn:${lib_1.Aws.PARTITION}:iam::123456789012:role/S3Access`);
        // THEN
        test.deepEqual(stack.resolve(parsed.partition), { Ref: 'AWS::Partition' });
        test.deepEqual(stack.resolve(parsed.service), 'iam');
        test.equal(stack.resolve(parsed.region), '');
        test.deepEqual(stack.resolve(parsed.account), '123456789012');
        test.deepEqual(stack.resolve(parsed.resource), 'role');
        test.deepEqual(stack.resolve(parsed.resourceName), 'S3Access');
        test.equal(parsed.sep, '/');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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