"use strict";
const lib_1 = require("../lib");
const context_provider_1 = require("../lib/context-provider");
const synthesis_1 = require("../lib/private/synthesis");
/**
 * Get the expected context key from a stack with missing parameters
 */
function expectedContextKey(stack) {
    const missing = synthesis_1.synthesize(stack).manifest.missing;
    if (!missing || missing.length !== 1) {
        throw new Error('Expecting assembly to include a single missing context report');
    }
    return missing[0].key;
}
module.exports = {
    'AvailabilityZoneProvider returns a list with dummy values if the context is not available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['dummy1a', 'dummy1b', 'dummy1c']);
        test.done();
    },
    'AvailabilityZoneProvider will return context list if available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, ['us-east-1a', 'us-east-1b']);
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['us-east-1a', 'us-east-1b']);
        test.done();
    },
    'AvailabilityZoneProvider will complain if not given a list'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, 'not-a-list');
        test.throws(() => stack.availabilityZones);
        test.done();
    },
    'ContextProvider consistently generates a key'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'ssm',
            props: {
                parameterName: 'foo',
                anyStringParam: 'bar',
            },
        });
        test.deepEqual(key, {
            key: 'ssm:account=12345:anyStringParam=bar:parameterName=foo:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                parameterName: 'foo',
                anyStringParam: 'bar',
            },
        });
        const complexKey = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'vpc',
            props: {
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            },
        });
        test.deepEqual(complexKey, {
            key: 'vpc:account=12345:cidrBlock=192.168.0.16:igw=false:region=us-east-1:tags.Env=Preprod:tags.Name=MyVPC',
            props: {
                account: '12345',
                region: 'us-east-1',
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            },
        });
        test.done();
    },
    'Key generation can contain arbitrarily deep structures'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            },
        });
        // THEN
        test.deepEqual(key, {
            key: 'provider:account=12345:list.0.key=key1:list.0.value=value1:list.1.key=key2:list.1.value=value2:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            },
        });
        test.done();
    },
    'Keys with undefined values are not serialized'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const result = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                p1: 42,
                p2: undefined,
            },
        });
        // THEN
        test.deepEqual(result, {
            key: 'provider:account=12345:p1=42:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                p1: 42,
                p2: undefined,
            },
        });
        test.done();
    },
    'context provider errors are attached to tree'(test) {
        const contextProps = { provider: 'availability-zones' };
        const contextKey = 'availability-zones:account=12345:region=us-east-1'; // Depends on the mangling algo
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // NOTE: error key is inlined here because it's part of the CX-API
        // compatibility surface.
        stack.node.setContext(contextKey, { $providerError: 'I had a boo-boo' });
        const construct = new lib_1.Construct(stack, 'Child');
        // Verify that we got the right hardcoded key above, give a descriptive error if not
        test.equals(context_provider_1.ContextProvider.getKey(construct, contextProps).key, contextKey);
        // WHEN
        context_provider_1.ContextProvider.getValue(construct, {
            ...contextProps,
            dummyValue: undefined,
        });
        // THEN
        const error = construct.node.metadata.find(m => m.type === 'aws:cdk:error');
        test.equals(error && error.data, 'I had a boo-boo');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jb250ZXh0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5jb250ZXh0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFDQSxnQ0FBMEM7QUFDMUMsOERBQTBEO0FBQzFELHdEQUFzRDtBQXdLdEQ7O0dBRUc7QUFDSCxTQUFTLGtCQUFrQixDQUFDLEtBQVk7SUFDdEMsTUFBTSxPQUFPLEdBQUcsc0JBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDO0lBQ25ELElBQUksQ0FBQyxPQUFPLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDO0tBQ2xGO0lBQ0QsT0FBTyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO0FBQ3hCLENBQUM7QUEvS0QsaUJBQVM7SUFDUCwyRkFBMkYsQ0FBQyxJQUFVO1FBQ3BHLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDcEcsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBRXBDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnRUFBZ0UsQ0FBQyxJQUFVO1FBQ3pFLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDcEcsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBQ3ZDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQzFELE1BQU0sR0FBRyxHQUFHLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXRDLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQVksRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO1FBRXpELE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQVksRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0REFBNEQsQ0FBQyxJQUFVO1FBQ3JFLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDcEcsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBQ3ZDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQzFELE1BQU0sR0FBRyxHQUFHLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXRDLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUV6QyxJQUFJLENBQUMsTUFBTSxDQUNULEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FDOUIsQ0FBQztRQUVGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO1FBQ3ZELE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDcEcsTUFBTSxHQUFHLEdBQUcsa0NBQWUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFO1lBQ3hDLFFBQVEsRUFBRSxLQUFLO1lBQ2YsS0FBSyxFQUFFO2dCQUNMLGFBQWEsRUFBRSxLQUFLO2dCQUNwQixjQUFjLEVBQUUsS0FBSzthQUN0QjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxFQUFFO1lBQ2xCLEdBQUcsRUFBRSx5RUFBeUU7WUFDOUUsS0FBSyxFQUFFO2dCQUNMLE9BQU8sRUFBRSxPQUFPO2dCQUNoQixNQUFNLEVBQUUsV0FBVztnQkFDbkIsYUFBYSxFQUFFLEtBQUs7Z0JBQ3BCLGNBQWMsRUFBRSxLQUFLO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxVQUFVLEdBQUcsa0NBQWUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFO1lBQy9DLFFBQVEsRUFBRSxLQUFLO1lBQ2YsS0FBSyxFQUFFO2dCQUNMLFNBQVMsRUFBRSxjQUFjO2dCQUN6QixJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUU7Z0JBQ3ZDLEdBQUcsRUFBRSxLQUFLO2FBQ1g7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRTtZQUN6QixHQUFHLEVBQUUsc0dBQXNHO1lBQzNHLEtBQUssRUFBRTtnQkFDTCxPQUFPLEVBQUUsT0FBTztnQkFDaEIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLFNBQVMsRUFBRSxjQUFjO2dCQUN6QixJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUU7Z0JBQ3ZDLEdBQUcsRUFBRSxLQUFLO2FBQ1g7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsd0RBQXdELENBQUMsSUFBVTtRQUNqRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsU0FBUyxFQUFFLFdBQVcsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVwRyxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsa0NBQWUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFO1lBQ3hDLFFBQVEsRUFBRSxVQUFVO1lBQ3BCLEtBQUssRUFBRTtnQkFDTCxJQUFJLEVBQUU7b0JBQ0osRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUU7b0JBQ2hDLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFO2lCQUNqQzthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxFQUFFO1lBQ2xCLEdBQUcsRUFBRSxpSEFBaUg7WUFDdEgsS0FBSyxFQUFFO2dCQUNMLE9BQU8sRUFBRSxPQUFPO2dCQUNoQixNQUFNLEVBQUUsV0FBVztnQkFDbkIsSUFBSSxFQUFFO29CQUNKLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFO29CQUNoQyxFQUFFLEdBQUcsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRTtpQkFDakM7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQ0FBK0MsQ0FBQyxJQUFVO1FBQ3hELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxTQUFTLEVBQUUsV0FBVyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRXBHLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxrQ0FBZSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUU7WUFDM0MsUUFBUSxFQUFFLFVBQVU7WUFDcEIsS0FBSyxFQUFFO2dCQUNMLEVBQUUsRUFBRSxFQUFFO2dCQUNOLEVBQUUsRUFBRSxTQUFTO2FBQ2Q7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUU7WUFDckIsR0FBRyxFQUFFLCtDQUErQztZQUNwRCxLQUFLLEVBQUU7Z0JBQ0wsT0FBTyxFQUFFLE9BQU87Z0JBQ2hCLE1BQU0sRUFBRSxXQUFXO2dCQUNuQixFQUFFLEVBQUUsRUFBRTtnQkFDTixFQUFFLEVBQUUsU0FBUzthQUNkO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhDQUE4QyxDQUFDLElBQVU7UUFDdkQsTUFBTSxZQUFZLEdBQUcsRUFBRSxRQUFRLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQztRQUN4RCxNQUFNLFVBQVUsR0FBRyxtREFBbUQsQ0FBQyxDQUFDLCtCQUErQjtRQUV2RyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsU0FBUyxFQUFFLFdBQVcsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVwRyxrRUFBa0U7UUFDbEUseUJBQXlCO1FBQ3pCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVUsRUFBRSxFQUFFLGNBQWMsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDekUsTUFBTSxTQUFTLEdBQUcsSUFBSSxlQUFTLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRWhELG9GQUFvRjtRQUNwRixJQUFJLENBQUMsTUFBTSxDQUFDLGtDQUFlLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUMsQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFN0UsT0FBTztRQUNQLGtDQUFlLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUNsQyxHQUFHLFlBQVk7WUFDZixVQUFVLEVBQUUsU0FBUztTQUN0QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxlQUFlLENBQUMsQ0FBQztRQUM1RSxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsSUFBSSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFFcEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBTdGFjayB9IGZyb20gJy4uL2xpYic7XG5pbXBvcnQgeyBDb250ZXh0UHJvdmlkZXIgfSBmcm9tICcuLi9saWIvY29udGV4dC1wcm92aWRlcic7XG5pbXBvcnQgeyBzeW50aGVzaXplIH0gZnJvbSAnLi4vbGliL3ByaXZhdGUvc3ludGhlc2lzJztcblxuZXhwb3J0ID0ge1xuICAnQXZhaWxhYmlsaXR5Wm9uZVByb3ZpZGVyIHJldHVybnMgYSBsaXN0IHdpdGggZHVtbXkgdmFsdWVzIGlmIHRoZSBjb250ZXh0IGlzIG5vdCBhdmFpbGFibGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDUnLCByZWdpb246ICd1cy1lYXN0LTEnIH0gfSk7XG4gICAgY29uc3QgYXpzID0gc3RhY2suYXZhaWxhYmlsaXR5Wm9uZXM7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChhenMsIFsnZHVtbXkxYScsICdkdW1teTFiJywgJ2R1bW15MWMnXSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0F2YWlsYWJpbGl0eVpvbmVQcm92aWRlciB3aWxsIHJldHVybiBjb250ZXh0IGxpc3QgaWYgYXZhaWxhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCAnVGVzdFN0YWNrJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1JywgcmVnaW9uOiAndXMtZWFzdC0xJyB9IH0pO1xuICAgIGNvbnN0IGJlZm9yZSA9IHN0YWNrLmF2YWlsYWJpbGl0eVpvbmVzO1xuICAgIHRlc3QuZGVlcEVxdWFsKGJlZm9yZSwgWydkdW1teTFhJywgJ2R1bW15MWInLCAnZHVtbXkxYyddKTtcbiAgICBjb25zdCBrZXkgPSBleHBlY3RlZENvbnRleHRLZXkoc3RhY2spO1xuXG4gICAgc3RhY2subm9kZS5zZXRDb250ZXh0KGtleSwgWyd1cy1lYXN0LTFhJywgJ3VzLWVhc3QtMWInXSk7XG5cbiAgICBjb25zdCBhenMgPSBzdGFjay5hdmFpbGFiaWxpdHlab25lcztcbiAgICB0ZXN0LmRlZXBFcXVhbChhenMsIFsndXMtZWFzdC0xYScsICd1cy1lYXN0LTFiJ10pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0F2YWlsYWJpbGl0eVpvbmVQcm92aWRlciB3aWxsIGNvbXBsYWluIGlmIG5vdCBnaXZlbiBhIGxpc3QnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDUnLCByZWdpb246ICd1cy1lYXN0LTEnIH0gfSk7XG4gICAgY29uc3QgYmVmb3JlID0gc3RhY2suYXZhaWxhYmlsaXR5Wm9uZXM7XG4gICAgdGVzdC5kZWVwRXF1YWwoYmVmb3JlLCBbJ2R1bW15MWEnLCAnZHVtbXkxYicsICdkdW1teTFjJ10pO1xuICAgIGNvbnN0IGtleSA9IGV4cGVjdGVkQ29udGV4dEtleShzdGFjayk7XG5cbiAgICBzdGFjay5ub2RlLnNldENvbnRleHQoa2V5LCAnbm90LWEtbGlzdCcpO1xuXG4gICAgdGVzdC50aHJvd3MoXG4gICAgICAoKSA9PiBzdGFjay5hdmFpbGFiaWxpdHlab25lcyxcbiAgICApO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0NvbnRleHRQcm92aWRlciBjb25zaXN0ZW50bHkgZ2VuZXJhdGVzIGEga2V5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCAnVGVzdFN0YWNrJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1JywgcmVnaW9uOiAndXMtZWFzdC0xJyB9IH0pO1xuICAgIGNvbnN0IGtleSA9IENvbnRleHRQcm92aWRlci5nZXRLZXkoc3RhY2ssIHtcbiAgICAgIHByb3ZpZGVyOiAnc3NtJyxcbiAgICAgIHByb3BzOiB7XG4gICAgICAgIHBhcmFtZXRlck5hbWU6ICdmb28nLFxuICAgICAgICBhbnlTdHJpbmdQYXJhbTogJ2JhcicsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoa2V5LCB7XG4gICAgICBrZXk6ICdzc206YWNjb3VudD0xMjM0NTphbnlTdHJpbmdQYXJhbT1iYXI6cGFyYW1ldGVyTmFtZT1mb286cmVnaW9uPXVzLWVhc3QtMScsXG4gICAgICBwcm9wczoge1xuICAgICAgICBhY2NvdW50OiAnMTIzNDUnLFxuICAgICAgICByZWdpb246ICd1cy1lYXN0LTEnLFxuICAgICAgICBwYXJhbWV0ZXJOYW1lOiAnZm9vJyxcbiAgICAgICAgYW55U3RyaW5nUGFyYW06ICdiYXInLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNvbXBsZXhLZXkgPSBDb250ZXh0UHJvdmlkZXIuZ2V0S2V5KHN0YWNrLCB7XG4gICAgICBwcm92aWRlcjogJ3ZwYycsXG4gICAgICBwcm9wczoge1xuICAgICAgICBjaWRyQmxvY2s6ICcxOTIuMTY4LjAuMTYnLFxuICAgICAgICB0YWdzOiB7IE5hbWU6ICdNeVZQQycsIEVudjogJ1ByZXByb2QnIH0sXG4gICAgICAgIGlndzogZmFsc2UsXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGNvbXBsZXhLZXksIHtcbiAgICAgIGtleTogJ3ZwYzphY2NvdW50PTEyMzQ1OmNpZHJCbG9jaz0xOTIuMTY4LjAuMTY6aWd3PWZhbHNlOnJlZ2lvbj11cy1lYXN0LTE6dGFncy5FbnY9UHJlcHJvZDp0YWdzLk5hbWU9TXlWUEMnLFxuICAgICAgcHJvcHM6IHtcbiAgICAgICAgYWNjb3VudDogJzEyMzQ1JyxcbiAgICAgICAgcmVnaW9uOiAndXMtZWFzdC0xJyxcbiAgICAgICAgY2lkckJsb2NrOiAnMTkyLjE2OC4wLjE2JyxcbiAgICAgICAgdGFnczogeyBOYW1lOiAnTXlWUEMnLCBFbnY6ICdQcmVwcm9kJyB9LFxuICAgICAgICBpZ3c6IGZhbHNlLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnS2V5IGdlbmVyYXRpb24gY2FuIGNvbnRhaW4gYXJiaXRyYXJpbHkgZGVlcCBzdHJ1Y3R1cmVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDUnLCByZWdpb246ICd1cy1lYXN0LTEnIH0gfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qga2V5ID0gQ29udGV4dFByb3ZpZGVyLmdldEtleShzdGFjaywge1xuICAgICAgcHJvdmlkZXI6ICdwcm92aWRlcicsXG4gICAgICBwcm9wczoge1xuICAgICAgICBsaXN0OiBbXG4gICAgICAgICAgeyBrZXk6ICdrZXkxJywgdmFsdWU6ICd2YWx1ZTEnIH0sXG4gICAgICAgICAgeyBrZXk6ICdrZXkyJywgdmFsdWU6ICd2YWx1ZTInIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKGtleSwge1xuICAgICAga2V5OiAncHJvdmlkZXI6YWNjb3VudD0xMjM0NTpsaXN0LjAua2V5PWtleTE6bGlzdC4wLnZhbHVlPXZhbHVlMTpsaXN0LjEua2V5PWtleTI6bGlzdC4xLnZhbHVlPXZhbHVlMjpyZWdpb249dXMtZWFzdC0xJyxcbiAgICAgIHByb3BzOiB7XG4gICAgICAgIGFjY291bnQ6ICcxMjM0NScsXG4gICAgICAgIHJlZ2lvbjogJ3VzLWVhc3QtMScsXG4gICAgICAgIGxpc3Q6IFtcbiAgICAgICAgICB7IGtleTogJ2tleTEnLCB2YWx1ZTogJ3ZhbHVlMScgfSxcbiAgICAgICAgICB7IGtleTogJ2tleTInLCB2YWx1ZTogJ3ZhbHVlMicgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnS2V5cyB3aXRoIHVuZGVmaW5lZCB2YWx1ZXMgYXJlIG5vdCBzZXJpYWxpemVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDUnLCByZWdpb246ICd1cy1lYXN0LTEnIH0gfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVzdWx0ID0gQ29udGV4dFByb3ZpZGVyLmdldEtleShzdGFjaywge1xuICAgICAgcHJvdmlkZXI6ICdwcm92aWRlcicsXG4gICAgICBwcm9wczoge1xuICAgICAgICBwMTogNDIsXG4gICAgICAgIHAyOiB1bmRlZmluZWQsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHJlc3VsdCwge1xuICAgICAga2V5OiAncHJvdmlkZXI6YWNjb3VudD0xMjM0NTpwMT00MjpyZWdpb249dXMtZWFzdC0xJyxcbiAgICAgIHByb3BzOiB7XG4gICAgICAgIGFjY291bnQ6ICcxMjM0NScsXG4gICAgICAgIHJlZ2lvbjogJ3VzLWVhc3QtMScsXG4gICAgICAgIHAxOiA0MixcbiAgICAgICAgcDI6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29udGV4dCBwcm92aWRlciBlcnJvcnMgYXJlIGF0dGFjaGVkIHRvIHRyZWUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBjb250ZXh0UHJvcHMgPSB7IHByb3ZpZGVyOiAnYXZhaWxhYmlsaXR5LXpvbmVzJyB9O1xuICAgIGNvbnN0IGNvbnRleHRLZXkgPSAnYXZhaWxhYmlsaXR5LXpvbmVzOmFjY291bnQ9MTIzNDU6cmVnaW9uPXVzLWVhc3QtMSc7IC8vIERlcGVuZHMgb24gdGhlIG1hbmdsaW5nIGFsZ29cblxuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCAnVGVzdFN0YWNrJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1JywgcmVnaW9uOiAndXMtZWFzdC0xJyB9IH0pO1xuXG4gICAgLy8gTk9URTogZXJyb3Iga2V5IGlzIGlubGluZWQgaGVyZSBiZWNhdXNlIGl0J3MgcGFydCBvZiB0aGUgQ1gtQVBJXG4gICAgLy8gY29tcGF0aWJpbGl0eSBzdXJmYWNlLlxuICAgIHN0YWNrLm5vZGUuc2V0Q29udGV4dChjb250ZXh0S2V5LCB7ICRwcm92aWRlckVycm9yOiAnSSBoYWQgYSBib28tYm9vJyB9KTtcbiAgICBjb25zdCBjb25zdHJ1Y3QgPSBuZXcgQ29uc3RydWN0KHN0YWNrLCAnQ2hpbGQnKTtcblxuICAgIC8vIFZlcmlmeSB0aGF0IHdlIGdvdCB0aGUgcmlnaHQgaGFyZGNvZGVkIGtleSBhYm92ZSwgZ2l2ZSBhIGRlc2NyaXB0aXZlIGVycm9yIGlmIG5vdFxuICAgIHRlc3QuZXF1YWxzKENvbnRleHRQcm92aWRlci5nZXRLZXkoY29uc3RydWN0LCBjb250ZXh0UHJvcHMpLmtleSwgY29udGV4dEtleSk7XG5cbiAgICAvLyBXSEVOXG4gICAgQ29udGV4dFByb3ZpZGVyLmdldFZhbHVlKGNvbnN0cnVjdCwge1xuICAgICAgLi4uY29udGV4dFByb3BzLFxuICAgICAgZHVtbXlWYWx1ZTogdW5kZWZpbmVkLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGVycm9yID0gY29uc3RydWN0Lm5vZGUubWV0YWRhdGEuZmluZChtID0+IG0udHlwZSA9PT0gJ2F3czpjZGs6ZXJyb3InKTtcbiAgICB0ZXN0LmVxdWFscyhlcnJvciAmJiBlcnJvci5kYXRhLCAnSSBoYWQgYSBib28tYm9vJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG5cbi8qKlxuICogR2V0IHRoZSBleHBlY3RlZCBjb250ZXh0IGtleSBmcm9tIGEgc3RhY2sgd2l0aCBtaXNzaW5nIHBhcmFtZXRlcnNcbiAqL1xuZnVuY3Rpb24gZXhwZWN0ZWRDb250ZXh0S2V5KHN0YWNrOiBTdGFjayk6IHN0cmluZyB7XG4gIGNvbnN0IG1pc3NpbmcgPSBzeW50aGVzaXplKHN0YWNrKS5tYW5pZmVzdC5taXNzaW5nO1xuICBpZiAoIW1pc3NpbmcgfHwgbWlzc2luZy5sZW5ndGggIT09IDEpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0V4cGVjdGluZyBhc3NlbWJseSB0byBpbmNsdWRlIGEgc2luZ2xlIG1pc3NpbmcgY29udGV4dCByZXBvcnQnKTtcbiAgfVxuICByZXR1cm4gbWlzc2luZ1swXS5rZXk7XG59XG4iXX0=