"use strict";
const lib_1 = require("../lib");
const util_1 = require("./util");
const template = {
    Parameters: {
        MyParam: {
            Type: 'String',
            Default: 'Hello',
        },
    },
    Resources: {
        MyResource1: {
            Type: 'ResourceType1',
            Properties: {
                P1: 1,
                P2: 2,
            },
        },
        MyResource2: {
            Type: 'ResourceType2',
        },
    },
};
/**
 * @param obj an object to clone
 * @returns a deep clone of ``obj`.
 */
function clone(obj) {
    switch (typeof obj) {
        case 'object':
            if (Array.isArray(obj)) {
                return obj.map(elt => clone(elt));
            }
            else {
                const cloned = {};
                for (const key of Object.keys(obj)) {
                    cloned[key] = clone(obj[key]);
                }
                return cloned;
            }
        default:
            return obj;
    }
}
module.exports = {
    'the Include construct can be used to embed an existing template as-is into a stack'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', { template: clone(template) });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Parameters: { MyParam: { Type: 'String', Default: 'Hello' } },
            Resources: {
                MyResource1: { Type: 'ResourceType1', Properties: { P1: 1, P2: 2 } },
                MyResource2: { Type: 'ResourceType2' },
            },
        });
        test.done();
    },
    'included templates can co-exist with elements created programmatically'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', { template: clone(template) });
        new lib_1.CfnResource(stack, 'MyResource3', { type: 'ResourceType3', properties: { P3: 'Hello' } });
        new lib_1.CfnOutput(stack, 'MyOutput', { description: 'Out!', value: 'hey' });
        new lib_1.CfnParameter(stack, 'MyParam2', { type: 'Integer' });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Parameters: {
                MyParam: { Type: 'String', Default: 'Hello' },
                MyParam2: { Type: 'Integer' },
            },
            Resources: {
                MyResource1: { Type: 'ResourceType1', Properties: { P1: 1, P2: 2 } },
                MyResource2: { Type: 'ResourceType2' },
                MyResource3: { Type: 'ResourceType3', Properties: { P3: 'Hello' } },
            },
            Outputs: { MyOutput: { Description: 'Out!', Value: 'hey' } },
        });
        test.done();
    },
    'exception is thrown in construction if an entity from an included template has the same id as a programmatic entity'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', { template });
        new lib_1.CfnResource(stack, 'MyResource3', { type: 'ResourceType3', properties: { P3: 'Hello' } });
        new lib_1.CfnOutput(stack, 'MyOutput', { description: 'Out!', value: 'in' });
        new lib_1.CfnParameter(stack, 'MyParam', { type: 'Integer' }); // duplicate!
        test.throws(() => util_1.toCloudFormation(stack));
        test.done();
    },
    'correctly merges template sections that contain strings'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', {
            template: {
                AWSTemplateFormatVersion: '2010-09-09',
                Description: 'Test 1',
            },
        });
        new lib_1.CfnInclude(stack, 'T2', {
            template: {
                AWSTemplateFormatVersion: '2010-09-09',
                Description: 'Test 2',
            },
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            AWSTemplateFormatVersion: '2010-09-09',
            Description: 'Test 1\nTest 2',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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