"""Friction"""

from __future__ import annotations

from typing import TYPE_CHECKING, ClassVar

from mastapy._private._internal.dataclasses import extended_dataclass
from mastapy._private._internal.python_net import (
    pythonnet_property_get,
    python_net_import,
)
from mastapy._private._internal import constructor, utility
from mastapy._private.bearings.bearing_results.rolling.skf_module import _2150
from mastapy._private._internal.cast_exception import CastException

_FRICTION = python_net_import(
    "SMT.MastaAPI.Bearings.BearingResults.Rolling.SkfModule", "Friction"
)

if TYPE_CHECKING:
    from typing import Any, Type, TypeVar

    from mastapy._private.bearings.bearing_results.rolling.skf_module import (
        _2139,
        _2138,
    )

    Self = TypeVar("Self", bound="Friction")
    CastSelf = TypeVar("CastSelf", bound="Friction._Cast_Friction")


__docformat__ = "restructuredtext en"
__all__ = ("Friction",)


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class _Cast_Friction:
    """Special nested class for casting Friction to subclasses."""

    __parent__: "Friction"

    @property
    def skf_calculation_result(self: "CastSelf") -> "_2150.SKFCalculationResult":
        return self.__parent__._cast(_2150.SKFCalculationResult)

    @property
    def friction(self: "CastSelf") -> "Friction":
        return self.__parent__

    def __getattr__(self: "CastSelf", name: str) -> "Any":
        try:
            return self.__getattribute__(name)
        except AttributeError:
            class_name = utility.camel(name)
            raise CastException(
                f'Detected an invalid cast. Cannot cast to type "{class_name}"'
            ) from None


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class Friction(_2150.SKFCalculationResult):
    """Friction

    This is a mastapy class.
    """

    TYPE: ClassVar["Type"] = _FRICTION

    wrapped: "Any"

    def __post_init__(self: "Self") -> None:
        """Override of the post initialisation magic method."""
        if not hasattr(self.wrapped, "reference_count"):
            self.wrapped.reference_count = 0

        self.wrapped.reference_count += 1

    @property
    def power_loss(self: "Self") -> "float":
        """float

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "PowerLoss")

        if temp is None:
            return 0.0

        return temp

    @property
    def friction_sources(self: "Self") -> "_2139.FrictionSources":
        """mastapy.bearings.bearing_results.rolling.skf_module.FrictionSources

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "FrictionSources")

        if temp is None:
            return None

        type_ = temp.GetType()
        return constructor.new(type_.Namespace, type_.Name)(temp)

    @property
    def frictional_moment(self: "Self") -> "_2138.FrictionalMoment":
        """mastapy.bearings.bearing_results.rolling.skf_module.FrictionalMoment

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "FrictionalMoment")

        if temp is None:
            return None

        type_ = temp.GetType()
        return constructor.new(type_.Namespace, type_.Name)(temp)

    @property
    def cast_to(self: "Self") -> "_Cast_Friction":
        """Cast to another type.

        Returns:
            _Cast_Friction
        """
        return _Cast_Friction(self)
