"""TimeUnit"""

from __future__ import annotations

from typing import TYPE_CHECKING, ClassVar

from mastapy._private._internal.dataclasses import extended_dataclass
from mastapy._private._internal.python_net import python_net_import
from mastapy._private.utility.units_and_measurements import _1660
from mastapy._private._internal.cast_exception import CastException
from mastapy._private._internal import utility

_TIME_UNIT = python_net_import("SMT.MastaAPI.Utility.UnitsAndMeasurements", "TimeUnit")

if TYPE_CHECKING:
    from typing import Any, Type, TypeVar

    Self = TypeVar("Self", bound="TimeUnit")
    CastSelf = TypeVar("CastSelf", bound="TimeUnit._Cast_TimeUnit")


__docformat__ = "restructuredtext en"
__all__ = ("TimeUnit",)


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class _Cast_TimeUnit:
    """Special nested class for casting TimeUnit to subclasses."""

    __parent__: "TimeUnit"

    @property
    def unit(self: "CastSelf") -> "_1660.Unit":
        return self.__parent__._cast(_1660.Unit)

    @property
    def time_unit(self: "CastSelf") -> "TimeUnit":
        return self.__parent__

    def __getattr__(self: "CastSelf", name: str) -> "Any":
        try:
            return self.__getattribute__(name)
        except AttributeError:
            class_name = utility.camel(name)
            raise CastException(
                f'Detected an invalid cast. Cannot cast to type "{class_name}"'
            ) from None


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class TimeUnit(_1660.Unit):
    """TimeUnit

    This is a mastapy class.
    """

    TYPE: ClassVar["Type"] = _TIME_UNIT

    wrapped: "Any"

    def __post_init__(self: "Self") -> None:
        """Override of the post initialisation magic method."""
        if not hasattr(self.wrapped, "reference_count"):
            self.wrapped.reference_count = 0

        self.wrapped.reference_count += 1

    @property
    def cast_to(self: "Self") -> "_Cast_TimeUnit":
        """Cast to another type.

        Returns:
            _Cast_TimeUnit
        """
        return _Cast_TimeUnit(self)
