"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const api = require("aws-cdk-lib/aws-apigateway");
const s3 = require("aws-cdk-lib/aws-s3");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    return new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        lambdaFunctionProps
    });
}
function useExistingFunc(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    return new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        existingLambdaObj: new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps)
    });
}
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.cloudFrontWebDistribution !== null);
    expect(construct.apiGateway !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.cloudFrontFunction !== null);
    expect(construct.cloudFrontLoggingBucket !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
});
test('check lambda function properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testcloudfrontapigatewaylambdaLambdaFunctionServiceRoleCB74590F",
                "Arn"
            ]
        },
        Runtime: "nodejs14.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1"
            }
        }
    });
});
test('check lambda function role for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        ManagedPolicyArns: [
            {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                    ]
                ]
            }
        ]
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check no prop', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('override api gateway properties with existingLambdaObj', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    const fn = new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps);
    new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        existingLambdaObj: fn,
        apiGatewayProps: {
            description: "Override description"
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Description: "Override description",
        EndpointConfiguration: {
            Types: [
                "REGIONAL"
            ]
        },
        Name: "LambdaRestApi"
    });
});
test('override api gateway properties without existingLambdaObj', () => {
    const stack = new cdk.Stack();
    new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.PRIVATE],
            },
            description: "Override description"
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Description: "Override description",
        EndpointConfiguration: {
            Types: [
                "PRIVATE"
            ]
        },
        Name: "LambdaRestApi"
    });
});
// --------------------------------------------------------------
// Cloudfront logging bucket with destroy removal policy and auto delete objects
// --------------------------------------------------------------
test('Cloudfront logging bucket with destroy removal policy and auto delete objects', () => {
    const stack = new cdk.Stack();
    new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.PRIVATE],
            }
        },
        cloudFrontLoggingBucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "testcloudfrontapigatewaylambdaCloudFrontToApiGatewayCloudfrontLoggingBucket7F467421"
        }
    });
});
// --------------------------------------------------------------
// Cloudfront logging bucket error providing existing log bucket and logBucketProps
// --------------------------------------------------------------
test('Cloudfront logging bucket error when providing existing log bucket and logBucketProps', () => {
    const stack = new cdk.Stack();
    const logBucket = new s3.Bucket(stack, 'cloudfront-log-bucket', {});
    const app = () => {
        new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cloudfront-s3', {
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: 'index.handler'
            },
            apiGatewayProps: {
                endpointConfiguration: {
                    types: [api.EndpointType.PRIVATE],
                }
            },
            cloudFrontLoggingBucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                autoDeleteObjects: true
            },
            cloudFrontDistributionProps: {
                logBucket
            },
        });
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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