"""
Comprehensive tests for ssl_cert parameter in Connection class.

Tests cover:
- Different ssl_cert input types (None, str, bytes)
- Integration with secure channel creation
- Error handling for invalid certificates
- Connection with and without SSL
- Edge cases and security scenarios
"""

import unittest
import os
import tempfile
from unittest.mock import Mock, patch, MagicMock, call
import grpc

from e6data_python_connector import Connection
from e6data_python_connector.common import get_ssl_credentials


class TestSSLCertParameter(unittest.TestCase):
    """Test ssl_cert parameter behavior in Connection class."""

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_none_uses_default_ca_bundle(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert=None uses system default CA bundle."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            ssl_cert=None,  # Explicit None
            cluster_name='test-cluster'
        )

        # Verify secure_channel was called
        self.assertTrue(mock_secure_channel.called)

        # Get the credentials argument
        call_args = mock_secure_channel.call_args
        credentials = call_args[1]['credentials']

        # Verify credentials were created
        self.assertIsNotNone(credentials)

        conn.close()

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_file_path(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert can be provided as a file path."""
        # Create a temporary cert file
        cert_content = b"""-----BEGIN CERTIFICATE-----
MIIDXTCCAkWgAwIBAgIJAKL0UG+mRKqzMA0GCSqGSIb3DQEBCwUAMEUxCzAJBgNV
BAYTAkFVMRMwEQYDVQQIDApTb21lLVN0YXRlMSEwHwYDVQQKDBhJbnRlcm5ldCBX
aWRnaXRzIFB0eSBMdGQwHhcNMTgwNjE0MDQyNjMxWhcNMjgwNjExMDQyNjMxWjBF
MQswCQYDVQQGEwJBVTETMBEGA1UECAwKU29tZS1TdGF0ZTEhMB8GA1UECgwYSW50
-----END CERTIFICATE-----"""

        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='.pem') as f:
            f.write(cert_content)
            cert_path = f.name

        try:
            mock_client = Mock()
            mock_stub.return_value = mock_client

            mock_auth_response = Mock()
            mock_auth_response.sessionId = 'test-session'
            mock_auth_response.new_strategy = None
            mock_client.authenticate.return_value = mock_auth_response

            conn = Connection(
                host='test.example.com',
                port=443,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert=cert_path,  # File path
                cluster_name='test-cluster'
            )

            # Verify secure_channel was called
            self.assertTrue(mock_secure_channel.called)

            # Verify the certificate was read
            call_args = mock_secure_channel.call_args
            credentials = call_args[1]['credentials']
            self.assertIsNotNone(credentials)

            conn.close()
        finally:
            os.unlink(cert_path)

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_bytes_content(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert can be provided as bytes content."""
        cert_content = b"""-----BEGIN CERTIFICATE-----
MIIDXTCCAkWgAwIBAgIJAKL0UG+mRKqzMA0GCSqGSIb3DQEBCwUAMEUxCzAJBgNV
BAYTAkFVMRMwEQYDVQQIDApTb21lLVN0YXRlMSEwHwYDVQQKDBhJbnRlcm5ldCBX
aWRnaXRzIFB0eSBMdGQwHhcNMTgwNjE0MDQyNjMxWhcNMjgwNjExMDQyNjMxWjBF
-----END CERTIFICATE-----"""

        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            ssl_cert=cert_content,  # Bytes
            cluster_name='test-cluster'
        )

        # Verify secure_channel was called
        self.assertTrue(mock_secure_channel.called)

        # Verify credentials were created with bytes
        call_args = mock_secure_channel.call_args
        credentials = call_args[1]['credentials']
        self.assertIsNotNone(credentials)

        conn.close()

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_invalid_file_path_raises_error(self, mock_stub, mock_secure_channel):
        """Test that invalid ssl_cert file path raises FileNotFoundError."""
        with self.assertRaises(FileNotFoundError):
            conn = Connection(
                host='test.example.com',
                port=443,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert='/nonexistent/path/to/cert.pem',  # Invalid path
                cluster_name='test-cluster'
            )

    @patch('e6data_python_connector.e6data_grpc.grpc.insecure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_ignored_when_secure_false(self, mock_stub, mock_insecure_channel):
        """Test that ssl_cert is ignored when secure=False."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        # Create a valid cert file
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"
        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='.pem') as f:
            f.write(cert_content)
            cert_path = f.name

        try:
            conn = Connection(
                host='test.example.com',
                port=80,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                secure=False,  # Not secure
                ssl_cert=cert_path,  # SSL cert provided but should be ignored
                cluster_name='test-cluster'
            )

            # Verify insecure_channel was called, not secure_channel
            self.assertTrue(mock_insecure_channel.called)

            conn.close()
        finally:
            os.unlink(cert_path)

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_empty_string(self, mock_stub, mock_secure_channel):
        """Test that empty string ssl_cert is handled gracefully."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        # Empty string should be treated as invalid path
        with self.assertRaises((FileNotFoundError, OSError)):
            conn = Connection(
                host='test.example.com',
                port=443,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert='',  # Empty string
                cluster_name='test-cluster'
            )

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_default_when_not_provided(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert defaults to None when not provided."""
        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            # ssl_cert not provided - should default to None
            cluster_name='test-cluster'
        )

        # Should use default CA bundle
        self.assertTrue(mock_secure_channel.called)

        conn.close()

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_relative_path(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert works with relative file paths."""
        # Create a cert file in current directory
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"
        cert_filename = 'test_cert_temp.pem'

        with open(cert_filename, 'wb') as f:
            f.write(cert_content)

        try:
            mock_client = Mock()
            mock_stub.return_value = mock_client

            mock_auth_response = Mock()
            mock_auth_response.sessionId = 'test-session'
            mock_auth_response.new_strategy = None
            mock_client.authenticate.return_value = mock_auth_response

            conn = Connection(
                host='test.example.com',
                port=443,
                username='test@example.com',
                password='test_token',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert=cert_filename,  # Relative path
                cluster_name='test-cluster'
            )

            self.assertTrue(mock_secure_channel.called)
            conn.close()
        finally:
            if os.path.exists(cert_filename):
                os.unlink(cert_filename)

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_large_certificate(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert works with large certificate content."""
        # Create a large certificate (realistic size)
        cert_content = b"""-----BEGIN CERTIFICATE-----
MIIFazCCA1OgAwIBAgIRAIIQz7DSQONZRGPgu2OCiwAwDQYJKoZIhvcNAQELBQAw
TzELMAkGA1UEBhMCVVMxKTAnBgNVBAoTIEludGVybmV0IFNlY3VyaXR5IFJlc2Vh
cmNoIEdyb3VwMRUwEwYDVQQDEwxJU1JHIFJvb3QgWDEwHhcNMTUwNjA0MTEwNDM4
WhcNMzUwNjA0MTEwNDM4WjBPMQswCQYDVQQGEwJVUzEpMCcGA1UEChMgSW50ZXJu
ZXQgU2VjdXJpdHkgUmVzZWFyY2ggR3JvdXAxFTATBgNVBAMTDElTUkcgUm9vdCBY
MTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAK3oJHP0FDfzm54rVygc
h77ct984kIxuPOZXoHj3dcKi/vVqbvYATyjb3miGbESTtrFj/RQSa78f0uoxmyF+
0TM8ukj13Xnfs7j/EvEhmkvBioZxaUpmZmyPfjxwv60pIgbz5MDmgK7iS4+3mX6U
A5/TR5d8mUgjU+g4rk8Kb4Mu0UlXjIB0ttov0DiNewNwIRt18jA8+o+u3dpjq+sW
T8KOEUt+zwvo/7V3LvSye0rgTBIlDHCNAymg4VMk7BPZ7hm/ELNKjD+Jo2FR3qyH
B5T0Y3HsLuJvW5iB4YlcNHlsdu87kGJ55tukmi8mxdAQ4Q7e2RCOFvu396j3x+UC
B5iPNgiV5+I3lg02dZ77DnKxHZu8A/lJBdiB3QW0KtZB6awBdpUKD9jf1b0SHzUv
KBds0pjBqAlkd25HN7rOrFleaJ1/ctaJxQZBKT5ZPt0m9STJEadao0xAH0ahmbWn
OlFuhjuefXKnEgV4We0+UXgVCwOPjdAvBbI+e0ocS3MFEvzG6uBQE3xDk3SzynTn
jh8BCNAw1FtxNrQHusEwMFxIt4I7mKZ9YIqioymCzLq9gwQbooMDQaHWBfEbwrbw
qHyGO0aoSCqI3Haadr8faqU9GY/rOPNk3sgrDQoo//fb4hVC1CLQJ13hef4Y53CI
rU7m2Ys6xt0nUW7/vGT1M0NPAgMBAAGjQjBAMA4GA1UdDwEB/wQEAwIBBjAPBgNV
HRMBAf8EBTADAQH/MB0GA1UdDgQWBBR5tFnme7bl5AFzgAiIyBpY9umbbjANBgkq
hkiG9w0BAQsFAAOCAgEAVR9YqbyyqFDQDLHYGmkgJykIrGF1XIpu+ILlaS/V9lZL
ubhzEFnTIZd+50xx+7LSYK05qAvqFyFWhfFQDlnrzuBZ6brJFe+GnY+EgPbk6ZGQ
3BebYhtF8GaV0nxvwuo77x/Py9auJ/GpsMiu/X1+mvoiBOv/2X/qkSsisRcOj/KK
NFtY2PwByVS5uCbMiogziUwthDyC3+6WVwW6LLv3xLfHTjuCvjHIInNzktHCgKQ5
ORAzI4JMPJ+GslWYHb4phowim57iaztXOoJwTdwJx4nLCgdNbOhdjsnvzqvHu7Ur
TkXWStAmzOVyyghqpZXjFaH3pO3JLF+l+/+sKAIuvtd7u+Nxe5AW0wdeRlN8NwdC
jNPElpzVmbUq4JUagEiuTDkHzsxHpFKVK7q4+63SM1N95R1NbdWhscdCb+ZAJzVc
oyi3B43njTOQ5yOf+1CceWxG1bQVs5ZufpsMljq4Ui0/1lvh+wjChP4kqKOJ2qxq
4RgqsahDYVvTH9w7jXbyLeiNdd8XM2w9U/t7y0Ff/9yi0GE44Za4rF2LN9d11TPA
mRGunUHBcnWEvgJBQl9nJEiU0Zsnvgc/ubhPgXRR4Xq37Z0j4r7g1SgEEzwxA57d
emyPxgcYxn/eR44/KJ4EBs+lVDR3veyJm+kXQ99b21/+jh5Xos1AnX5iItreGCc=
-----END CERTIFICATE-----"""

        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            ssl_cert=cert_content,  # Large bytes content
            cluster_name='test-cluster'
        )

        self.assertTrue(mock_secure_channel.called)
        conn.close()

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_multiple_connections_with_different_ssl_certs(self, mock_stub, mock_secure_channel):
        """Test that multiple connections can use different SSL certificates."""
        # Create two different cert files
        cert1_content = b"-----BEGIN CERTIFICATE-----\ncert1\n-----END CERTIFICATE-----"
        cert2_content = b"-----BEGIN CERTIFICATE-----\ncert2\n-----END CERTIFICATE-----"

        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='_1.pem') as f1:
            f1.write(cert1_content)
            cert1_path = f1.name

        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='_2.pem') as f2:
            f2.write(cert2_content)
            cert2_path = f2.name

        try:
            mock_client = Mock()
            mock_stub.return_value = mock_client

            mock_auth_response = Mock()
            mock_auth_response.sessionId = 'test-session'
            mock_auth_response.new_strategy = None
            mock_client.authenticate.return_value = mock_auth_response

            # Create first connection with cert1
            conn1 = Connection(
                host='test1.example.com',
                port=443,
                username='test1@example.com',
                password='token1',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert=cert1_path,
                cluster_name='cluster1'
            )

            # Create second connection with cert2
            conn2 = Connection(
                host='test2.example.com',
                port=443,
                username='test2@example.com',
                password='token2',
                database='test_db',
                catalog='test_catalog',
                secure=True,
                ssl_cert=cert2_path,
                cluster_name='cluster2'
            )

            # Both should have created secure channels
            self.assertEqual(mock_secure_channel.call_count, 2)

            # Clean up
            conn1.close()
            conn2.close()
        finally:
            os.unlink(cert1_path)
            os.unlink(cert2_path)

    def test_get_ssl_credentials_utility_function(self):
        """Test the get_ssl_credentials utility function directly."""
        # Test with None
        creds_none = get_ssl_credentials(None)
        self.assertIsNotNone(creds_none)

        # Test with valid file
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"
        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='.pem') as f:
            f.write(cert_content)
            cert_path = f.name

        try:
            creds_file = get_ssl_credentials(cert_path)
            self.assertIsNotNone(creds_file)
        finally:
            os.unlink(cert_path)

        # Test with bytes
        creds_bytes = get_ssl_credentials(cert_content)
        self.assertIsNotNone(creds_bytes)

        # Test with invalid file path
        with self.assertRaises(FileNotFoundError):
            get_ssl_credentials('/nonexistent/cert.pem')


class TestSSLCertSecurityScenarios(unittest.TestCase):
    """Test security-related scenarios for ssl_cert parameter."""

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_pem_format_validation(self, mock_stub, mock_secure_channel):
        """Test that PEM format certificates are handled correctly."""
        # Valid PEM format
        valid_pem = b"""-----BEGIN CERTIFICATE-----
MIIBkTCB+wIJAKHHCgVZU9NEMA0GCSqGSIb3DQEBCwUAMBExDzANBgNVBAMMBnNl
cnZlcjAeFw0yMDA3MTUwOTU4MjhaFw0yMTA3MTUwOTU4MjhaMBExDzANBgNVBAMM
BnNlcnZlcjCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAwaRIlG6n6vnQq3YA
-----END CERTIFICATE-----"""

        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            ssl_cert=valid_pem,
            cluster_name='test-cluster'
        )

        self.assertTrue(mock_secure_channel.called)
        conn.close()

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_with_permission_error(self, mock_stub, mock_secure_channel):
        """Test handling of files with permission issues."""
        # Create a file with no read permissions
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"
        with tempfile.NamedTemporaryFile(mode='wb', delete=False, suffix='.pem') as f:
            f.write(cert_content)
            cert_path = f.name

        try:
            # Remove read permissions
            os.chmod(cert_path, 0o000)

            # Should raise IOError or PermissionError
            with self.assertRaises((IOError, PermissionError)):
                conn = Connection(
                    host='test.example.com',
                    port=443,
                    username='test@example.com',
                    password='test_token',
                    database='test_db',
                    catalog='test_catalog',
                    secure=True,
                    ssl_cert=cert_path,
                    cluster_name='test-cluster'
                )
        finally:
            # Restore permissions before deleting
            os.chmod(cert_path, 0o644)
            os.unlink(cert_path)

    @patch('e6data_python_connector.e6data_grpc.grpc.secure_channel')
    @patch('e6data_python_connector.e6data_grpc.e6x_engine_pb2_grpc.QueryEngineServiceStub')
    def test_ssl_cert_connection_property(self, mock_stub, mock_secure_channel):
        """Test that ssl_cert is stored as a connection property."""
        cert_content = b"-----BEGIN CERTIFICATE-----\ntest\n-----END CERTIFICATE-----"

        mock_client = Mock()
        mock_stub.return_value = mock_client

        mock_auth_response = Mock()
        mock_auth_response.sessionId = 'test-session'
        mock_auth_response.new_strategy = None
        mock_client.authenticate.return_value = mock_auth_response

        conn = Connection(
            host='test.example.com',
            port=443,
            username='test@example.com',
            password='test_token',
            database='test_db',
            catalog='test_catalog',
            secure=True,
            ssl_cert=cert_content,
            cluster_name='test-cluster'
        )

        # Verify ssl_cert is stored in connection
        self.assertEqual(conn._ssl_cert, cert_content)

        conn.close()


if __name__ == '__main__':
    unittest.main()
