"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MysqlDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const triggers_1 = require("aws-cdk-lib/triggers");
const base_database_1 = require("./base-database");
const HANDLER_PATH = path.join(__dirname, "../../assets/rds/triggers/mysql");
/**
 * A MySQL database.
 */
class MysqlDatabase extends base_database_1.BaseDatabase {
    /**
     * Creates a new MysqlDatabase.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param props - The configuration properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.ownerSecrets = [];
        this.readerSecrets = [];
        this.unprivilegedSecrets = [];
        const { adminSecret, vpc, characterSet = "utf8mb4" } = props;
        const environment = {
            AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
            ADMIN_SECRET_ARN: adminSecret.secretArn,
            OWNER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.ownerSecrets.map((s) => s.secretArn)),
            }),
            READER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.readerSecrets.map((s) => s.secretArn)),
            }),
            UNPRIVILEGED_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.unprivilegedSecrets.map((s) => s.secretArn)),
            }),
            DB_NAME: this.databaseName,
            DB_CHARACTER_SET: characterSet,
        };
        if (props.collation) {
            environment.DB_COLLATION = props.collation;
        }
        this.lambdaFunction = new aws_lambda_1.Function(this, "Function", {
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            handler: "index.handler",
            description: "Creates a schema and possibly some secondary users",
            vpc,
            vpcSubnets: this.subnetSelection,
            securityGroups: [this.securityGroup],
            environment,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        adminSecret.grantRead(this.lambdaFunction);
        this.trigger = new triggers_1.Trigger(this, "Trigger", {
            handler: this.lambdaFunction,
        });
        this.trigger.executeAfter(this.lambdaFunction);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forCluster(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            vpc: cluster.vpc,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forClusterFromSnapshot(scope, id, cluster, options) {
        // The DatabaseClusterFromSnapshot type is equivalent to DatabaseCluster.
        return MysqlDatabase.forCluster(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessCluster(scope, id, cluster, options) {
        // The ServerlessClusterFromSnapshot type is a subset of ServerlessCluster.
        return MysqlDatabase.forServerlessClusterFromSnapshot(scope, id, cluster, options);
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessClusterFromSnapshot(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstance.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstance(scope, id, instance, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || instance.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new MysqlDatabase(scope, id, {
            target: instance,
            endpoint: instance.instanceEndpoint,
            adminSecret: clusterSecret,
            vpc: instance.vpc,
            ...props,
        });
        database.trigger.executeAfter(instance);
        return database;
    }
    /**
     * Create a new MysqlDatabase inside a DatabaseInstanceFromSnapshot.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstanceFromSnapshot(scope, id, instance, options) {
        // The DatabaseInstanceFromSnapshot type is equivalent to DatabaseInstance.
        return MysqlDatabase.forInstance(scope, id, instance, options);
    }
    addUserAsOwner(secret) {
        this.ownerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsReader(secret) {
        this.readerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsUnprivileged(secret) {
        this.unprivilegedSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
}
exports.MysqlDatabase = MysqlDatabase;
_a = JSII_RTTI_SYMBOL_1;
MysqlDatabase[_a] = { fqn: "shady-island.MysqlDatabase", version: "0.0.57" };
//# sourceMappingURL=data:application/json;base64,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