"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration Test for private HTTP API with a existing function and ALB';
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, 'lambda-sg', { vpc: myVpc, allowAllOutbound: false });
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        vpc: myVpc,
        securityGroups: [testSg]
    }
});
const loadBalancer = defaults.ObtainAlb(stack, 'existing-alb', {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: false
};
const albToLambda = new lib_1.AlbToLambda(stack, 'test-one', props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.addCfnSuppressRules(testSg, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuYWxibGFtLXByaXZhdGVBcGlFeGlzdGluZ1Jlc291cmNlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImludGVnLmFsYmxhbS1wcml2YXRlQXBpRXhpc3RpbmdSZXNvdXJjZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILFVBQVU7QUFDViw2Q0FBNkQ7QUFDN0QsZ0NBQXVEO0FBQ3ZELHlEQUF3RTtBQUN4RSxpREFBaUQ7QUFDakQsMkRBQTJEO0FBQzNELGlEQUFzRTtBQUV0RSxpRkFBaUY7QUFDakYseUZBQXlGO0FBQ3pGLDJFQUEyRTtBQUUzRSxRQUFRO0FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxpQkFBRyxFQUFFLENBQUM7QUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxDQUFDLEdBQUcsRUFBRSw2QkFBc0IsQ0FBQyxVQUFVLENBQUMsRUFBRTtJQUMvRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsaUJBQUcsQ0FBQyxVQUFVLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtDQUN0RCxDQUFDLENBQUM7QUFDSCxLQUFLLENBQUMsZUFBZSxDQUFDLFdBQVcsR0FBRyx3RUFBd0UsQ0FBQztBQUU3RyxNQUFNLEtBQUssR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBRXpDLE1BQU0sTUFBTSxHQUFHLElBQUksdUJBQWEsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDO0FBRTdGLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUU7SUFDekQsbUJBQW1CLEVBQUU7UUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtRQUN4QixHQUFHLEVBQUUsS0FBSztRQUNWLGNBQWMsRUFBRSxDQUFFLE1BQU0sQ0FBRTtLQUMzQjtDQUNGLENBQUMsQ0FBQztBQUVILE1BQU0sWUFBWSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtJQUM3RCxHQUFHLEVBQUUsS0FBSztJQUNWLFNBQVMsRUFBRSxLQUFLO0lBQ2hCLGlCQUFpQixFQUFFO1FBQ2pCLGNBQWMsRUFBRSxLQUFLO1FBQ3JCLEdBQUcsRUFBRSxLQUFLO0tBQ1g7SUFDRCxhQUFhLEVBQUUsSUFBSTtJQUNuQixrQkFBa0IsRUFBRTtRQUNsQixhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO1FBQ3BDLGlCQUFpQixFQUFFLElBQUk7S0FDeEI7Q0FDRixDQUFDLENBQUM7QUFFSCxNQUFNLEtBQUssR0FBcUI7SUFDOUIsaUJBQWlCLEVBQUUsY0FBYztJQUNqQyx1QkFBdUIsRUFBRSxZQUFZO0lBQ3JDLFdBQVcsRUFBRSxLQUFLO0lBQ2xCLGFBQWEsRUFBRTtRQUNiLFFBQVEsRUFBRSxNQUFNO0tBQ2pCO0lBQ0QsU0FBUyxFQUFFLEtBQUs7Q0FDakIsQ0FBQztBQUVGLE1BQU0sV0FBVyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBRTlELFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFO0lBQ2pELEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsd0VBQXdFLEVBQUU7Q0FDaEcsQ0FBQyxDQUFDO0FBRUgsTUFBTSxnQkFBZ0IsR0FBRyxXQUFXLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQWdDLENBQUM7QUFDdEgsUUFBUSxDQUFDLG1CQUFtQixDQUFDLGdCQUFnQixFQUFFO0lBQzdDLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsMkNBQTJDLEVBQUM7SUFDakUsRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSw4QkFBOEIsRUFBQztJQUNuRCxFQUFFLEVBQUUsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLDhCQUE4QixFQUFDO0NBQ3BELENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLEVBQUU7SUFDbkMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSwyQ0FBMkMsRUFBQztDQUNsRSxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsaUNBQWlDLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDbEQsUUFBUTtBQUNSLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCB7IEF3cywgQXBwLCBTdGFjaywgUmVtb3ZhbFBvbGljeSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgQWxiVG9MYW1iZGEsIEFsYlRvTGFtYmRhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgeyBnZW5lcmF0ZUludGVnU3RhY2tOYW1lIH0gZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJ0Bhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZSc7XG5pbXBvcnQgeyBTZWN1cml0eUdyb3VwLCBDZm5TZWN1cml0eUdyb3VwIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1lYzJcIjtcblxuLy8gTm90ZTogQWxsIGludGVncmF0aW9uIHRlc3RzIGZvciBhbGIgYXJlIGZvciBIVFRQIEFQSXMsIGFzIGNlcnRpZmljYXRlcyByZXF1aXJlXG4vLyB2YWxpZGF0aW9uIHRocm91Z2ggRE5TIGFuZCBlbWFpbC4gVGhpcyB2YWxpZGF0aW9uIGlzIGltcG9zc2libGUgZHVyaW5nIG91ciBpbnRlZ3JhdGlvblxuLy8gdGVzdHMgYW5kIHRoZSBzdGFjayB3aWxsIGZhaWwgdG8gbGF1bmNoIHdpdGggYW4gdW52YWxpZGF0ZWQgY2VydGlmaWNhdGUuXG5cbi8vIFNldHVwXG5jb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5jb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsIGdlbmVyYXRlSW50ZWdTdGFja05hbWUoX19maWxlbmFtZSksIHtcbiAgZW52OiB7IGFjY291bnQ6IEF3cy5BQ0NPVU5UX0lELCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG59KTtcbnN0YWNrLnRlbXBsYXRlT3B0aW9ucy5kZXNjcmlwdGlvbiA9ICdJbnRlZ3JhdGlvbiBUZXN0IGZvciBwcml2YXRlIEhUVFAgQVBJIHdpdGggYSBleGlzdGluZyBmdW5jdGlvbiBhbmQgQUxCJztcblxuY29uc3QgbXlWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuY29uc3QgdGVzdFNnID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdsYW1iZGEtc2cnLCB7IHZwYzogbXlWcGMsIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlfSk7XG5cbmNvbnN0IGxhbWJkYUZ1bmN0aW9uID0gZGVmYXVsdHMuYnVpbGRMYW1iZGFGdW5jdGlvbihzdGFjaywge1xuICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIHZwYzogbXlWcGMsXG4gICAgc2VjdXJpdHlHcm91cHM6IFsgdGVzdFNnIF1cbiAgfVxufSk7XG5cbmNvbnN0IGxvYWRCYWxhbmNlciA9IGRlZmF1bHRzLk9idGFpbkFsYihzdGFjaywgJ2V4aXN0aW5nLWFsYicsIHtcbiAgdnBjOiBteVZwYyxcbiAgcHVibGljQXBpOiBmYWxzZSxcbiAgbG9hZEJhbGFuY2VyUHJvcHM6IHtcbiAgICBpbnRlcm5ldEZhY2luZzogZmFsc2UsXG4gICAgdnBjOiBteVZwYyxcbiAgfSxcbiAgbG9nQWNjZXNzTG9nczogdHJ1ZSxcbiAgbG9nZ2luZ0J1Y2tldFByb3BzOiB7XG4gICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIGF1dG9EZWxldGVPYmplY3RzOiB0cnVlXG4gIH1cbn0pO1xuXG5jb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgZXhpc3RpbmdMYW1iZGFPYmo6IGxhbWJkYUZ1bmN0aW9uLFxuICBleGlzdGluZ0xvYWRCYWxhbmNlck9iajogbG9hZEJhbGFuY2VyLFxuICBleGlzdGluZ1ZwYzogbXlWcGMsXG4gIGxpc3RlbmVyUHJvcHM6IHtcbiAgICBwcm90b2NvbDogJ0hUVFAnXG4gIH0sXG4gIHB1YmxpY0FwaTogZmFsc2Vcbn07XG5cbmNvbnN0IGFsYlRvTGFtYmRhID0gbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC1vbmUnLCBwcm9wcyk7XG5cbmRlZmF1bHRzLmFkZENmblN1cHByZXNzUnVsZXMoYWxiVG9MYW1iZGEubGlzdGVuZXIsIFtcbiAgeyBpZDogJ1c1NicsIHJlYXNvbjogJ0FsbCBpbnRlZ3JhdGlvbiB0ZXN0cyBtdXN0IGJlIEhUVFAgYmVjYXVzZSBvZiBjZXJ0aWZpY2F0ZSBsaW1pdGF0aW9ucy4nIH0sXG5dKTtcblxuY29uc3QgbmV3U2VjdXJpdHlHcm91cCA9IGFsYlRvTGFtYmRhLmxvYWRCYWxhbmNlci5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5TZWN1cml0eUdyb3VwO1xuZGVmYXVsdHMuYWRkQ2ZuU3VwcHJlc3NSdWxlcyhuZXdTZWN1cml0eUdyb3VwLCBbXG4gIHsgaWQ6ICdXMjknLCByZWFzb246ICdDREsgY3JlYXRlZCBydWxlIHRoYXQgYmxvY2tzIGFsbCB0cmFmZmljLid9LFxuICB7IGlkOiAnVzInLCByZWFzb246ICdSdWxlIGRvZXMgbm90IGFwcGx5IGZvciBFTEIuJ30sXG4gIHsgaWQ6ICdXOScsIHJlYXNvbjogJ1J1bGUgZG9lcyBub3QgYXBwbHkgZm9yIEVMQi4nfVxuXSk7XG5cbmRlZmF1bHRzLmFkZENmblN1cHByZXNzUnVsZXModGVzdFNnLCBbXG4gIHsgaWQ6ICdXMjknLCByZWFzb246ICdDREsgY3JlYXRlZCBydWxlIHRoYXQgYmxvY2tzIGFsbCB0cmFmZmljLid9LFxuXSk7XG5cbmRlZmF1bHRzLnN1cHByZXNzQXV0b0RlbGV0ZUhhbmRsZXJXYXJuaW5ncyhzdGFjayk7XG4vLyBTeW50aFxuYXBwLnN5bnRoKCk7XG4iXX0=