"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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