"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const defaults = require("@aws-solutions-constructs/core");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration test for alb with 2 Lambda targets';
const props = {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    },
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: true,
    albLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    }
};
const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
const secondProps = {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    },
    ruleProps: {
        conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
        priority: 10
    },
    existingVpc: firstConstruct.vpc,
    existingLoadBalancerObj: firstConstruct.loadBalancer,
    publicApi: true
};
new lib_1.AlbToLambda(stack, 'test-two', secondProps);
defaults.addCfnSuppressRules(firstConstruct.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = firstConstruct.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuYWxibGFtLXR3b1RhcmdldHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnRlZy5hbGJsYW0tdHdvVGFyZ2V0cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsVUFBVTtBQUNWLDZDQUE2RDtBQUM3RCxnQ0FBdUQ7QUFDdkQseURBQXdFO0FBQ3hFLGlEQUFpRDtBQUNqRCw4REFBOEQ7QUFDOUQsMkRBQTJEO0FBRzNELGlGQUFpRjtBQUNqRix5RkFBeUY7QUFDekYsMkVBQTJFO0FBRTNFLFFBQVE7QUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLGlCQUFHLEVBQUUsQ0FBQztBQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLENBQUMsR0FBRyxFQUFFLDZCQUFzQixDQUFDLFVBQVUsQ0FBQyxFQUFFO0lBQy9ELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxpQkFBRyxDQUFDLFVBQVUsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0NBQ3RELENBQUMsQ0FBQztBQUNILEtBQUssQ0FBQyxlQUFlLENBQUMsV0FBVyxHQUFHLGdEQUFnRCxDQUFDO0FBRXJGLE1BQU0sS0FBSyxHQUFxQjtJQUM5QixtQkFBbUIsRUFBRTtRQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCO0lBQ0QsYUFBYSxFQUFFO1FBQ2IsUUFBUSxFQUFFLE1BQU07S0FDakI7SUFDRCxTQUFTLEVBQUUsSUFBSTtJQUNmLHFCQUFxQixFQUFFO1FBQ3JCLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87UUFDcEMsaUJBQWlCLEVBQUUsSUFBSTtLQUN4QjtDQUNGLENBQUM7QUFDRixNQUFNLGNBQWMsR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUVqRSxNQUFNLFdBQVcsR0FBcUI7SUFDcEMsbUJBQW1CLEVBQUU7UUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QjtJQUNELFNBQVMsRUFBRTtRQUNULFVBQVUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQzdELFFBQVEsRUFBRSxFQUFFO0tBQ2I7SUFDRCxXQUFXLEVBQUUsY0FBYyxDQUFDLEdBQUc7SUFDL0IsdUJBQXVCLEVBQUUsY0FBYyxDQUFDLFlBQVk7SUFDcEQsU0FBUyxFQUFFLElBQUk7Q0FDaEIsQ0FBQztBQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0FBRWhELFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxjQUFjLENBQUMsUUFBUSxFQUFFO0lBQ3BELEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsd0VBQXdFLEVBQUU7Q0FDaEcsQ0FBQyxDQUFDO0FBRUgsTUFBTSxnQkFBZ0IsR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQWdDLENBQUM7QUFDekgsUUFBUSxDQUFDLG1CQUFtQixDQUFDLGdCQUFnQixFQUFFO0lBQzdDLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsMkNBQTJDLEVBQUM7SUFDakUsRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSw4QkFBOEIsRUFBQztJQUNuRCxFQUFFLEVBQUUsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLDhCQUE4QixFQUFDO0NBQ3BELENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxpQ0FBaUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNsRCxRQUFRO0FBQ1IsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0IHsgQXdzLCBBcHAsIFN0YWNrLCBSZW1vdmFsUG9saWN5IH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQgeyBBbGJUb0xhbWJkYSwgQWxiVG9MYW1iZGFQcm9wcyB9IGZyb20gXCIuLi9saWJcIjtcbmltcG9ydCB7IGdlbmVyYXRlSW50ZWdTdGFja05hbWUgfSBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgZWxiIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJ0Bhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZSc7XG5pbXBvcnQgeyBDZm5TZWN1cml0eUdyb3VwIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1lYzJcIjtcblxuLy8gTm90ZTogQWxsIGludGVncmF0aW9uIHRlc3RzIGZvciBhbGIgYXJlIGZvciBIVFRQIEFQSXMsIGFzIGNlcnRpZmljYXRlcyByZXF1aXJlXG4vLyB2YWxpZGF0aW9uIHRocm91Z2ggRE5TIGFuZCBlbWFpbC4gVGhpcyB2YWxpZGF0aW9uIGlzIGltcG9zc2libGUgZHVyaW5nIG91ciBpbnRlZ3JhdGlvblxuLy8gdGVzdHMgYW5kIHRoZSBzdGFjayB3aWxsIGZhaWwgdG8gbGF1bmNoIHdpdGggYW4gdW52YWxpZGF0ZWQgY2VydGlmaWNhdGUuXG5cbi8vIFNldHVwXG5jb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5jb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsIGdlbmVyYXRlSW50ZWdTdGFja05hbWUoX19maWxlbmFtZSksIHtcbiAgZW52OiB7IGFjY291bnQ6IEF3cy5BQ0NPVU5UX0lELCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG59KTtcbnN0YWNrLnRlbXBsYXRlT3B0aW9ucy5kZXNjcmlwdGlvbiA9ICdJbnRlZ3JhdGlvbiB0ZXN0IGZvciBhbGIgd2l0aCAyIExhbWJkYSB0YXJnZXRzJztcblxuY29uc3QgcHJvcHM6IEFsYlRvTGFtYmRhUHJvcHMgPSB7XG4gIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfSxcbiAgbGlzdGVuZXJQcm9wczoge1xuICAgIHByb3RvY29sOiAnSFRUUCdcbiAgfSxcbiAgcHVibGljQXBpOiB0cnVlLFxuICBhbGJMb2dnaW5nQnVja2V0UHJvcHM6IHtcbiAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgYXV0b0RlbGV0ZU9iamVjdHM6IHRydWVcbiAgfVxufTtcbmNvbnN0IGZpcnN0Q29uc3RydWN0ID0gbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC1vbmUnLCBwcm9wcyk7XG5cbmNvbnN0IHNlY29uZFByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH0sXG4gIHJ1bGVQcm9wczoge1xuICAgIGNvbmRpdGlvbnM6IFtlbGIuTGlzdGVuZXJDb25kaXRpb24ucGF0aFBhdHRlcm5zKFtcIiphZG1pbipcIl0pXSxcbiAgICBwcmlvcml0eTogMTBcbiAgfSxcbiAgZXhpc3RpbmdWcGM6IGZpcnN0Q29uc3RydWN0LnZwYyxcbiAgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmo6IGZpcnN0Q29uc3RydWN0LmxvYWRCYWxhbmNlcixcbiAgcHVibGljQXBpOiB0cnVlXG59O1xubmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC10d28nLCBzZWNvbmRQcm9wcyk7XG5cbmRlZmF1bHRzLmFkZENmblN1cHByZXNzUnVsZXMoZmlyc3RDb25zdHJ1Y3QubGlzdGVuZXIsIFtcbiAgeyBpZDogJ1c1NicsIHJlYXNvbjogJ0FsbCBpbnRlZ3JhdGlvbiB0ZXN0cyBtdXN0IGJlIEhUVFAgYmVjYXVzZSBvZiBjZXJ0aWZpY2F0ZSBsaW1pdGF0aW9ucy4nIH0sXG5dKTtcblxuY29uc3QgbmV3U2VjdXJpdHlHcm91cCA9IGZpcnN0Q29uc3RydWN0LmxvYWRCYWxhbmNlci5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5TZWN1cml0eUdyb3VwO1xuZGVmYXVsdHMuYWRkQ2ZuU3VwcHJlc3NSdWxlcyhuZXdTZWN1cml0eUdyb3VwLCBbXG4gIHsgaWQ6ICdXMjknLCByZWFzb246ICdDREsgY3JlYXRlZCBydWxlIHRoYXQgYmxvY2tzIGFsbCB0cmFmZmljLid9LFxuICB7IGlkOiAnVzInLCByZWFzb246ICdSdWxlIGRvZXMgbm90IGFwcGx5IGZvciBFTEIuJ30sXG4gIHsgaWQ6ICdXOScsIHJlYXNvbjogJ1J1bGUgZG9lcyBub3QgYXBwbHkgZm9yIEVMQi4nfVxuXSk7XG5cbmRlZmF1bHRzLnN1cHByZXNzQXV0b0RlbGV0ZUhhbmRsZXJXYXJuaW5ncyhzdGFjayk7XG4vLyBTeW50aFxuYXBwLnN5bnRoKCk7XG4iXX0=