#!/usr/bin/env python3
"""CLI dispatcher for skills.

This allows Big Pickle to interact with skills via bash:
    innerloop-skill list
    innerloop-skill describe data-viz
    innerloop-skill invoke data-viz

Skills are passed from the Loop instance via the INNERLOOP_SKILLS
environment variable (JSON serialized).
"""

import json
import os
import sys
from typing import Any


def get_skills_from_env() -> list[dict[str, Any]]:
    """Load skills from environment variable.

    Returns:
        List of skill definitions (as dicts)
    """
    skills_json = os.getenv("INNERLOOP_SKILLS", "[]")
    try:
        skills = json.loads(skills_json)
        if not isinstance(skills, list):
            print(
                "Error: INNERLOOP_SKILLS must be a JSON array", file=sys.stderr
            )
            return []
        return skills
    except json.JSONDecodeError as e:
        print(f"Error: Failed to parse INNERLOOP_SKILLS: {e}", file=sys.stderr)
        return []


def cmd_list() -> None:
    """List all available skills."""
    skills = get_skills_from_env()
    if not skills:
        print("No skills available")
        return

    print("Available skills:")
    for skill in skills:
        name = skill.get("name", "unknown")
        desc = skill.get("description_short", "")
        print(f"  - {name}: {desc}")


def cmd_describe(skill_name: str) -> None:
    """Show full description of a skill.

    Args:
        skill_name: Name of the skill to describe
    """
    skills = get_skills_from_env()
    skill = next((s for s in skills if s.get("name") == skill_name), None)

    if not skill:
        print(f"Error: Skill '{skill_name}' not found", file=sys.stderr)
        print("\nAvailable skills:", file=sys.stderr)
        for s in skills:
            print(f"  - {s.get('name', 'unknown')}", file=sys.stderr)
        sys.exit(1)

    # Build full description
    name = skill.get("name", "unknown")
    desc_short = skill.get("description_short", "")
    desc_long = skill.get("description_long", desc_short)
    python_tools = skill.get("python_tools", [])
    bash_patterns = skill.get("bash_patterns", {})
    allowed_ops = skill.get("allowed_operations", [])
    instructions = skill.get("instructions", "")
    examples = skill.get("examples", "")

    print(f"# Skill: {name}")
    print()
    print(f"**Description**: {desc_long}")
    print()

    if python_tools:
        print("**Python Tools**:")
        for tool_name in python_tools:
            print(f"- {tool_name}")
        print()

    if bash_patterns:
        print("**Bash Patterns**:")
        for pattern, permission in bash_patterns.items():
            print(f"- `{pattern}`: {permission}")
        print()

    if allowed_ops:
        print(f"**Allowed Operations**: {', '.join(allowed_ops)}")
        print()

    if instructions:
        print("**Instructions**:")
        print(instructions)
        print()

    if examples:
        print("**Examples**:")
        print(examples)
        print()


def cmd_invoke(skill_name: str) -> None:
    """Activate a skill.

    Args:
        skill_name: Name of the skill to activate
    """
    skills = get_skills_from_env()
    skill = next((s for s in skills if s.get("name") == skill_name), None)

    if not skill:
        print(f"Error: Skill '{skill_name}' not found", file=sys.stderr)
        print("\nAvailable skills:", file=sys.stderr)
        for s in skills:
            print(f"  - {s.get('name', 'unknown')}", file=sys.stderr)
        sys.exit(1)

    # Return activation confirmation
    print(f"✓ Skill '{skill_name}' activated")
    print()
    print("You can now use the following tools and capabilities:")

    python_tools = skill.get("python_tools", [])
    if python_tools:
        print("\nPython Tools:")
        for tool_name in python_tools:
            print(f"  - {tool_name}")

    bash_patterns = skill.get("bash_patterns", {})
    if bash_patterns:
        print("\nBash Patterns:")
        for pattern, permission in bash_patterns.items():
            print(f"  - `{pattern}`: {permission}")

    allowed_ops = skill.get("allowed_operations", [])
    if allowed_ops:
        print(f"\nAllowed Operations: {', '.join(allowed_ops)}")

    instructions = skill.get("instructions", "")
    if instructions:
        print("\nInstructions:")
        print(instructions)


def main() -> None:
    """CLI entry point for skill dispatcher."""
    if len(sys.argv) < 2:
        print("Usage: innerloop-skill <command> [args...]", file=sys.stderr)
        print("\nCommands:", file=sys.stderr)
        print("  list                    - List all skills", file=sys.stderr)
        print(
            "  describe <skill-name>   - Show full skill details",
            file=sys.stderr,
        )
        print("  invoke <skill-name>     - Activate a skill", file=sys.stderr)
        sys.exit(1)

    command = sys.argv[1]

    if command == "list":
        cmd_list()
    elif command == "describe":
        if len(sys.argv) < 3:
            print("Error: 'describe' requires a skill name", file=sys.stderr)
            print(
                "Usage: innerloop-skill describe <skill-name>", file=sys.stderr
            )
            sys.exit(1)
        cmd_describe(sys.argv[2])
    elif command == "invoke":
        if len(sys.argv) < 3:
            print("Error: 'invoke' requires a skill name", file=sys.stderr)
            print(
                "Usage: innerloop-skill invoke <skill-name>", file=sys.stderr
            )
            sys.exit(1)
        cmd_invoke(sys.argv[2])
    else:
        print(f"Error: Unknown command '{command}'", file=sys.stderr)
        print("\nAvailable commands: list, describe, invoke", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
