#!/usr/bin/env python3
"""CLI dispatcher for @tool decorated functions.

This allows Big Pickle to call tools via bash:
    innerloop-tool add a=5 b=3
    innerloop-tool analyze_data '{"filepath": "data.csv"}'

Results are printed to stdout, which Big Pickle sees through bash output.
"""

import json
import sys
from typing import Any

from pydantic import BaseModel, ValidationError


def parse_args(args: list[str]) -> dict[str, Any]:
    """Parse CLI arguments into kwargs.

    Supports:
    - key=value pairs: a=5 b=3
    - JSON object: '{"a": 5, "b": 3}'
    - Mixed: a=5 b='{"nested": true}'
    """
    if len(args) == 1 and args[0].startswith("{"):
        # Single JSON object
        try:
            result = json.loads(args[0])
            if not isinstance(result, dict):
                print(
                    f"Error: JSON must be an object, not {type(result).__name__}",
                    file=sys.stderr,
                )
                sys.exit(1)
            return result
        except json.JSONDecodeError as e:
            print(f"Error: Invalid JSON: {e}", file=sys.stderr)
            sys.exit(1)

    # Parse key=value pairs
    kwargs = {}
    for arg in args:
        if "=" not in arg:
            print(
                f"Error: Invalid argument '{arg}'. Use key=value format.",
                file=sys.stderr,
            )
            sys.exit(1)

        key, value = arg.split("=", 1)
        key = key.strip()

        # Try to parse value as JSON first (for objects/arrays)
        try:
            kwargs[key] = json.loads(value)
        except json.JSONDecodeError:
            # Not JSON, try as literal
            # Try int
            try:
                kwargs[key] = int(value)
                continue
            except ValueError:
                pass

            # Try float
            try:
                kwargs[key] = float(value)
                continue
            except ValueError:
                pass

            # Try bool
            if value.lower() in ("true", "false"):
                kwargs[key] = value.lower() == "true"
                continue

            # Keep as string
            kwargs[key] = value

    return kwargs


def convert_to_pydantic(
    kwargs: dict[str, Any], param_types: dict[str, type]
) -> dict[str, Any]:
    """Convert kwargs to Pydantic models if parameter types require it.

    Args:
        kwargs: Parsed arguments
        param_types: Parameter type hints from function signature

    Returns:
        Converted kwargs with Pydantic models instantiated
    """
    converted = {}

    for key, value in kwargs.items():
        param_type = param_types.get(key)

        if (
            param_type
            and isinstance(param_type, type)
            and issubclass(param_type, BaseModel)
        ):
            # Parameter expects a Pydantic model
            if isinstance(value, dict):
                try:
                    converted[key] = param_type(**value)
                except ValidationError as e:
                    print(
                        f"Error: Validation failed for {key}: {e}",
                        file=sys.stderr,
                    )
                    sys.exit(1)
            else:
                print(
                    f"Error: Expected object for {key}, got {type(value).__name__}",
                    file=sys.stderr,
                )
                sys.exit(1)
        else:
            converted[key] = value

    return converted


def format_output(result: Any) -> str:
    """Format function result for CLI output.

    Args:
        result: Function return value

    Returns:
        Formatted string for stdout
    """
    if isinstance(result, BaseModel):
        # Pydantic model - serialize to JSON
        return result.model_dump_json(indent=2)
    elif isinstance(result, (dict, list)):
        # Dict or list - serialize to JSON
        return json.dumps(result, indent=2)
    else:
        # Primitive type - convert to string
        return str(result)


def discover_tools() -> None:
    """Discover and import tool modules.

    Looks for:
    1. INNERLOOP_TOOLS_MODULE environment variable
    2. tools.py in current directory
    3. Current module if run as __main__
    """
    import importlib
    import os
    from pathlib import Path

    # Try environment variable first
    tools_module = os.getenv("INNERLOOP_TOOLS_MODULE")
    if tools_module:
        try:
            importlib.import_module(tools_module)
            return
        except ImportError as e:
            print(
                f"Warning: Failed to import {tools_module}: {e}",
                file=sys.stderr,
            )

    # Try tools.py in current directory
    tools_file = Path.cwd() / "tools.py"
    if tools_file.exists():
        try:
            import importlib.util

            spec = importlib.util.spec_from_file_location("tools", tools_file)
            if spec and spec.loader:
                tools = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(tools)
                return
        except Exception as e:
            print(
                f"Warning: Failed to load {tools_file}: {e}", file=sys.stderr
            )


def main() -> None:
    """CLI entry point for tool dispatcher."""
    if len(sys.argv) < 2:
        print(
            "Usage: innerloop-tool <function_name> [args...]", file=sys.stderr
        )
        print("\nExamples:", file=sys.stderr)
        print("  innerloop-tool add a=5 b=3", file=sys.stderr)
        print(
            "  innerloop-tool greet name='Alice' greeting='Hello'",
            file=sys.stderr,
        )
        print(
            '  innerloop-tool analyze_data \'{"filepath": "data.csv"}\'',
            file=sys.stderr,
        )
        print("\nTool Discovery:", file=sys.stderr)
        print(
            "  Set INNERLOOP_TOOLS_MODULE to specify a module to import",
            file=sys.stderr,
        )
        print("  Or create tools.py in the current directory", file=sys.stderr)
        sys.exit(1)

    tool_name = sys.argv[1]
    args = sys.argv[2:]

    # Import tools module to get registered tools
    try:
        from innerloop.tools import get_tool
    except ImportError as e:
        print(f"Error: Failed to import innerloop.tools: {e}", file=sys.stderr)
        sys.exit(1)

    # Discover and import tool definitions
    discover_tools()

    # Get the tool
    tool_def = get_tool(tool_name)
    if not tool_def:
        print(f"Error: Tool '{tool_name}' not found", file=sys.stderr)
        print("\nAvailable tools:", file=sys.stderr)
        from innerloop.tools import list_tools

        for name in list_tools():
            print(f"  - {name}", file=sys.stderr)
        sys.exit(1)

    # Parse arguments
    try:
        kwargs = parse_args(args)
    except Exception as e:
        print(f"Error: Failed to parse arguments: {e}", file=sys.stderr)
        sys.exit(1)

    # Handle single JSON argument for tools with single Pydantic parameter
    # If we got a JSON object and the tool expects a single parameter that's a Pydantic model,
    # wrap the JSON under the parameter name
    if len(args) == 1 and args[0].startswith("{"):
        params = list(tool_def.signature.parameters.keys())
        if len(params) == 1:
            param_name = params[0]
            param_type = tool_def.type_hints.get(param_name)
            if (
                param_type
                and isinstance(param_type, type)
                and issubclass(param_type, BaseModel)
            ):
                # Wrap the JSON under the parameter name
                kwargs = {param_name: kwargs}

    # Convert to Pydantic models if needed
    try:
        kwargs = convert_to_pydantic(kwargs, tool_def.type_hints)
    except Exception as e:
        print(f"Error: Failed to convert arguments: {e}", file=sys.stderr)
        sys.exit(1)

    # Call the tool
    try:
        result = tool_def.call(**kwargs)
    except TypeError as e:
        print(
            f"Error: Invalid arguments for {tool_name}: {e}", file=sys.stderr
        )
        print(
            f"\nExpected signature: {tool_def.to_prompt_schema()}",
            file=sys.stderr,
        )
        sys.exit(1)
    except Exception as e:
        print(f"Error: Tool execution failed: {e}", file=sys.stderr)
        sys.exit(1)

    # Format and print output
    try:
        output = format_output(result)
        print(output)
    except Exception as e:
        print(f"Error: Failed to format output: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
