"""Skills system for InnerLoop.

Skills package tools, permissions, and instructions into permission-scoped
workflows that Big Pickle can discover and use appropriately.

A Skill defines:
- Identity: name, descriptions (short and long)
- Capabilities: python_tools, bash_patterns, allowed_operations
- Guidance: instructions, examples

Skills are in-memory only - defined in calling code and passed to Loop.
No files, no discovery, pure runtime configuration.
"""

from __future__ import annotations

from pydantic import BaseModel, Field

from .permissions import PermissionLevel


class Skill(BaseModel):
    """A skill defines a permission-scoped workflow for Big Pickle.

    Skills group tools, bash permissions, and instructions into discoverable
    workflows. They provide progressive disclosure (short description always
    visible, full details on demand) and permission narrowing (skills can
    restrict but never expand Loop permissions).

    Attributes:
        name: Lowercase hyphenated identifier (e.g., "data-viz")
        description_short: One-line summary (always in prompt)
        description_long: Full instructions (on-demand via CLI)
        python_tools: List of @tool decorated function names
        bash_patterns: Dict mapping command patterns to permissions
        allowed_operations: Subset of ["read", "write", "bash"]
        instructions: Step-by-step workflow text
        examples: Concrete usage patterns

    Example:
        >>> from innerloop import Loop, allow, tool
        >>> from innerloop.skills import Skill
        >>>
        >>> @tool
        >>> def plot_data(filepath: str, chart_type: str) -> str:
        ...     return "chart.png"
        >>>
        >>> viz_skill = Skill(
        ...     name="data-viz",
        ...     description_short="Create charts from data files",
        ...     description_long="Reads CSV/JSON, generates matplotlib charts",
        ...     python_tools=["plot_data"],
        ...     bash_patterns={"python *.py": "allow"},
        ...     allowed_operations=["read", "write"],
        ...     instructions="1. Read data file\\n2. Call plot_data\\n3. Save output"
        ... )
        >>>
        >>> loop = Loop(
        ...     model="opencode/big-pickle",
        ...     perms=allow("bash"),
        ...     skills=[viz_skill]
        ... )
    """

    name: str = Field(
        ..., description="Lowercase hyphenated identifier (e.g., 'data-viz')"
    )
    description_short: str = Field(
        ..., description="One-line summary (always in prompt)"
    )
    description_long: str | None = Field(
        None, description="Full instructions (on-demand via CLI)"
    )
    python_tools: list[str] = Field(
        default_factory=list,
        description="List of @tool decorated function names",
    )
    bash_patterns: dict[str, PermissionLevel | str] = Field(
        default_factory=dict,
        description="Dict mapping command patterns to permissions (allow/deny)",
    )
    allowed_operations: list[str] = Field(
        default_factory=lambda: ["read"],
        description="Subset of ['read', 'write', 'bash']",
    )
    instructions: str | None = Field(
        None, description="Step-by-step workflow text"
    )
    examples: str | None = Field(None, description="Concrete usage patterns")

    def to_short_description(self) -> str:
        """Generate short description for prompt injection.

        Returns:
            Formatted short description for always-visible prompt
        """
        return f"**{self.name}**: {self.description_short}"

    def to_full_description(self) -> str:
        """Generate full description for on-demand disclosure.

        Returns:
            Complete skill documentation with all details
        """
        lines = [
            f"# Skill: {self.name}",
            "",
            f"**Description**: {self.description_long or self.description_short}",
            "",
        ]

        if self.python_tools:
            lines.append("**Python Tools**:")
            for tool_name in self.python_tools:
                lines.append(f"- {tool_name}")
            lines.append("")

        if self.bash_patterns:
            lines.append("**Bash Patterns**:")
            for pattern, permission in self.bash_patterns.items():
                lines.append(f"- `{pattern}`: {permission}")
            lines.append("")

        if self.allowed_operations:
            lines.append(
                f"**Allowed Operations**: {', '.join(self.allowed_operations)}"
            )
            lines.append("")

        if self.instructions:
            lines.append("**Instructions**:")
            lines.append(self.instructions)
            lines.append("")

        if self.examples:
            lines.append("**Examples**:")
            lines.append(self.examples)
            lines.append("")

        return "\n".join(lines)


__all__ = ["Skill"]
