#! /usr/bin/env python3

import os
import sys
import numpy as np
from scipy import linalg, stats
from scipy.spatial import ConvexHull
import random
import matplotlib
import matplotlib.pyplot as plt
import matplotlib.cm as cm
from matplotlib.widgets import Slider, Button, RadioButtons, TextBox, CheckButtons, Cursor, LassoSelector, RectangleSelector
from matplotlib.path import Path
import seaborn as sns
import nmrglue as ng
import lmfit as l
from datetime import datetime
import warnings

from . import fit, misc, sim, figures, processing
#from .__init__ import CM
from .config import CM
""" 
Collection of all-purpose functions
"""

def noise_std(y):
    """
    Calculates the standard deviation of the noise using the Bruker formula. Taken y as an array of N points, and yi its i-th entry:
    -------
    Parameters
    - y : 1darray
        The spectral region you would like to use to calculate the standard deviation of the noise.
    -------
    Returns
    - noisestd : float
        The standard deviation of the noise.
    """
    N = len(y)
    n = N//2 + 1
    # W
    W = 0
    for k in range(N):
        W += y[k]
    W = W**2
    # Y
    Y=0
    for k in range(N):
        Y += y[k]**2
    # X
    X=0
    for k in range(1,n):
        X += k * (y[-k] - y[k-1])
    noisestd = (N-1)**(-0.5) * np.sqrt( Y - 1/N * (W + 3 * X**2 / (N**2 -1) ))
    return noisestd


def SNR(data, signal=None, n_reg=None):
    """
    Computes the signal to noise ratio of a 1D spectrum.
    -------
    Parameters
    - data : 1darray
        The spectrum of which you want to compute the SNR
    - signal : float, optional
        If provided, uses this value as maximum signal. Otherwise, it is selected as the maximum value in "data"
    - n_reg : list or tuple, optional
        If provided, contains the points that delimit the noise region. Otherwise, the whole spectrum is used.
    -------
    Returns
    - snr : float
        The SNR of the spectrum
    """
    # Computes the SNR of a 1D spectrum (or 2D projection).
    # n_reg is a list/tuple of 2 values that delimitates the noise region
    if signal is None:
        signal = np.max(data)

    if n_reg is None:
        y = data
    else:
        y = data[min(n_reg[0], n_reg[1]):max(n_reg[0], n_reg[1])]
    snr = signal / (2 * misc.noise_std(y))
    return snr

def SNR_2D(data, n_reg=None):
    """
    Computes the signal to noise ratio of a 2D spectrum.
    -------
    Parameters
    - data : 1darray
        The spectrum of which you want to compute the SNR
    - n_reg : list or tuple
        If provided, the points of F1 scale and F2 scale, respectively, of which to extract the projections.
    Otherwise, opens the tool for interactive selection.
    --------
    Returns
    - snr_f1 : float
        The SNR of the indirect dimension
    - snr_f2 : float
        The SNR of the direct dimension
    """
    # Computes the SNR of a 2D spectrum.
    # n_reg is: (ppmf1 for f2 trace, ppmf2 for f1 trace)
    if n_reg is None:
        x_scale = np.arange(data.shape[-1])
        y_scale = np.arange(data.shape[0])
        coord = misc.select_traces(y_scale, x_scale, data)
        n_reg = (coord[0][0], coord[0][1])
        print('index for SNR (F1 | F2): ',n_reg)

    f1_trace = data[:,n_reg[0]]
    f2_trace = data[n_reg[1],:]

    snr_f1 = misc.SNR(f1_trace, signal=np.max(data))
    snr_f2 = misc.SNR(f2_trace, signal=np.max(data))

    return snr_f1, snr_f2

def makeacqus_1D(dic):
    """
    Given a NMRGLUE dictionary from a 1D spectrum (generated by ng.bruker.read), this function builds the acqus file with only the "important" parameters. 
    -------
    Parameters
    - dic: dict
        NMRglue dictionary returned by ng.bruker.read
    -------
    Returns
    - acqus : dict
        Dictionary with only few parameters
    """
    acqus = {}
    acqus['nuc'] = dic['acqus']['NUC1']
    acqus['SFO1'] = dic['acqus']['SFO1']
    acqus['SWp'] = dic['acqus']['SW']
    acqus['TD'] = int(dic['acqus']['TD'])//2    # Fuckin' Bruker
    acqus['o1'] = dic['acqus']['O1']

    acqus['B0'] = acqus['SFO1'] / sim.gamma[acqus['nuc']]
    acqus['o1p'] = acqus['o1'] / acqus['SFO1']
    acqus['SW'] = acqus['SWp'] * np.abs(acqus['SFO1'])
    acqus['dw'] = 1 / acqus['SW']
    acqus['t1'] = np.linspace(0, acqus['TD']*acqus['dw'], acqus['TD'])
    acqus['AQ'] = acqus['t1'][-1]
    return acqus

def makeacqus_2D(dic):
    """
    Given a NMRGLUE dictionary from a 2D spectrum (generated by ng.bruker.read ), this function builds the acqus file with only the "important" parameters.
    -------
    Parameters
    - dic: dict
        NMRglue dictionary returned by ng.bruker.read
    -------
    Returns
    - acqus : dict
        Dictionary with only few parameters
    """
    acqus = {}
    acqus['nuc1'] = dic['acqu2s']['NUC1']
    acqus['nuc2'] = dic['acqus']['NUC1']
    acqus['SFO1'] = dic['acqu2s']['SFO1']
    acqus['SFO2'] = dic['acqus']['SFO1']
    acqus['SW1p'] = dic['acqu2s']['SW']
    acqus['SW2p'] = dic['acqus']['SW']
    acqus['TD1'] = int(dic['acqu2s']['TD'])         # Indirect evolution is not /2
    acqus['TD2'] = int(dic['acqus']['TD'])//2       # Fuckin' Bruker
    acqus['o1'] = dic['acqu2s']['O1']
    acqus['o2'] = dic['acqus']['O1']

    acqus['B0'] = acqus['SFO2'] / sim.gamma[acqus['nuc2']]
    acqus['o1p'] = acqus['o1'] / acqus['SFO1']
    acqus['o2p'] = acqus['o2'] / acqus['SFO2']
    acqus['SW1'] = acqus['SW1p'] * np.abs(acqus['SFO1'])
    acqus['SW2'] = acqus['SW2p'] * np.abs(acqus['SFO2'])
    acqus['dw1'] = 1 / acqus['SW1']
    acqus['dw2'] = 1 / acqus['SW2']
    acqus['t1'] = np.linspace(0, acqus['TD1']*acqus['dw1'], acqus['TD1'])
    acqus['t2'] = np.linspace(0, acqus['TD2']*acqus['dw2'], acqus['TD2'])
    acqus['AQ1'] = acqus['t1'][-1]
    acqus['AQ2'] = acqus['t2'][-1]
    return acqus

def write_acqus_1D(acqus, path='sim_in_1D'):
    """
    Writes the input file for a simulated spectrum, basing on a dictionary of parameters.
    -------
    Parameters
    - acqus : dict
        The dictionary containing the parameters for the simulation
    - path : str, optional
        Directory where the file will be saved.
    """
    f = open(path, 'w')
    keylist = acqus.keys()
    for key in keylist:
        if key[:1] == 't':
            pass
        else:
            if isinstance(acqus[key], (list, tuple)):
                f.write('{}\t'.format(key))
                for w in acqus[key]:
                    f.write('{}, '.format(w))
                f.write('\n')
            else:
                f.write('{}\t{}\n'.format(key, acqus[key]))

        
    f.close()

def write_acqus_2D(acqus, path='sim_in_2D'):
    """
    Writes the input file for a simulated spectrum, basing on a dictionary of parameters.
    -------
    Parameters
    - acqus : dict
        The dictionary containing the parameters for the simulation
    - path : str, optional
        Directory where the file will be saved.
    """
    f = open(path, 'w')
    keylist = acqus.keys()
    for key in keylist:
        if key[:1] == 't':
            pass
        else:
            if isinstance(acqus[key], (list, tuple)):
                f.write('{}\t'.format(key))
                for w in acqus[key]:
                    f.write('{}, '.format(w))
                f.write('\n')
            else:
                f.write('{}\t{}\n'.format(key, acqus[key]))
    f.close()

def calcres(fqscale):
    """
    Calculates the frequency resolution of an axis scale, i.e. how many Hz is a "tick".
    --------
    Parameters
    - fqscale : 1darray
        Scale to be processed
    -------
	Returns
    --------
    - res: float
        The resolution of the scale
    """
    return np.abs(fqscale[1]-fqscale[0])

def hz2pt(fqscale, hz):
    """
    Converts hzfrom frequency units to points, on the basis of its scale.
    --------
    Parameters
    - fqscale : 1darray
        Scale to be processed
    - hz :float
        Value to be converted
    -------
	Returns
    - pt :float
        The frequency value converted in points
    """
    hzpt = misc.calcres(fqscale)
    pt = int(round(hz / hzpt))
    return pt


def find_nearest(array, value):
    """
    Finds the value in array which is the nearest to value .
    -------
	Parameters
    - array : 1darray
        Self-explanatory
    - value : float
        Value to be found
    -------
	Returns
    - val :float
        The closest value in array tovalue
    """
    # Finds the value in 'array' which is the nearest to 'value'
    array = np.asarray(array)
    idx = (np.abs(array - value)).argmin()
    return array[idx]

def trim_data(ppm_scale, y, sx, dx):
    """
    Trims the frequency scale and correspondant 1D dataset y from sx (ppm) to dx (ppm).
    -------
	Parameters
    - ppm_scale : 1darray
        ppm scale of the spectrum
    - y :1darray
        spectrum
    - sx :float
        ppm value where to start trimming
    - dx :float
        ppm value where to finish trimming
    -------
	Returns
    - xtrim : 1darray
        Trimmed ppm scale
    - ytrim : 1darray
        Trimmed spectrum
    """
    SX = misc.ppmfind(ppm_scale, sx)[0]
    DX = misc.ppmfind(ppm_scale, dx)[0]
    xtrim = ppm_scale[min(SX,DX):max(SX,DX)]
    if np.iscomplexobj(y):
        ytrim_re = y.real[...,min(SX,DX):max(SX,DX)]
        ytrim_im = y.imag[...,min(SX,DX):max(SX,DX)]
        ytrim = ytrim_re + 1j*ytrim_im
    else:
        ytrim = y[...,min(SX,DX):max(SX,DX)]
    return xtrim, ytrim

def ppmfind(ppm_scale, value):
    """
    Finds the exact value in ppm_scale.
    -------
	Parameters
    - ppm_scale : 1darray
        Self-explanatory
    - value : float
        The value to be found
    -------
	Returns
    - I :int
        The index correspondant to ’V’ in ’ppm_scale’
    - V :float
        The closest value to ’value’ in ’ppm_scale’
    """
    # Finds the exact 'value' in ppm scale 'ppm_1h'
    # Returns the found value 'V' and its index 'I'
    avgstep = np.abs((ppm_scale[0]-ppm_scale[1])/2)
    I, V = None, None
    for i, delta in enumerate(ppm_scale):
        if value-avgstep <= delta and delta < value+avgstep:
            I = i
            V = ppm_scale[i]
            break
        else:
            continue
    if I is None or V is None:
        raise ValueError('Value {} not found.'.format(value))
    else:
        return I, V

def ppm2freq(x, B0=701.125, o1p=0):
    """
    Converts xfrom ppm to Hz.
    -------
	Parameters
    - x :float
        Value to be converted
    - B0 :float
        Field frequency, in MHz. Default: 700 MHz
    - o1p : float
        Carrier frequency, in ppm. Default: 0.
    -------
	Returns
    - y :float
        The converted value
    """
    # Converts 'x' from ppm to Hz.
    # B0 is the frequency of the field in MHz.
    y = (x-o1p)*B0
    return y

def freq2ppm(x, B0=701.125, o1p=0):
    """
    Converts xfrom Hz to ppm.
    -------
	Parameters
    - x :float
        Value to be converted
    - B0 :float
        Field frequency, in MHz. Default: 700 MHz
    - o1p : float
        Carrier frequency, in ppm. Default: 0.
    -------
	Returns
    - y :float
        The converted value
    """
    y = x/B0 + o1p
    return y

def readlistfile(datafile):
    """
    Takes as input the path of a file containing one entry for each row. Returns a list of the aforementioned entries.
    -------
	Parameters
    - datafile: str
        Path to a file that contains one entry for each row
    -------
	Returns
    - files : list
        List of the entries contained in the file
    """
    with open(datafile) as F:
        names = F.readlines()

    files = []
    for i in range(len(names)):
        files.append(names[i].strip())

    return files

def procpar(txt):
    """
    Takes as input the path of a file containing a "key" in the first column and a "value" in the second
    column. Returns a dictionary of shape "key" : "value".
    -------
	Parameters
    - txt :str
        Path to a file that contains "key" in first column and "value" in the second
    -------
	Returns
    - procpars : dict
        Dictionary of shape "key":"value"
    """
    fyle = open(txt).readlines()
    procpars = {}
    for line in fyle:
        if line[0] == '#':
            continue    # Skip comments
        string = line.split('\t')
        procpars[string[0]] = float(string[1].strip())
    return procpars

def get_trace(data, ppm_f2, ppm_f1, a, b=None, column=True):
    """
    Takes as input a 2D dataset and the ppm scales of direct and indirect dimensions respectively.
    Calculates the projection on the given axis summing from a (ppm) to b(ppm). 
    Default: indirect dimension projection (i.e. column=True), change it to "False" for the direct dimension projection.
    Returns the calculated 1D projection.
    -------
	Parameters
    - data : 2darray
        Spectrum of which to extract the projections
    - ppm_f2 : 1darray
        ppm scale of the direct dimension
    - ppm_f1 : 1darray
        ppm scale of the indirect dimension
    - a :float
        The ppm value from which to start extracting the projection.
    - b :float, optional
        If provided, the ppm value at which to stop extracting the projection. Otherwise, returns only the 'a' trace.
    - column : bool
        If True, extracts the F1 projection. If False, extracts the F2 projection.
    -------
	Returns
    - y :1darray
        Computed projection
    """
    if not b:
        b = a

    if column:
        A = int(misc.ppmfind(ppm_f2, a)[0])
        B = int(misc.ppmfind(ppm_f2, b)[0])
        if A==B:
            y = data[:,A]
        else:
            y = np.sum(data[:, min(A,B):max(A,B)],axis=1)
    else:
        A = int(misc.ppmfind(ppm_f1, a)[0])
        B = int(misc.ppmfind(ppm_f1, b)[0])
        if A==B:
            y = data[A,:]
        else:
            y = np.sum(data[min(A,B):max(A,B), :],axis=0)
    return y


def select_traces(ppm_f1, ppm_f2, data, Neg=True, grid=False):
    """
    Select traces from a 2D spectrum, save the coordinates in a list. 
    Left click to select a point, right click to remove it.
    -------
	Parameters
    - ppm_f1 : 1darray
        ppm scale of the indirect dimension
    - ppm_f2 : 1darray
        ppm scale of the direct dimension
    - data : 2darray
        Spectrum
    - Neg : bool
        Choose if to show the negative contours ( True) or not ( False )
    - grid : bool
        Choose if to display the grid ( True) or not ( False )
    -------
	Returns
    - coord: list
        List containing the ’[x,y]’ coordinates of the selected points.
    """
    cmaps = CM['Blues_r'], CM['Reds_r']
    # Select traces from a 2D spectrum, save the coordinates in a list
    lvlstep = 0.02                  # for mouse scroll

    # Make the figure
    fig = plt.figure()
    fig.set_size_inches(figures.figsize_large)
    ax = fig.add_subplot(1,1,1)
    ax.set_title('Left double click to add point, right click to remove point')
    plt.subplots_adjust(left=0.1, bottom=0.1, right=0.95, top=0.90)

    # Set figure borders
    xsx = max(ppm_f2)
    xdx = min(ppm_f2)
    ysx = max(ppm_f1)
    ydx = min(ppm_f1)

    # set level for contour
    livello = 0.2
    cnt = figures.ax2D(ax, ppm_f2, ppm_f1, data, xlims=(xsx, xdx), ylims=(ysx, ydx), cmap=cmaps[0], c_fac=1.4, lvl=livello, lw=0.5, X_label='', Y_label='')
    if Neg:
        Ncnt = figures.ax2D(ax, ppm_f2, ppm_f1, -data, xlims=(xsx, xdx), ylims=(ysx, ydx), cmap=cmaps[0], c_fac=1.4, lvl=livello, lw=0.5)
    else: 
        Ncnt = None

    # Make pretty scales
    misc.pretty_scale(ax, (xsx, xdx), 'x')
    misc.pretty_scale(ax, (ysx, ydx), 'y')

    xgrid = ppm_f2
    ygrid = ppm_f1
    if grid:        # Set grid to visible
        for i in xgrid:
            ax.axvline(i, color='grey', lw=0.1)
        for j in ygrid:
            ax.axhline(j, color='grey', lw=0.1)

    # Parameters to save coordinates
    coord = []          # Final list of coordinates
    dot = []            # Bullets in figure
    dothline = []       # Horizontal lines
    dotvline = []       # Vertical lines

    def on_click(event):
        # What happens if you click?
        x, y = event.xdata, event.ydata     # x,y position of cursor
        if event.inaxes == ax:     # You are inside the figure
            ix, iy = misc.find_nearest(xgrid, x), misc.find_nearest(ygrid, y)       # Handle to the grid
            if str(event.button) == 'MouseButton.LEFT' and event.dblclick:     # Left click: add point
                if [ix, iy] not in coord:       # Avoid superimposed peaks
                    coord.append([ix,iy])       # Update list
                    # Update figure:
                    #   add bullet
                    line, = ax.plot(ix, iy, 'ro', markersize=2)
                    dot.append(line)
                    #   add horizontal line
                    dothline.append(ax.axhline(iy, c='r', lw=0.4))
                    #   add vertical line
                    dotvline.append(ax.axvline(ix, c='r', lw=0.4))
            if str(event.button) == 'MouseButton.RIGHT':    # Right click: remove point
                if [ix, iy] in coord:       # only if the point is already selected
                    # Remove coordinates and all figure elements
                    i = coord.index([ix, iy])
                    coord.remove([ix, iy])
                    killd = dot.pop(i)
                    killd.remove()
                    killh = dothline.pop(i)
                    killh.remove()
                    killv = dotvline.pop(i)
                    killv.remove()

        fig.canvas.draw()
    
    def on_scroll(event):
        # Zoom
        nonlocal livello, cnt
        if Neg:
            nonlocal Ncnt

        xsx, xdx = ax.get_xlim()
        ysx, ydx = ax.get_ylim()

        if event.button == 'up':
            livello += lvlstep 
        if event.button == 'down':
            livello += -lvlstep
        if livello < 1e-5:
            livello = 1e-5
        if livello > 1:
            livello = 1
        cnt, Ncnt = figures.redraw_contours(ax, ppm_f2, ppm_f1, data, lvl=livello, cnt=cnt, Neg=Neg, Ncnt=Ncnt, lw=0.5, cmap=cmaps)
        misc.pretty_scale(ax, (xsx, xdx), 'x')
        misc.pretty_scale(ax, (ysx, ydx), 'y')
        fig.canvas.draw()

    # Widgets
    cursor = Cursor(ax, useblit=True, color='red', linewidth=0.4)
    mouse = fig.canvas.mpl_connect('button_press_event', on_click)
    scroll = fig.canvas.mpl_connect('scroll_event', on_scroll)

    plt.show()
    plt.close()

    return coord

def select_for_integration(ppm_f1, ppm_f2, data, Neg=True):
    """
    Select the peaks of a 2D spectrum to integrate.
    First, select the area where your peak is located by dragging the red square.
    Then, select the center of the peak by right_clicking. 
    Finally, click 'ADD' to store the peak. Repeat the procedure for as many peaks as you want.
    -------
	Parameters
    - ppm_f1 : 1darray
        ppm scale of the indirect dimension
    - ppm_f2 : 1darray
        ppm scale of the direct dimension
    - data : 2darray
        Spectrum
    - Neg : bool
        Choose if to show the negative contours ( True) or not ( False )
    -------
	Returns
    - peaks: list of dict
        For each peak there are two keys, 'f1' and 'f2', whose meaning is obvious. 
        For each of these keys, you have 'u': center of the peak /ppm, and 'lim': the limits of the square you drew before.
    """

    cmaps = CM['Blues_r'], CM['Reds_r']
    lvlstep = 0.02                  # Increase step for contours when scroll the mouse1

    # Make an underlying grid to snap the pointer
    xgrid = np.copy(ppm_f2)
    ygrid = np.copy(ppm_f1)
    # Parameters to save coordinates
    coord = []          # Final list of coordinates
    rekt = []           # Rectangles
    # Set figure borders
    xsx, xdx = max(ppm_f2), min(ppm_f2)
    ysx, ydx = max(ppm_f1), min(ppm_f1)
    # set base level for contour
    lvl0 = 0.2



    
    # -----------------------------------------------------------------------------------------------------------------
    # Functions connected to the widgets
    def add_crosshair(coord, ix, iy):
        """ Add blue crosshair in (ix, iy) """
        if [ix, iy] not in coord:       # Avoid superimposed peaks
            coord.append([ix,iy])       # Update list
            ax.plot(ix, iy, 'bo', markersize=2) # add dot
            ax.axhline(iy, c='b', lw=0.4)   # add horizontal line
            ax.axvline(ix, c='b', lw=0.4)   # add vertical line
            for obj in (tmp_dot, tmp_hline, tmp_vline):
                obj.set_visible(False)      # Set the red crosshair invisible
        return coord

    def on_click(event):
        """ Right click moves the red crosshair """
        x, y = event.xdata, event.ydata     # x,y position of cursor
        if event.inaxes == ax: # You are inside the figure
            ix, iy = misc.find_nearest(xgrid, x), misc.find_nearest(ygrid, y)       # Snap to the grid
            if str(event.button) == 'MouseButton.RIGHT':    
                # Update figure:
                tmp_dot.set_data(ix, iy)
                tmp_hline.set_ydata(iy)
                tmp_vline.set_xdata(ix)
                # Make visible the red crosshair
                for obj in (tmp_dot, tmp_hline, tmp_vline):
                    obj.set_visible(True)
        else:
            pass
        fig.canvas.draw()
    
    def on_scroll(event):
        """ Redraw contours with more/less levels """
        nonlocal lvl0, cnt
        if Neg:
            nonlocal Ncnt

        # Read the input
        if event.button == 'up':
            lvl0 += lvlstep 
        if event.button == 'down':
            lvl0 += -lvlstep
        if lvl0 < 0:
            lvl0 = 1e-10
        if lvl0 > 1:
            lvl0 = 1

        # Redraw contours
        if Neg:
            cnt, Ncnt = figures.redraw_contours(ax, ppm_f2, ppm_f1, data, lvl=lvl0, cnt=cnt, Neg=Neg, Ncnt=Ncnt, lw=0.5, cmap=[cmaps[0], cmaps[1]])
        else:
            cnt, _ = figures.redraw_contours(ax, ppm_f2, ppm_f1, data, lvl=lvl0, cnt=cnt, Neg=Neg, Ncnt=None, lw=0.5, cmap=[cmaps[0], cmaps[1]])
        # Draw the pretty things again
        misc.pretty_scale(ax, (xsx, xdx), 'x')
        misc.pretty_scale(ax, (ysx, ydx), 'y')
        misc.set_fontsizes(ax, 14)
        fig.canvas.draw()

    def onselect(epress, erelease):
        """ Drag rectangle """
        if epress.button == 1: # left click
            # Vertices of the rectangle, counterclockwise
            X = np.array(span.extents[0:2])
            Y = np.array(span.extents[2:4])
            vertX = X[0], X[1], X[1], X[0]
            vertY = Y[0], Y[0], Y[1], Y[1]

            # Make visible the red rectangle
            if not tmp_rekt.get_visible():
                tmp_rekt.set_visible(True)
            tmp_rekt.set_xy(np.array((vertX, vertY)).T) # .T because (vertX, vertY).shape = (2, 4)
        else:
            pass
        fig.canvas.draw()

    def add_func(event):
        """ ADD button """
        nonlocal tmp_rekt, coord
        # Draw blue crosshair reading data from the red dot
        ix, iy = tmp_dot.get_data()
        coord = add_crosshair(coord, ix, iy)    # Update coord with the new peak

        # Draw blue rectangle reading data from the red rectangle
        verts = np.array(tmp_rekt.get_xy())[:-1]    # Skip the latter because it knows it has to close the perimeter
        dummy_rekt, = ax.fill(verts[:,0], verts[:,1], 'tab:blue', alpha=0.25)
        rekt.append(dummy_rekt)
        # Set red rectangle to invisible
        tmp_rekt.set_visible(False)
        fig.canvas.draw()

    # -----------------------------------------------------------------------------------------------------------------

    # Make the figure
    fig = plt.figure()
    fig.set_size_inches(figures.figsize_large)
    ax = fig.add_subplot(1,1,1)
    ax.set_title('Drag with left peak for region; select peak with right click')
    plt.subplots_adjust(left=0.1, bottom=0.1, right=0.875, top=0.90)
    
    # ADD button
    add_box = plt.axes([0.925, 0.70, 0.05, 0.05])
    add_button = Button(add_box, 'ADD', hovercolor='0.975')

    # Draw contour
    cnt = figures.ax2D(ax, ppm_f2, ppm_f1, data, cmap=cmaps[0], c_fac=1.4, lvl=lvl0, lw=0.5)
    if Neg:   
        Ncnt = figures.ax2D(ax, ppm_f2, ppm_f1, -data, cmap=cmaps[1], c_fac=1.4, lvl=lvl0, lw=0.5)

    # Initialize the red curves
    tmp_rekt, = ax.fill(np.array([0.1,0.2,0.3]), np.array([0.1,0.2,0.3]), 'tab:red', alpha=0.25, visible=False) # Rectangle
    tmp_dot, = ax.plot(0, 0, 'ro', markersize=2, visible=False) # Dot
    tmp_hline = ax.axhline(0, 0, c='r', lw=0.4, visible=False)  # Horizontal line
    tmp_vline = ax.axvline(0, 0, c='r', lw=0.4, visible=False)  # Vertical line

    # Pretty things
    misc.pretty_scale(ax, (xsx, xdx), 'x')
    misc.pretty_scale(ax, (ysx, ydx), 'y')
    misc.set_fontsizes(ax, 14)

    # Widgets
    cursor = Cursor(ax, useblit=True, color='red', linewidth=0.4)       # Moving crosshair
    mouse = fig.canvas.mpl_connect('button_press_event', on_click)      # Right click
    scroll = fig.canvas.mpl_connect('scroll_event', on_scroll)          # Mouse scroll
    span = RectangleSelector(ax, onselect, useblit=False, props=dict(facecolor='tab:red', alpha=0.5)) # Draggable rectangle
    add_button.on_clicked(add_func) # Button

    plt.show()
    plt.close()

    # -----------------------------------------------------------------------------------------------------------------

    # collect results
    peaks = [] 

    def calc_borders(rect):
        """ Calculate the limits of the rectangle """
        vert = rect.get_xy()
        vertX, vertY = vert[:,0], vert[:,1]
        x_lims = min(vertX), max(vertX)
        y_lims = min(vertY), max(vertY)
        return x_lims, y_lims

    for dot, rec in zip(coord, rekt):
        x_lims, y_lims = calc_borders(rec)
        # Create an entry for each peak as stated in the description
        peaks.append({
            'f1' : {
                'u' : dot[1],
                'lim' : y_lims,
                },
            'f2' : { 
                'u' : dot[0], 
                'lim' : x_lims,
                },
            })
    return peaks


def polyn(x, c):
    """
    Computes p(x), polynomion of degree n−1, where nis the number of provided coefficients.
    -------
	Parameters
    - x :1darray
        Scale upon which to build the polynomion
    - c :list or 1darray
        Sequence of the polynomion coeffiecient, starting from the 0-th order coefficient
    -------
	Returns
    - px :1darray
        Polynomion of degree n−1.
    """
    # Computes p(x) polynomion of degree n-1.
    # c is a list/array of the n coefficients, sorted starting 
    # from the 0th-order coefficient 
    degree = len(c)
    px = np.zeros_like(x)
    for i in range(degree):
        px += c[i] * x**i
    return px



def write_ser(fid, path='./', BYTORDA=0, DTYPA=0, overwrite=True):
    """
    Writes the FID file in directory 'path', in a TopSpin-readable way (i.e. little endian, int32).
    The binary file is named 'fid' if 1D, 'ser' if multiD.
    The parameters BYTORDA and DTYPA can be found in the acqus file.
    - BYTORDA = 1   =>  big endian      =>  '>'
    - BYTORDA = 0   =>  little endian   =>  '<'
    - DTYPA = 0     =>  int32           =>  'i4'
    - DTYPA = 2     =>  float64         =>  'f8'
    -------
	Parameters
    - fid :ndarray
        FID array to be written
    - path : str
        Directory where to save the file
    """

    if BYTORDA == 0:
        endian = '<' 
    elif BYTORDA == 1:
        endian = '>' 
    else:
        raise ValueError('Endianness not defined')

    if DTYPA == 0:
        dtype = 'i4'
    elif DTYPA == 2:
        dtype = 'f8'
    else:
        raise ValueError('Data type not defined')


    def uncomplexify_data(data_in):
        # Uncomplexify data (pack real,imag) into a int32 array
        size = list(data_in.shape)
        size[-1] = size[-1] * 2
        data_out = np.empty(size, dtype="int32")
        data_out[..., ::2] = data_in.real
        data_out[..., 1::2] = data_in.imag
        return data_out

    def open_towrite(filename):
        # Open filename for writing and return file object
        p, fn = os.path.split(filename)  # split into filename and path
        # create directories if needed
        if p != '' and os.path.exists(p) is False:
            os.makedirs(p)
        return open(filename, 'wb')

    if path[-1] != '/':
        path = path+'/'
    if len(fid.shape) == 1:
        filename = 'fid'
    else:
        filename = 'ser'

    if os.path.exists(path + filename):
        if overwrite is True:
            os.remove(path + filename)
        else:
            what_to_do = input('{} already exists. Overwrite it? [YES/no]'.format(filename+path))
            if what_to_do.lower()[0] == 'n':
                return 0
            else:
                os.remove(path + filename)
    f = open_towrite(path + filename)
    if np.iscomplexobj(fid):
        fid = uncomplexify_data(fid)
    print('Writing \'{}\' file in {}...'.format(filename, path))
    f.write(fid.astype(endian+dtype).tobytes())
    f.close()
    print('Done.')


def pretty_scale(ax, limits, axis='x', n_major_ticks=10):
    """
    This function computes a pretty scale for your plot. Calculates and sets a scale made of 'n_major_ticks' numbered ticks, spaced by 5*n_major_ticks unnumbered ticks. After that, the plot borders are trimmed according to the given limits.
    --------
    Parameters:
    - ax: matplotlib.AxesSubplot object
        Panel of the figure of which to calculate the scale
    - limits: tuple
        limits to apply of the given axis. (left, right)
    - axis: str
        'x' for x-axis, 'y' for y-axis
    - n_major_ticks: int
        Number of numbered ticks in the final scale. An oculated choice gives very pleasant results.
    """

    import matplotlib.ticker as TKR

    if axis=='x':
        ax.set_xlim(limits)
        sx, dx = ax.get_xlim()
    elif axis=='y':
        ax.set_ylim(limits)
        sx, dx = ax.get_ylim()
    else:
        raise ValueError('Unknown options for "axis".')

    # Compute major ticks
    steps = [1, 2, 4, 5, 10]
    majorlocs = TKR.MaxNLocator(nbins=n_major_ticks, steps=steps).tick_values(sx, dx)

    # Compute minor ticks manually because matplotlib is complicated
    ndivs = 5
    majorstep = majorlocs[1] - majorlocs[0]
    minorstep = majorstep / ndivs

    vmin, vmax = sx, dx
    if vmin > vmax:
        vmin, vmax = vmax, vmin

    t0 = majorlocs[0]
    tmin = ((vmin - t0) // minorstep + 1) * minorstep
    tmax = ((vmax - t0) // minorstep + 1) * minorstep
    minorlocs = np.arange(tmin, tmax, minorstep) + t0

    # Set the computed ticks and update the limits
    if axis == 'x':
        ax.set_xticks(majorlocs)
        ax.set_xticks(minorlocs, minor=True)
        ax.set_xlim(sx,dx)
    elif axis == 'y':
        ax.set_yticks(majorlocs)
        ax.set_yticks(minorlocs, minor=True)
        ax.set_ylim(sx,dx)


def molfrac(n):
    """
    Computes the "molar fraction" 'x' of the array 'n'.
    Returns also the total amount.
    -------
    Parameters:
    - n: list or 1darray
        list of values
    -------
    Returns:
    - x: list or 1darray
        molar fraction array
    - N: float
        sum of all the elements in 'n'
    """
    if isinstance(n, list):
        n = np.array(n)
    N = np.sum(n)
    x = [n[i]/N for i in range(len(n))]
    if isinstance(n, list):
        x = np.array(x)
    return x, N

def split_acqus_2D(acqus):
    """
    Split the acqus dictionary of a 2D spectrum into two separate 1D-like acqus dictionaries.
    --------
    Parameters:
    - acqus: dict
        acqus dictionary of a 2D spectrum
    --------
    Returns:
    - acqu1s: dict
        acqus dictionary of the indirect dimension
    - acqu2s: dict
        acqus dictionary of the direct dimension
    """
    keys = ['B0', 'nuc', 'o1p', 'SWp', 'TD', 'SFO1', 'SW', 'dw', 't1', 'o1', 'AQ1']
    acqu1v = [
            acqus['B0'],
            acqus['nuc1'],
            acqus['o1p'],
            acqus['SW1p'],
            acqus['TD1'],
            acqus['SFO1'],
            acqus['SW1'],
            acqus['dw1'],
            acqus['t1'],
            acqus['AQ1'],
            acqus['o1']]
    acqu2v = [
            acqus['B0'],
            acqus['nuc2'],
            acqus['o2p'],
            acqus['SW2p'],
            acqus['TD2'],
            acqus['SFO2'],
            acqus['SW2'],
            acqus['dw2'],
            acqus['t2'],
            acqus['AQ2'],
            acqus['o2']]
    acqu1s = {}
    acqu2s = {}
    for k, key in enumerate(keys):
        acqu1s[key] = acqu1v[k]
        acqu2s[key] = acqu2v[k]
    return acqu1s, acqu2s

def split_procs_2D(procs):
    """
    Split the procs dictionary of a 2D spectrum into two separate 1D-like procs dictionaries.
    --------
    Parameters:
    - procs: dict
        procs dictionary of a 2D spectrum
    --------
    Returns:
    - proc1s: dict
        procs dictionary of the indirect dimension
    - proc2s: dict
        procs dictionary of the direct dimension
    """
    keys = ['wf', 'zf', 'fcor', 'tdeff', 'p0', 'p1', 'pv']
    proc1v = [
            procs['wf'][0],
            procs['zf'][0],
            procs['fcor'][0],
            procs['tdeff'][0],
            procs['p0_1'],
            procs['p1_1'],
            procs['pv_1']]
    proc2v = [
            procs['wf'][1],
            procs['zf'][1],
            procs['fcor'][1],
            procs['tdeff'][1],
            procs['p0_2'],
            procs['p1_2'],
            procs['pv_2']]
    proc1s = {}
    proc2s = {}
    for k, key in enumerate(keys):
        proc1s[key] = proc1v[k]
        proc2s[key] = proc2v[k]
    return proc1s, proc2s

def nuc_format(nuc):
    """
    Converts the 'nuc' key you may find in acqus in the formatted label, e.g. '13C' -> '$^{13}$C'
    --------
    Parameters:
    - nuc: str
        Unformatted string
    --------
    Returns:
    - fnuc: str
        Formatted string.
    """
    import re
    fnuc = re.split('(\D+)',nuc)
    f_nuc = '$^{' + str(fnuc[0]) + '}$'+str(fnuc[1])
    return f_nuc

def set_ylim(ax, data_inp): 
    """ 
    Sets the y-limits of ax as follows: 
    Bottom:     min(data) - 5% max(data) 
    Top:        max(data) + 5% max(data) 
    ------- 
    Parameters: 
    - ax: matplotlib.Subplot Object 
        Panel of the figure where to apply this scale 
    - data_inp: ndarray or list 
        Input data. If it is a list, data_inp is converted to array. 
    """ 
    if isinstance(data_inp, list): 
        datain = np.concatenate(data_inp)
    else:
        datain = np.copy(data_inp)
 
    T = np.max(datain.real) 
    B = np.min(datain.real) 
    try:    # T and B can raise errors in certain situations 
        ax.set_ylim(B - 0.05*T, T + 0.05*T) 
    except: 
        pass

def get_ylim(data_inp): 
    """ 
    Calculates the y-limits of ax as follows: 
    Bottom:     min(data) - 5% max(data) 
    Top:        max(data) + 5% max(data) 
    ------- 
    Parameters: 
    - data_inp: ndarray or list 
        Input data. If it is a list, data_inp is converted to array. 
    """ 
    if isinstance(data_inp, list): 
        datain = np.concatenate(data_inp)
    else:
        datain = np.copy(data_inp)
 
    T = np.max(datain.real) 
    B = np.min(datain.real) 
    return B, T


def mathformat(ax, axis='y', limits=(-2,2)):
    """
    Apply exponential formatting to the given axis of the given figure panel. The offset text size is uniformed to the tick labels' size.
    -------
    Parameters:
    - ax: matplotlib.Subplot Object
        Panel of the figure to edit
    - axis: str
        'x', 'y' or 'both'.
    - limits: tuple
        tuple of ints that indicate the order of magnitude range outside which the exponential format is applied.
    """
    ax.ticklabel_format(axis=axis, style='scientific', scilimits=limits, useMathText=True)
    if axis=='y' or axis=='both':
        tmp = (ax.get_yticklabels())
        fontsize = tmp[0].get_fontsize()
        ax.yaxis.get_offset_text().set_size(fontsize)

    if axis=='x' or axis=='both':
        tmp = (ax.get_xticklabels())
        fontsize = tmp[0].get_fontsize()
        ax.xaxis.get_offset_text().set_size(fontsize)

        
def set_fontsizes(ax, fontsize=10):
    """
    Automatically adjusts the fontsizes of all the figure elements.
    In particular:
    - title = fontsize
    - axis labels = fontsize - 2
    - ticks labels = fontsize - 3
    - legend entries = fontsize - 4
    --------
    Parameters:
    - ax: matplotlib.Subplot Object
        Subplot of interest
    - fontsize: float
        Starting fontsize
    -------
    """

    # ---------------------------------------------------------------------
    def _modify_legend(ax, **kwargs):
        """
        Copied from StackOverflow: 
            https://stackoverflow.com/questions/23689728/how-to-modify-matplotlib-legend-after-it-has-been-created
        """

        l = ax.legend_
        defaults = dict(
            loc = l._loc,
            numpoints = l.numpoints,
            markerscale = l.markerscale,
            scatterpoints = l.scatterpoints,
            scatteryoffsets = l._scatteryoffsets,
            prop = l.prop,
            borderpad = l.borderpad,
            labelspacing = l.labelspacing,
            handlelength = l.handlelength,
            handleheight = l.handleheight,
            handletextpad = l.handletextpad,
            borderaxespad = l.borderaxespad,
            columnspacing = l.columnspacing,
            ncol = l._ncol,
            mode = l._mode,
            fancybox = type(l.legendPatch.get_boxstyle())==matplotlib.patches.BoxStyle.Round,
            shadow = l.shadow,
            title = l.get_title().get_text() if l._legend_title_box.get_visible() else None,
            framealpha = l.get_frame().get_alpha(),
            bbox_to_anchor = l.get_bbox_to_anchor()._bbox,
            bbox_transform = l.get_bbox_to_anchor()._transform,
            #frameon = l._drawFrame,
            frameon = l.draw_frame,
            handler_map = l._custom_handler_map,
        )

        if "fontsize" in kwargs and "prop" not in kwargs:
            defaults["prop"].set_size(kwargs["fontsize"])

        ax.legend(**dict(list(defaults.items()) + list(kwargs.items())))
    # ---------------------------------------------------------------------

    # Set the dimensions
    title_font = fontsize
    label_font = fontsize - 2
    ticks_font = fontsize - 3
    legen_font = fontsize - 4

    ax.title.set_fontsize(title_font)                   # title
    ax.xaxis.label.set_fontsize(label_font)             # xlabel
    ax.yaxis.label.set_fontsize(label_font)             # xlabel
    # Ticks
    for label in (ax.get_xticklabels() + ax.get_yticklabels()):
        label.set_fontsize(ticks_font)
    # Offset text
    ax.xaxis.get_offset_text().set_size(ticks_font)
    ax.yaxis.get_offset_text().set_size(ticks_font)

    # Legend
    if ax.legend_ is not None:
        _modify_legend(ax, fontsize=legen_font)


def hankel(data, n=None):
    """
    Computes a Hankel matrix from data.
    If data is a 1darray of length N, computes the correspondant Hankel matrix of dimensions (N-n+1, n).
    If data id a 2darray, computes the closest Hankel matrix in the Frobenius norm sense by averaging the values on the antidiagonals.
    -------
    Parameters:
    - data: 1darray
        Vector to be Hankel-ized, of length N
    - n: int
        Number of columns that the Hankel matrix will have
    Returns:
    - H: 2darray
        Hankel matrix of dimensions (N-n+1, n)
    """
    if isinstance(data, np.ndarray):
        if len(data.shape) == 1:
            if n is None:
                raise ValueError('You must specify the number of columns of the Hankel matrix.')
            H = linalg.hankel(data[:n], data[n-1:]).T
        elif len(data.shape) == 2:
            H = misc.avg_antidiag(data)
        else:
            raise ValueError('{}D arrays are not supported.'.format(len(data.shape)))
    else:
        raise ValueError('Input data is not an array.')
    return H

def unhankel(H):
    """
    Concatenates the first row and the last column of the matrix H, which should have Hankel-like structure, so to build the array of independent parameters.
    ------
    Parameters:
    - H: 2darray
        Hankel-like matrix
    ------
    Returns:
    - h: 1darray
        First row and last column, concatenated
    """
    h = np.concatenate((H[0,:], H[1:, -1]), axis=-1)
    return h


def avg_antidiag(X):
    """
    Given a matrix X without any specific structure, finds the closest Hankel matrix in the Frobenius norm sense by averaging the antidiagonals.
    ---------
    Parameters:
    - X: 2darray
        Input matrix
    --------
    Returns:
    - Xp: 2darray
        Hankel matrix obtained from X
    """
    m, n = X.shape  # Get dimensions of X
    N = m + n - 1   # Degrees of freedom that Xp will have
    data = np.array([np.mean(np.diag(X[:, ::-1], w)) for w in range(-N+n, n)])[::-1]      # Mean on the antidiagonals
    Xp = misc.hankel(data, n)    # Transform the "data" array into a matrix

    return Xp

def write_help(request, file=None):
    """
    Gets the documentation of request, and tries to save it in a text file.
    -------
    Parameters:
    - request: function or class or package
        Whatever you need documentation of
    - file: str or None or False
        Name of the output documentation file. If it is None, a default name is given. If it is False, the output is printed on screen.
    """
    import pydoc
    if file is None:
        file = request.__name__+'.hlp'
    hlp_text = pydoc.render_doc(request, renderer=pydoc.plaintext)
    if bool(file):
        with open(file, 'w') as F:
            F.write(hlp_text)
    else:
        print(hlp_text)

        
def print_list(mylist):
    """
    Prints a list, one entry per row.
    -------
    Parameters:
    - mylist: list
        The list you want to print
    -------
    Returns:
    - outstring: str
        The printed text formatted as single string
    """
    outstring = ''
    for entry in mylist:
        outstring += '{}\n'.format(entry)
    print(outstring)
    return outstring
        
def print_dict(mydict):
    """
    Prints a dictionary one entry per row, in the format key: value. Nested dictionaries are printed with an indentation
    -------
    Parameters:
    - mydict: dict
        The dictionary you want to print
    -------
    Returns:
    - outstring: str
        The printed text formatted as single string
    """
    outstring = ''
    for key, value in mydict.items():
        if isinstance(value, dict):
            outstring += '{}:\n'.format(key)
            for inner_key, inner_value in value.items():
                outstring += '\t{:10}: {}\n'.format(inner_key, inner_value)
        else:
            outstring += '{:10}: {}\n'.format(key, value)
    print(outstring)
    return outstring


def show_cmap(cmap, N=10, start=0, end=1):
    """ 
    Plot the colors extracted from a colormap.
    -------
    Parameters:
    - cmap: matplotlib.Colormap Object
        The colormap from which you want to extract the list of colors
    - N: int
        Number of samples to extract
    - start: float
        Start point of the sampling. 0 = beginning of the cmap; 1 = end of the cmap.
    - end: float
        End point of the sampling. 0 = beginning of the cmap; 1 = end of the cmap.
    """

    x = np.linspace(start, end, N)
    colors = cmap2list(cmap, N, start, end)

    # To fill the space
    width = (end - start) / (N - 1)

    # Make the figure
    fig = plt.figure()
    fig.set_size_inches(figures.figsize_large)
    fig.subplots_adjust(left=0.01, right=0.99)
    ax = fig.add_subplot(1,1,1)

    ax.set_title(cmap.name)

    # Draw the colors
    ax.bar(x, 1, width=width, bottom=None, align='center', data=None, color=colors)

    # Remove the white spaces
    ax.set_xlim(start-width/2, end+width/2)
    ax.set_ylim(0,1)
    ax.tick_params(axis='y', left=False, labelleft=False)

    plt.show()
    plt.close()


def cmap2list(cmap, N=10, start=0, end=1):
    """ 
    Extract the colors from a colormap and return it as a list.
    -------
    Parameters:
    - cmap: matplotlib.Colormap Object
        The colormap from which you want to extract the list of colors
    - N: int
        Number of samples to extract
    - start: float
        Start point of the sampling. 0 = beginning of the cmap; 1 = end of the cmap.
    - end: float
        End point of the sampling. 0 = beginning of the cmap; 1 = end of the cmap.
    -------
    Returns:
    - colors: list
        List of the extracted colors.
    """
    x = np.linspace(start, end, N)
    colors = cmap(x)
    return colors

def edit_checkboxes(checkbox, xadj=0, yadj=0, length=None, height=None, color=None):
    """ 
    Edit the size of the box to be checked, and adjust the lines accordingly.
    -----------
    Parameters:
    - checkbox: matplotlib.widgets.CheckBox Object
        The checkbox to edit
    - xadj: float
        modifier value for bottom left corner x-coordinate of the rectangle, in checkbox.ax coordinates
    - yadj: float
        modifier value for bottom left corner y-coordinate of the rectangle, in checkbox.ax coordinates
    - length: float
        length of the rectangle, in checkbox.ax coordinates
    - height: float
        height of the rectangle, in checkbox.ax coordinates
    - color: str or list or None
        If it is not None, change color to the lines
    """
    for rekt, lnz in zip(checkbox.rectangles, checkbox.lines):
        orig = rekt.get_xy()
        x, y = orig[0]+xadj, orig[1]+yadj
        rekt.set_xy((x, y))
        if length is not None:
            rekt.set_width(length)
        if height is not None:
            rekt.set_height(height)
        length, height = rekt.get_width(), rekt.get_height()

        lx = x, x+length
        ly = y, y+height
    
        lnz[0].set_data(lx, ly[::-1])
        lnz[1].set_data(lx, ly)

    if color is not None:
        if isinstance(color, (tuple, list)):
            for col, lnz in zip(color, checkbox.lines):
                [line.set_color(col) for line in lnz]
        else:
            for lnz in checkbox.lines:
                [line.set_color(color) for line in lnz]

def binomial_triangle(n):
    """
    Calculates the n-th row of the binomial triangle. The first row is n=1, not 0.
    Example:
    In: > binomial_triangle(4)
        > 1 3 3 1
    --------
    Parameters:
    n: int
        Row index
    --------
    Returns:
    row: 1darray
        The n-th row of binomial triangle.
    """
    row = []
    n -= 1
    for k in range(n+1):
        row.append(
                np.math.factorial(n) / ( np.math.factorial(k) * np.math.factorial(n-k) )
                )
    return np.array(row)
