"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_1 = require("./construct");
const runtime_info_1 = require("./private/runtime-info");
const tree_metadata_1 = require("./private/tree-metadata");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps_and_stacks.html
 */
class App extends construct_1.Construct {
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Synthesizes a cloud assembly for this app. Emits it to the directory
     * specified by `outdir`.
     *
     * @returns a `CloudAssembly` which can be used to inspect synthesized
     * artifacts such as CloudFormation templates and assets.
     */
    synth() {
        // we already have a cloud assembly, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const assembly = construct_1.ConstructNode.synth(this.node, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined
        });
        this._assembly = assembly;
        return assembly;
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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