"use strict";
const fc = require("fast-check");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
function asyncTest(cb) {
    return async (test) => {
        let error;
        try {
            await cb(test);
        }
        catch (e) {
            error = e;
        }
        finally {
            test.doesNotThrow(() => {
                if (error) {
                    throw error;
                }
            });
            test.done();
        }
    };
}
const nonEmptyString = fc.string(1, 16);
const tokenish = fc.array(nonEmptyString, 2, 2).map(arr => ({ [arr[0]]: arr[1] }));
const anyValue = fc.oneof(nonEmptyString, tokenish);
function stringListToken(o) {
    return lib_1.Token.asList(new intrinsic_1.Intrinsic(o));
}
function stringToken(o) {
    return lib_1.Token.asString(new intrinsic_1.Intrinsic(o));
}
module.exports = nodeunit.testCase({
    'eager resolution for non-tokens': {
        'Fn.select'(test) {
            test.deepEqual(lib_1.Fn.select(2, ['hello', 'you', 'dude']), 'dude');
            test.done();
        },
        'Fn.split'(test) {
            test.deepEqual(lib_1.Fn.split(':', 'hello:world:yeah'), ['hello', 'world', 'yeah']);
            test.done();
        }
    },
    'FnJoin': {
        'rejects empty list of arguments to join'(test) {
            test.throws(() => lib_1.Fn.join('.', []));
            test.done();
        },
        'collapse nested FnJoins even if they contain tokens'(test) {
            const stack = new lib_1.Stack();
            const obj = lib_1.Fn.join('', [
                'a',
                lib_1.Fn.join('', [lib_1.Fn.getAtt('a', 'bc').toString(), 'c']),
                'd'
            ]);
            test.deepEqual(stack.resolve(obj), { 'Fn::Join': ["",
                    [
                        "a",
                        { 'Fn::GetAtt': ['a', 'bc'] },
                        'cd',
                    ]
                ] });
            test.done();
        },
        'resolves to the value if only one value is joined': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), anyValue, (delimiter, value) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [value])), value)), { verbose: true });
        }),
        'pre-concatenates string literals': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 15), (delimiter, values) => stack.resolve(lib_1.Fn.join(delimiter, values)) === values.join(delimiter)), { verbose: true });
        }),
        'pre-concatenates around tokens': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 3), tokenish, fc.array(nonEmptyString, 1, 3), (delimiter, prefix, obj, suffix) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, stringToken(obj), ...suffix])), { 'Fn::Join': [delimiter, [prefix.join(delimiter), obj, suffix.join(delimiter)]] })), { verbose: true, seed: 1539874645005, path: "0:0:0:0:0:0:0:0:0" });
        }),
        'flattens joins nested under joins with same delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(anyValue), fc.array(anyValue, 1, 3), fc.array(anyValue), (delimiter, prefix, nested, suffix) => 
            // Gonna test
            _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, lib_1.Fn.join(delimiter, nested), ...suffix])), stack.resolve(lib_1.Fn.join(delimiter, [...prefix, ...nested, ...suffix])))), { verbose: true });
        }),
        'does not flatten joins nested under joins with different delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.string(), fc.array(anyValue, 1, 3), fc.array(tokenish, 2, 3), fc.array(anyValue, 3), (delimiter1, delimiter2, prefix, nested, suffix) => {
                fc.pre(delimiter1 !== delimiter2);
                const join = lib_1.Fn.join(delimiter1, [...prefix, lib_1.Fn.join(delimiter2, stringListToken(nested)), ...suffix]);
                const resolved = stack.resolve(join);
                return resolved['Fn::Join'][1].find((e) => typeof e === 'object'
                    && ('Fn::Join' in e)
                    && e['Fn::Join'][0] === delimiter2) != null;
            }), { verbose: true });
        }),
        'Fn::EachMemberIn': asyncTest(async (test) => {
            const stack = new lib_1.Stack();
            const eachMemberIn = lib_1.Fn.conditionEachMemberIn(lib_1.Fn.valueOfAll('AWS::EC2::Subnet::Id', 'VpcId'), lib_1.Fn.refAll('AWS::EC2::VPC::Id'));
            test.deepEqual(stack.resolve(eachMemberIn), {
                'Fn::EachMemberIn': [
                    { 'Fn::ValueOfAll': ['AWS::EC2::Subnet::Id', 'VpcId'] },
                    { 'Fn::RefAll': 'AWS::EC2::VPC::Id' }
                ]
            });
        }),
        'cross-stack FnJoin elements are properly resolved': asyncTest(async (test) => {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1');
            const stack2 = new lib_1.Stack(app, 'Stack2');
            // WHEN
            new lib_1.CfnOutput(stack2, 'Stack1Id', {
                value: lib_1.Fn.join(' = ', ['Stack1Id', stack1.stackId])
            });
            // THEN
            const template = app.synth().getStack('Stack2').template;
            test.deepEqual(template, {
                Outputs: {
                    Stack1Id: {
                        Value: {
                            'Fn::Join': [' = ', [
                                    'Stack1Id',
                                    { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSStackIdB2DD5BAA' }
                                ]]
                        }
                    }
                }
            });
        }),
    },
});
//# sourceMappingURL=data:application/json;base64,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