"use strict";
const cfn_resource_1 = require("../lib/cfn-resource");
const tag_manager_1 = require("../lib/tag-manager");
module.exports = {
    '#setTag() supports setting a tag regardless of Type'(test) {
        const notTaggable = new tag_manager_1.TagManager(cfn_resource_1.TagType.NOT_TAGGABLE, 'AWS::Resource::Type');
        notTaggable.setTag('key', 'value');
        test.deepEqual(notTaggable.renderTags(), undefined);
        test.done();
    },
    'when a tag does not exist': {
        '#removeTag() does not throw an error'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.doesNotThrow(() => (mgr.removeTag('dne', 0)));
            test.done();
        },
        '#setTag() creates the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'notanymore' }]);
            test.done();
        }
    },
    'when a tag does exist': {
        '#removeTag() deletes the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore', 0);
            mgr.removeTag('dne', 0);
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#setTag() overwrites the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            mgr.setTag('dne', 'iwin');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'iwin' }]);
            test.done();
        }
    },
    'when there are no tags': {
        '#renderTags() returns undefined'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#hasTags() returns false'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.equal(mgr.hasTags(), false);
            test.done();
        }
    },
    '#renderTags() handles standard, map, keyValue, and ASG tag formats'(test) {
        const tagged = [];
        const standard = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        const asg = new tag_manager_1.TagManager(cfn_resource_1.TagType.AUTOSCALING_GROUP, 'AWS::Resource::Type');
        const keyValue = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'AWS::Resource::Type');
        const mapper = new tag_manager_1.TagManager(cfn_resource_1.TagType.MAP, 'AWS::Resource::Type');
        tagged.push(standard);
        tagged.push(asg);
        tagged.push(keyValue);
        tagged.push(mapper);
        for (const res of tagged) {
            res.setTag('foo', 'bar');
            res.setTag('asg', 'only', 0, false);
        }
        test.deepEqual(standard.renderTags(), [
            { key: 'foo', value: 'bar' },
            { key: 'asg', value: 'only' },
        ]);
        test.deepEqual(asg.renderTags(), [
            { key: 'foo', value: 'bar', propagateAtLaunch: true },
            { key: 'asg', value: 'only', propagateAtLaunch: false },
        ]);
        test.deepEqual(keyValue.renderTags(), [
            { Key: 'foo', Value: 'bar' },
            { Key: 'asg', Value: 'only' }
        ]);
        test.deepEqual(mapper.renderTags(), {
            foo: 'bar',
            asg: 'only',
        });
        test.done();
    },
    'when there are tags it hasTags returns true'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.equal(mgr.hasTags(), true);
        test.done();
    },
    'tags with higher or equal priority always take precedence'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 2);
        test.deepEqual(mgr.renderTags(), undefined);
        test.done();
    },
    'excludeResourceTypes only tags resources that do not match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Fake::Resource');
        test.equal(false, mgr.applyTagAspectHere([], ['AWS::Fake::Resource']));
        test.equal(true, mgr.applyTagAspectHere([], ['AWS::Wrong::Resource']));
        test.done();
    },
    'includeResourceTypes only tags resources that match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Fake::Resource');
        test.equal(true, mgr.applyTagAspectHere(['AWS::Fake::Resource'], []));
        test.equal(false, mgr.applyTagAspectHere(['AWS::Wrong::Resource'], []));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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