"use strict";
const fc = require("fast-check");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
function asyncTest(cb) {
    return async (test) => {
        let error;
        try {
            await cb(test);
        }
        catch (e) {
            error = e;
        }
        finally {
            test.doesNotThrow(() => {
                if (error) {
                    throw error;
                }
            });
            test.done();
        }
    };
}
const nonEmptyString = fc.string(1, 16);
const tokenish = fc.array(nonEmptyString, 2, 2).map(arr => ({ [arr[0]]: arr[1] }));
const anyValue = fc.oneof(nonEmptyString, tokenish);
function stringListToken(o) {
    return lib_1.Token.asList(new intrinsic_1.Intrinsic(o));
}
function stringToken(o) {
    return lib_1.Token.asString(new intrinsic_1.Intrinsic(o));
}
module.exports = nodeunit.testCase({
    'eager resolution for non-tokens': {
        'Fn.select'(test) {
            test.deepEqual(lib_1.Fn.select(2, ['hello', 'you', 'dude']), 'dude');
            test.done();
        },
        'Fn.split'(test) {
            test.deepEqual(lib_1.Fn.split(':', 'hello:world:yeah'), ['hello', 'world', 'yeah']);
            test.done();
        }
    },
    'FnJoin': {
        'rejects empty list of arguments to join'(test) {
            test.throws(() => lib_1.Fn.join('.', []));
            test.done();
        },
        'collapse nested FnJoins even if they contain tokens'(test) {
            const stack = new lib_1.Stack();
            const obj = lib_1.Fn.join('', [
                'a',
                lib_1.Fn.join('', [lib_1.Fn.getAtt('a', 'bc').toString(), 'c']),
                'd'
            ]);
            test.deepEqual(stack.resolve(obj), { 'Fn::Join': ["",
                    [
                        "a",
                        { 'Fn::GetAtt': ['a', 'bc'] },
                        'cd',
                    ]
                ] });
            test.done();
        },
        'resolves to the value if only one value is joined': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), anyValue, (delimiter, value) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [value])), value)), { verbose: true });
        }),
        'pre-concatenates string literals': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 15), (delimiter, values) => stack.resolve(lib_1.Fn.join(delimiter, values)) === values.join(delimiter)), { verbose: true });
        }),
        'pre-concatenates around tokens': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 3), tokenish, fc.array(nonEmptyString, 1, 3), (delimiter, prefix, obj, suffix) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, stringToken(obj), ...suffix])), { 'Fn::Join': [delimiter, [prefix.join(delimiter), obj, suffix.join(delimiter)]] })), { verbose: true, seed: 1539874645005, path: "0:0:0:0:0:0:0:0:0" });
        }),
        'flattens joins nested under joins with same delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.array(anyValue), fc.array(anyValue, 1, 3), fc.array(anyValue), (delimiter, prefix, nested, suffix) => 
            // Gonna test
            _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, lib_1.Fn.join(delimiter, nested), ...suffix])), stack.resolve(lib_1.Fn.join(delimiter, [...prefix, ...nested, ...suffix])))), { verbose: true });
        }),
        'does not flatten joins nested under joins with different delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            await fc.assert(fc.property(fc.string(), fc.string(), fc.array(anyValue, 1, 3), fc.array(tokenish, 2, 3), fc.array(anyValue, 3), (delimiter1, delimiter2, prefix, nested, suffix) => {
                fc.pre(delimiter1 !== delimiter2);
                const join = lib_1.Fn.join(delimiter1, [...prefix, lib_1.Fn.join(delimiter2, stringListToken(nested)), ...suffix]);
                const resolved = stack.resolve(join);
                return resolved['Fn::Join'][1].find((e) => typeof e === 'object'
                    && ('Fn::Join' in e)
                    && e['Fn::Join'][0] === delimiter2) != null;
            }), { verbose: true });
        }),
        'Fn::EachMemberIn': asyncTest(async (test) => {
            const stack = new lib_1.Stack();
            const eachMemberIn = lib_1.Fn.conditionEachMemberIn(lib_1.Fn.valueOfAll('AWS::EC2::Subnet::Id', 'VpcId'), lib_1.Fn.refAll('AWS::EC2::VPC::Id'));
            test.deepEqual(stack.resolve(eachMemberIn), {
                'Fn::EachMemberIn': [
                    { 'Fn::ValueOfAll': ['AWS::EC2::Subnet::Id', 'VpcId'] },
                    { 'Fn::RefAll': 'AWS::EC2::VPC::Id' }
                ]
            });
        }),
        'cross-stack FnJoin elements are properly resolved': asyncTest(async (test) => {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1');
            const stack2 = new lib_1.Stack(app, 'Stack2');
            // WHEN
            new lib_1.CfnOutput(stack2, 'Stack1Id', {
                value: lib_1.Fn.join(' = ', ['Stack1Id', stack1.stackId])
            });
            // THEN
            const template = app.synth().getStackByName('Stack2').template;
            test.deepEqual(template, {
                Outputs: {
                    Stack1Id: {
                        Value: {
                            'Fn::Join': [' = ', [
                                    'Stack1Id',
                                    { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSStackIdB2DD5BAA' }
                                ]]
                        }
                    }
                }
            });
        }),
    },
    'Ref': {
        'returns a reference given a logical name'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(lib_1.Fn.ref('hello')), {
                Ref: 'hello'
            });
            test.done();
        }
    },
    'nested Fn::Join with list token'(test) {
        const stack = new lib_1.Stack();
        const inner = lib_1.Fn.join(',', lib_1.Token.asList({ NotReallyList: true }));
        const outer = lib_1.Fn.join(',', [inner, 'Foo']);
        test.deepEqual(stack.resolve(outer), {
            'Fn::Join': [
                ',',
                [
                    { 'Fn::Join': [',', { NotReallyList: true }] },
                    'Foo'
                ]
            ]
        });
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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