import iam = require('@aws-cdk/aws-iam');
import kms = require('@aws-cdk/aws-kms');
import { Construct, IResource, Resource } from '@aws-cdk/core';
export interface IStream extends IResource {
    /**
     * The ARN of the stream.
     *
     * @attribute
     */
    readonly streamArn: string;
    /**
     * The name of the stream
     *
     * @attribute
     */
    readonly streamName: string;
    /**
     * Optional KMS encryption key associated with this stream.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to encrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee: iam.IGrantable): iam.Grant;
}
/**
 * A reference to a stream. The easiest way to instantiate is to call
 * `stream.export()`. Then, the consumer can use `Stream.import(this, ref)` and
 * get a `Stream`.
 */
export interface StreamAttributes {
    /**
     * The ARN of the stream.
     */
    readonly streamArn: string;
    /**
     * The KMS key securing the contents of the stream if encryption is enabled.
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * Represents a Kinesis Stream.
 *
 * Streams can be either defined within this stack:
 *
 *   new Stream(this, 'MyStream', { props });
 *
 * Or imported from an existing stream:
 *
 *   Stream.import(this, 'MyImportedStream', { streamArn: ... });
 *
 * You can also export a stream and import it into another stack:
 *
 *   const ref = myStream.export();
 *   Stream.import(this, 'MyImportedStream', ref);
 *
 */
declare abstract class StreamBase extends Resource implements IStream {
    /**
     * The ARN of the stream.
     */
    abstract readonly streamArn: string;
    /**
     * The name of the stream
     */
    abstract readonly streamName: string;
    /**
     * Optional KMS encryption key associated with this stream.
     */
    abstract readonly encryptionKey?: kms.IKey;
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee: iam.IGrantable): iam.Grant;
    private grant;
}
export interface StreamProps {
    /**
     * Enforces a particular physical stream name.
     * @default <generated>
     */
    readonly streamName?: string;
    /**
     * The number of hours for the data records that are stored in shards to remain accessible.
     * @default 24
     */
    readonly retentionPeriodHours?: number;
    /**
     * The number of shards for the stream.
     * @default 1
     */
    readonly shardCount?: number;
    /**
     * The kind of server-side encryption to apply to this stream.
     *
     * If you choose KMS, you can specify a KMS key via `encryptionKey`. If
     * encryption key is not specified, a key will automatically be created.
     *
     * @default Unencrypted
     */
    readonly encryption?: StreamEncryption;
    /**
     * External KMS key to use for stream encryption.
     *
     * The 'encryption' property must be set to "Kms".
     *
     * @default If encryption is set to "Kms" and this property is undefined, a
     * new KMS key will be created and associated with this stream.
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
export declare class Stream extends StreamBase {
    static fromStreamArn(scope: Construct, id: string, streamArn: string): IStream;
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope: Construct, id: string, attrs: StreamAttributes): IStream;
    readonly streamArn: string;
    readonly streamName: string;
    readonly encryptionKey?: kms.IKey;
    private readonly stream;
    constructor(scope: Construct, id: string, props?: StreamProps);
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    private parseEncryption;
}
/**
 * What kind of server-side encryption to apply to this stream
 */
export declare enum StreamEncryption {
    /**
     * Records in the stream are not encrypted.
     */
    UNENCRYPTED = "NONE",
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    KMS = "KMS"
}
export {};
