"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateArrayLength = validateArrayLength;
exports.validateNumberRange = validateNumberRange;
exports.validateObjectAttributes = validateObjectAttributes;
exports.validateStringLength = validateStringLength;
exports.validateRegExp = validateRegExp;
/**
 * Validates that the length of an array falls within optional minimum and/or maximum bounds.
 *
 * @param value - The array to validate.
 * @param min - (Optional) Minimum allowed length of the array.
 * @param max - (Optional) Maximum allowed length of the array.
 * @param message - (Optional) Custom error message to use if the validation fails.
 * @param messagePrefix - (Optional) String to prepend to the error message(s).
 * @returns An array of error messages. Returns an empty array if the length is valid or no bounds are specified.
 *
 * @throws Error if both `min` and `max` are provided and `min` is greater than `max`.
 */
function validateArrayLength({ value, min, max, message, messagePrefix, }) {
    if (min === undefined && max === undefined) {
        return [];
    }
    if (min !== undefined && max !== undefined && min > max) {
        throw new Error('min must be less than max');
    }
    const errors = [];
    if (min !== undefined && value.length < min) {
        errors.push((messagePrefix || '') + (message || `must have at least ${min} element${min === 1 ? '' : 's'}`));
    }
    if (max !== undefined && value.length > max) {
        errors.push((messagePrefix || '') + (message || `must have at most ${max} element${max === 1 ? '' : 's'}`));
    }
    return errors;
}
/**
 * Validates that a number falls within optional minimum and/or maximum bounds.
 *
 * @param value - The number to validate.
 * @param min - (Optional) Minimum allowed value of the number.
 * @param max - (Optional) Maximum allowed value of the number.
 * @param message - (Optional) Custom error message to use if the validation fails.
 * @param messagePrefix - (Optional) String to prepend to the error message(s).
 * @returns An array of error messages. Returns an empty array if the number is valid or no bounds are specified.
 *
 * @throws Error if both `min` and `max` are provided and `min` is greater than `max`.
 *
 */
function validateNumberRange({ value, min, max, message, messagePrefix, }) {
    if (min === undefined && max === undefined) {
        return [];
    }
    if (min !== undefined && max !== undefined && min > max) {
        throw new Error('min must be less than max');
    }
    const errors = [];
    if (min !== undefined && value < min) {
        errors.push((messagePrefix || '') + (message || `must be at least ${min}`));
    }
    if (max !== undefined && value > max) {
        errors.push((messagePrefix || '') + (message || `must be at most ${max}`));
    }
    return errors;
}
/**
 * Validates that an object contains only the specified mandatory and optional attributes.
 *
 * @param inputObject - The object to validate.
 * @param mandatoryAttributes - An array of mandatory attribute names.
 * @param optionalAttributes - An array of optional attribute names.
 * @param message - (Optional) Custom error message to use if the validation fails.
 * @param messagePrefix - (Optional) String to prepend to the error message(s).
 * @returns An array of error messages. Returns an empty array if the object is valid.
 */
function validateObjectAttributes({ inputObject, mandatoryAttributes, optionalAttributes, message, messagePrefix, }) {
    const errors = [];
    const keys = Object.keys(inputObject);
    const allAttributes = mandatoryAttributes.concat(optionalAttributes);
    for (const mandatoryAttribute of mandatoryAttributes) {
        if (!keys.includes(mandatoryAttribute)) {
            errors.push(messagePrefix ||
                '' + (message || `missing mandatory attribute: ${mandatoryAttribute}`));
        }
    }
    for (const key of keys) {
        if (!allAttributes.includes(key)) {
            errors.push((messagePrefix || '') + (message || `contains unknown attribute: ${key}`));
        }
    }
    return errors;
}
/**
 * Validates that a string falls within optional minimum and/or maximum length bounds.
 *
 * @param value - The string to validate.
 * @param min - (Optional) Minimum allowed length of the string.
 * @param max - (Optional) Maximum allowed length of the string.
 * @param message - (Optional) Custom error message to use if the validation fails.
 * @param messagePrefix - (Optional) String to prepend to the error message(s).
 * @returns An array of error messages. Returns an empty array if the length is valid or no bounds are specified.
 *
 * @throws Error if both `min` and `max` are provided and `min` is greater than `max`.
 */
function validateStringLength({ value, min, max, message, messagePrefix, }) {
    if (min === undefined && max === undefined) {
        return [];
    }
    if (min !== undefined && max !== undefined && min > max) {
        throw new Error('min must be less than max');
    }
    const errors = [];
    if (min !== undefined && value.length < min) {
        errors.push((messagePrefix ||
            '') + (message || `must be at least ${min} character${min === 1 ? '' : 's'} long`));
    }
    if (max !== undefined && value.length > max) {
        errors.push((messagePrefix ||
            '') + (message || `must be at most ${max} character${max === 1 ? '' : 's'} long`));
    }
    return errors;
}
/**
 * Validates that a string matches a regular expression.
 *
 * @param value - The string to validate.
 * @param regExp - The regular expression to match.
 * @param message - (Optional) Custom error message to use if the validation fails.
 * @param messagePrefix - (Optional) String to prepend to the error message(s).
 * @returns An array of error messages. Returns an empty array if the string matches the regular expression.
 */
function validateRegExp({ value, regExp, message, messagePrefix, }) {
    const errors = [];
    if (!regExp.test(value)) {
        errors.push((messagePrefix || '') + (message || `must match regExp ${regExp}`));
    }
    return errors;
}
//# sourceMappingURL=data:application/json;base64,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