"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LoadBalancer = exports.LoadBalancerGeneration = void 0;
/**
 * The generations of AWS load balancing solutions.
 */
var LoadBalancerGeneration;
(function (LoadBalancerGeneration) {
    /**
     * The first generation (ELB Classic).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["FIRST"] = 0] = "FIRST";
    /**
     * The second generation (ALB and NLB).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["SECOND"] = 1] = "SECOND";
})(LoadBalancerGeneration = exports.LoadBalancerGeneration || (exports.LoadBalancerGeneration = {}));
/**
 * An interface of an abstract load balancer, as needed by CodeDeploy.
 * Create instances using the static factory methods:
 * {@link #classic}, {@link #application} and {@link #network}.
 */
class LoadBalancer {
    /**
     * Creates a new CodeDeploy load balancer from a Classic ELB Load Balancer.
     *
     * @param loadBalancer a classic ELB Load Balancer
     */
    static classic(loadBalancer) {
        class ClassicLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.FIRST;
                this.name = loadBalancer.loadBalancerName;
            }
        }
        return new ClassicLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from an Application Load Balancer Target Group.
     *
     * @param albTargetGroup an ALB Target Group
     */
    static application(albTargetGroup) {
        class AlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = albTargetGroup.targetGroupName;
            }
        }
        return new AlbLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from a Network Load Balancer Target Group.
     *
     * @param nlbTargetGroup an NLB Target Group
     */
    static network(nlbTargetGroup) {
        class NlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = nlbTargetGroup.targetGroupName;
            }
        }
        return new NlbLoadBalancer();
    }
}
exports.LoadBalancer = LoadBalancer;
//# sourceMappingURL=data:application/json;base64,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