"use strict";
const assert_1 = require("@aws-cdk/assert");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const lbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/core");
const codedeploy = require("../../lib");
module.exports = {
    'CodeDeploy Server Deployment Group': {
        'can be created by explicitly passing an Application'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.ServerApplication(stack, 'MyApp');
            new codedeploy.ServerDeploymentGroup(stack, 'MyDG', {
                application,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'ApplicationName': {
                    'Ref': 'MyApp3CE31C26',
                },
            }));
            test.done();
        },
        'can be imported'(test) {
            const stack = new cdk.Stack();
            const application = codedeploy.ServerApplication.fromServerApplicationName(stack, 'MyApp', 'MyApp');
            const deploymentGroup = codedeploy.ServerDeploymentGroup.fromServerDeploymentGroupAttributes(stack, 'MyDG', {
                application,
                deploymentGroupName: 'MyDG',
            });
            test.notEqual(deploymentGroup, undefined);
            test.done();
        },
        'created with ASGs contains the ASG names'(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.STANDARD3, ec2.InstanceSize.SMALL),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoScalingGroups: [asg],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoScalingGroups': [
                    {
                        'Ref': 'ASG46ED3070',
                    },
                ],
            }));
            test.done();
        },
        'created without ASGs but adding them later contains the ASG names'(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.STANDARD3, ec2.InstanceSize.SMALL),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAutoScalingGroup(asg);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoScalingGroups': [
                    {
                        'Ref': 'ASG46ED3070',
                    },
                ],
            }));
            test.done();
        },
        'can be created with an ALB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const alb = new lbv2.ApplicationLoadBalancer(stack, 'ALB', {
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const listener = alb.addListener('Listener', { protocol: lbv2.ApplicationProtocol.HTTP });
            const targetGroup = listener.addTargets('Fleet', { protocol: lbv2.ApplicationProtocol.HTTP });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: codedeploy.LoadBalancer.application(targetGroup),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'LoadBalancerInfo': {
                    'TargetGroupInfoList': [
                        {
                            'Name': {
                                'Fn::GetAtt': [
                                    'ALBListenerFleetGroup008CEEE4',
                                    'TargetGroupName',
                                ],
                            },
                        },
                    ],
                },
                'DeploymentStyle': {
                    'DeploymentOption': 'WITH_TRAFFIC_CONTROL',
                },
            }));
            test.done();
        },
        'can be created with an NLB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const nlb = new lbv2.NetworkLoadBalancer(stack, 'NLB', {
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const listener = nlb.addListener('Listener', { port: 80 });
            const targetGroup = listener.addTargets('Fleet', { port: 80 });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: codedeploy.LoadBalancer.network(targetGroup),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'LoadBalancerInfo': {
                    'TargetGroupInfoList': [
                        {
                            'Name': {
                                'Fn::GetAtt': [
                                    'NLBListenerFleetGroupB882EC86',
                                    'TargetGroupName',
                                ],
                            },
                        },
                    ],
                },
                'DeploymentStyle': {
                    'DeploymentOption': 'WITH_TRAFFIC_CONTROL',
                },
            }));
            test.done();
        },
        'can be created with a single EC2 instance tag set with a single or no value'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                ec2InstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value'],
                    'other-key': [],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'Ec2TagSet': {
                    'Ec2TagSetList': [
                        {
                            'Ec2TagGroup': [
                                {
                                    'Key': 'some-key',
                                    'Value': 'some-value',
                                    'Type': 'KEY_AND_VALUE',
                                },
                                {
                                    'Key': 'other-key',
                                    'Type': 'KEY_ONLY',
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'can be created with two on-premise instance tag sets with multiple values or without a key'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value', 'another-value'],
                }, {
                    '': ['keyless-value'],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'OnPremisesTagSet': {
                    'OnPremisesTagSetList': [
                        {
                            'OnPremisesTagGroup': [
                                {
                                    'Key': 'some-key',
                                    'Value': 'some-value',
                                    'Type': 'KEY_AND_VALUE',
                                },
                                {
                                    'Key': 'some-key',
                                    'Value': 'another-value',
                                    'Type': 'KEY_AND_VALUE',
                                },
                            ],
                        },
                        {
                            'OnPremisesTagGroup': [
                                {
                                    'Value': 'keyless-value',
                                    'Type': 'VALUE_ONLY',
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'cannot be created with an instance tag set containing a keyless, valueless filter'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                        '': [],
                    }),
                });
            });
            test.done();
        },
        'cannot be created with an instance tag set containing 4 instance tag groups'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({}, {}, {}, {}),
                });
            }, /3/);
            test.done();
        },
        'can have alarms added to it after being created'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AlarmConfiguration': {
                    'Alarms': [
                        {
                            'Name': {
                                'Ref': 'Alarm1F9009D71',
                            },
                        },
                    ],
                    'Enabled': true,
                },
            }));
            test.done();
        },
        'only automatically rolls back failed deployments by default'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoRollbackConfiguration': {
                    'Enabled': true,
                    'Events': [
                        'DEPLOYMENT_FAILURE',
                    ],
                },
            }));
            test.done();
        },
        'rolls back alarmed deployments if at least one alarm has been added'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    failedDeployment: false,
                },
            });
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoRollbackConfiguration': {
                    'Enabled': true,
                    'Events': [
                        'DEPLOYMENT_STOP_ON_ALARM',
                    ],
                },
            }));
            test.done();
        },
        'setting to roll back on alarms without providing any results in an exception'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    deploymentInAlarm: true,
                },
            });
            test.throws(() => assert_1.SynthUtils.toCloudFormation(stack), /deploymentInAlarm/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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