# pylint: disable=line-too-long, unused-wildcard-import
"""
The |sw1d.DOCNAME.long| model family member |sw1d_gate_out| is a routing submodel,
which calculates the flow under a submerged gate.

Principally, |sw1d_gate_out| is similar to |sw1d_weir_out|, which calculates the flow
over a weir.  Both models must be placed at a network outlet.  However, while
|sw1d_weir_out| calculates free flow and thus requires no information from downstream
locations, |sw1d_gate_out| allows for flows in both directions depending on the
(dynamically calculated) upstream water level and an (externally supplied) downstream
water level.  Another key difference is that |sw1d_gate_out| allows model users to
define project specific-functions for controlling the gate opening.

Integration tests
=================

.. how_to_understand_integration_tests::

We create a test setting identical to the one of the |sw1d_channel| documentation,
except adding a |sw1d_gate_out| submodel at the channel outlet:

>>> from hydpy import pub
>>> pub.timegrids = "2000-01-01 00:00", "2000-01-01 05:00", "5m"

>>> from hydpy.models.sw1d_channel import *
>>> parameterstep()
>>> nmbsegments(8)
>>> lengths = 2.0, 3.0, 2.0, 3.0, 2.0, 3.0, 2.0, 3.0

>>> for i, length_ in enumerate(lengths):
...     with model.add_storagemodel_v1("sw1d_storage", position=i):
...         length(length_)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

>>> for i in range(1, nmbsegments.value):
...     with model.add_routingmodel_v2("sw1d_lias", position=i):
...         lengthupstream(2.0 if i % 2 else 3.0)
...         lengthdownstream(3.0 if i % 2 else 2.0)
...         stricklercoefficient(1.0/0.03)
...         timestepfactor(0.7)
...         diffusionfactor(0.2)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

>>> with model.add_routingmodel_v3("sw1d_gate_out", position=8):
...     lengthupstream(lengths[-1])
...     bottomlevel(5.0)
...     gateheight(6.5)
...     gatewidth(2.0)
...     flowcoefficient(0.6)
...     timestepfactor(0.7)
...     dampingradius(0.0)

We need to connect two nodes with the |sw1d_gate_out| submodel.  One node for querying
the outflow, which must be connectible to the outlet sequence |sw1d_outlets.LongQ|.
And another node that supplies the downstream water levels, which must be connectible
to the receiver sequence |sw1d_receivers.WaterLevel|:

>>> from hydpy import Element, Node
>>> outflow = Node("outflow", variable="LongQ")
>>> waterlevel = Node("waterlevel", variable="WaterLevel")
>>> channel = Element("channel", outlets=outflow, receivers=waterlevel)
>>> channel.model = model

>>> from hydpy import IntegrationTest
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

>>> def prepare_inits(hs):
...     if isinstance(hs, float):
...         hs = nmbsegments.value * [hs]
...     inits = []
...     for h, s in zip(hs, model.storagemodels):
...         length = s.parameters.control.length
...         c = s.crosssection.parameters.control
...         v = h * (c.bottomwidths[0] + h * c.sideslopes[0]) * length
...         inits.append((s.sequences.states.watervolume, v))
...     for r in model.routingmodels[1:]:
...         inits.append((r.sequences.states.discharge, 0.0))
...     test.inits = inits

.. _sw1d_gate_out_low_outer_waterlevel:

Low outer water level
_____________________

The first example deals with a constant outer water level lower than the initial
upstream water level:

>>> waterlevel.sequences.sim.series = 6.0
>>> prepare_inits(hs=2.0)

The water gradient causes a downstream flow and, thus, falling upstream water levels.
There is no notable shift in behaviour when the upstream water loses contact with the
gate after about a half hour:

.. integration-test::

    >>> conditions = test("sw1d_gate_out_low_outer_waterlevel", get_conditions="2000-01-01 00:00")
    |                date | timestep |                                                                       waterlevels |                                                                            discharges |  outflow | waterlevel |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |    300.0 |      7.0       7.0       7.0       7.0       7.0       7.0       7.0      6.84054 | 0.0       0.0       0.0       0.0       0.0       0.0       0.0       0.0    7.973004 | 7.973004 |        6.0 |
    | 2000-01-01 00:05:00 |    300.0 |      7.0       7.0       7.0       7.0       7.0       7.0  6.945481     6.738462 | 0.0       0.0       0.0       0.0       0.0       0.0       0.0  1.817298    6.921191 | 6.921191 |        6.0 |
    | 2000-01-01 00:10:00 |    300.0 |      7.0       7.0       7.0       7.0       7.0  6.987374  6.874537       6.6733 | 0.0       0.0       0.0       0.0       0.0       0.0    0.6313  2.996118    6.254197 | 6.254197 |        6.0 |
    | 2000-01-01 00:15:00 |    300.0 |      7.0       7.0       7.0       7.0  6.995552  6.954362  6.843308     6.613501 | 0.0       0.0       0.0       0.0       0.0  0.148258  1.798845  2.839794    5.829791 | 5.829791 |        6.0 |
    | 2000-01-01 00:20:00 |    300.0 |      7.0       7.0       7.0  6.998954  6.974556  6.928642  6.815668     6.563883 | 0.0       0.0       0.0       0.0  0.052289  0.752166  2.038193   2.95952      5.4404 |   5.4404 |        6.0 |
    | 2000-01-01 00:25:00 |    300.0 |      7.0       7.0  6.999631  6.991257  6.952415  6.909067  6.789464     6.521296 | 0.0       0.0       0.0  0.012308  0.397166  1.135184  2.113901  2.987387    5.116741 | 5.116741 |        6.0 |
    | 2000-01-01 00:30:00 |    300.0 |      7.0  6.999913  6.995335  6.978076  6.937773  6.891541  6.765109     6.484192 | 0.0       0.0  0.004346  0.147536  0.806565  1.294661  2.171003  2.982815    4.838009 | 4.838009 |        6.0 |
    | 2000-01-01 00:35:00 |    300.0 | 6.999969  6.998494   6.98586  6.964833  6.927103  6.875491  6.742387     6.451622 | 0.0  0.001023  0.071974   0.38781  1.049983  1.405649  2.208146  2.965559    4.594039 | 4.594039 |        6.0 |
    | 2000-01-01 00:40:00 |    300.0 | 6.999208  6.994191   6.97473  6.953933  6.916757  6.860787  6.721308     6.422833 | 0.0  0.025371    0.2405  0.611512  1.156498  1.501343  2.236548  2.939186    4.378668 | 4.378668 |        6.0 |
    | 2000-01-01 00:45:00 |    300.0 | 6.996127  6.987096  6.965669  6.944816  6.906164  6.846957  6.701853     6.397261 | 0.0  0.102719  0.457477  0.759512  1.215391  1.568503  2.259962  2.908463    4.187053 | 4.187053 |        6.0 |
    | 2000-01-01 00:50:00 |    300.0 | 6.989275  6.979004  6.958884  6.936733  6.895718  6.833671  6.683849     6.374468 | 0.0  0.228394  0.633028  0.859189  1.263334  1.611534  2.275849  2.875984    4.015633 | 4.015633 |        6.0 |
    | 2000-01-01 00:55:00 |    300.0 | 6.978849  6.971204  6.952884  6.929323    6.8857  6.820833  6.667091     6.354083 | 0.0  0.347526  0.737497  0.937479  1.307966  1.641909  2.283818  2.842405    3.861663 | 3.861663 |        6.0 |
    | 2000-01-01 01:00:00 |    300.0 | 6.966627  6.963535  6.946678  6.922282  6.876179  6.808462   6.65142     6.335787 | 0.0  0.407391  0.790845  0.997725  1.349792  1.667145  2.285674   2.80804    3.722844 | 3.722844 |        6.0 |
    | 2000-01-01 01:05:00 |    300.0 |  6.95457  6.955317  6.939916   6.91531   6.86708  6.796596  6.636725     6.319309 | 0.0  0.401917  0.812847  1.038242  1.386817  1.690122  2.283453  2.773301    3.597209 | 3.597209 |        6.0 |
    | 2000-01-01 01:10:00 |    300.0 | 6.943781  6.946279  6.932447  6.908183  6.858259  6.785241  6.622924     6.304419 | 0.0  0.359609  0.811472  1.060441  1.416821  1.710853  2.278609  2.738631    3.483089 | 3.483089 |        6.0 |
    | 2000-01-01 01:15:00 |    300.0 |  6.93439  6.936689  6.924219  6.900744  6.849575  6.774367   6.60995     6.290926 | 0.0  0.313035  0.792573  1.066851   1.43878  1.728245  2.271922  2.704405     3.37908 |  3.37908 |        6.0 |
    | 2000-01-01 01:20:00 |    300.0 | 6.925931  6.927028  6.915361  6.892897  6.840915  6.763919  6.597734     6.278664 | 0.0   0.28199  0.765027  1.060273  1.452641  1.741304  2.263694   2.67088    3.283999 | 3.283999 |        6.0 |
    | 2000-01-01 01:25:00 |    300.0 | 6.917804  6.917676  6.906171  6.884615  6.832195   6.75383  6.586208      6.26749 | 0.0  0.270883  0.738459  1.044801  1.458874  1.749527  2.253985  2.638183    3.196841 | 3.196841 |        6.0 |
    | 2000-01-01 01:30:00 |    300.0 | 6.909595  6.908763  6.896975  6.875962  6.823366  6.744031  6.575301     6.257282 | 0.0  0.273637  0.719323  1.025872  1.458532  1.752844  2.242784  2.606344    3.116738 | 3.116738 |        6.0 |
    | 2000-01-01 01:35:00 |    300.0 | 6.901165    6.9002  6.887986   6.86707   6.81442  6.734461  6.564944      6.24793 | 0.0  0.281004  0.709135  1.008748  1.453342  1.751553  2.230076  2.575325    3.042929 | 3.042929 |        6.0 |
    | 2000-01-01 01:40:00 |    300.0 | 6.892577  6.891814  6.879253  6.858095  6.805393   6.72507  6.555069     6.239337 | 0.0  0.286275  0.705562  0.996681  1.445427  1.746324  2.215888  2.545051    2.974737 | 2.974737 |        6.0 |
    | 2000-01-01 01:45:00 |    300.0 |  6.88396  6.883459  6.870699  6.849163   6.79635  6.715826  6.545616     6.231414 | 0.0  0.287219  0.704991  0.990107  1.436709  1.738133  2.200317  2.515428    2.911554 | 2.911554 |        6.0 |
    | 2000-01-01 01:50:00 |    300.0 | 6.875413  6.875064  6.862221  6.840341   6.78736  6.706716  6.536531     6.224085 | 0.0  0.284918  0.704679  0.987299   1.42839  1.728067   2.18355   2.48637     2.85283 |  2.85283 |        6.0 |
    | 2000-01-01 01:55:00 |    300.0 | 6.866965  6.866626  6.853749  6.831642  6.778472  6.697741  6.527772      6.21728 | 0.0  0.281585  0.703459  0.985851  1.420809  1.717069  2.165843   2.45781     2.79807 |  2.79807 |        6.0 |
    | 2000-01-01 02:00:00 |    300.0 | 6.858597  6.858178   6.84527  6.823047   6.76971  6.688906  6.519305     6.210938 | 0.0  0.278941  0.701324  0.983952  1.413685  1.705755  2.147477  2.429711    2.746825 | 2.746825 |        6.0 |
    | 2000-01-01 02:05:00 |    300.0 | 6.850269  6.849756  6.836805  6.814534  6.761075   6.68022  6.511104     6.205005 | 0.0  0.277593  0.698696   0.98085  1.406539  1.694381  2.128701  2.402063    2.698697 | 2.698697 |        6.0 |
    | 2000-01-01 02:10:00 |    300.0 | 6.841952  6.841382  6.828384  6.806086  6.752556  6.671685  6.503149     6.199436 | 0.0  0.277221  0.695922  0.976625  1.399006   1.68295  2.109708   2.37487    2.653337 | 2.653337 |        6.0 |
    | 2000-01-01 02:15:00 |    300.0 | 6.833639  6.833063  6.820026  6.797701  6.744144  6.663299  6.495423      6.19419 | 0.0  0.277116  0.693105  0.971699  1.390949   1.67136  2.090618  2.348144    2.610437 | 2.610437 |        6.0 |
    | 2000-01-01 02:20:00 |    300.0 | 6.825339  6.824793  6.811737  6.789382  6.735831   6.65506  6.487912     6.189233 | 0.0  0.276667  0.690151  0.966454   1.38241  1.659524  2.071502  2.321893    2.569733 | 2.569733 |        6.0 |
    | 2000-01-01 02:25:00 |    300.0 | 6.817071  6.816567  6.803513  6.781133  6.727613   6.64696    6.4806     6.184535 | 0.0  0.275606   0.68691  0.961069  1.373505  1.647414    2.0524  2.296117    2.530993 | 2.530993 |        6.0 |
    | 2000-01-01 02:30:00 |    300.0 | 6.808851  6.808381  6.795345  6.772957  6.719492  6.638995  6.473476     6.180071 | 0.0  0.273991   0.68328  0.955542  1.364335  1.635056  2.033338  2.270814    2.494018 | 2.494018 |        6.0 |
    | 2000-01-01 02:35:00 |    300.0 | 6.800689  6.800237  6.787229  6.764854  6.711465  6.631158  6.466527     6.175818 | 0.0  0.272052  0.679252  0.949791  1.354958  1.622502  2.014342  2.245972    2.458632 | 2.458632 |        6.0 |
    | 2000-01-01 02:40:00 |    300.0 | 6.792589   6.79214  6.779163  6.756821  6.703533  6.623445  6.459743     6.171756 | 0.0  0.270023  0.674896  0.943747   1.34539    1.6098  1.995435  2.221581    2.424683 | 2.424683 |        6.0 |
    | 2000-01-01 02:45:00 |    300.0 | 6.784547  6.784094  6.771151  6.748857  6.695692  6.615852  6.453113     6.167868 | 0.0  0.268047  0.670316  0.937398  1.335632  1.596986  1.976639  2.197628    2.392036 | 2.392036 |        6.0 |
    | 2000-01-01 02:50:00 |    300.0 | 6.776563  6.776105  6.763195  6.740959  6.687941  6.608374  6.446629     6.164138 | 0.0  0.266158  0.665604  0.930789  1.325686  1.584081   1.95797    2.1741    2.360572 | 2.360572 |        6.0 |
    | 2000-01-01 02:55:00 |    300.0 | 6.768633  6.768175    6.7553  6.733127  6.680275  6.601007  6.440283     6.160554 | 0.0  0.264319   0.66082  0.923992  1.315569  1.571099  1.939442  2.150985    2.330187 | 2.330187 |        6.0 |
    | 2000-01-01 03:00:00 |    300.0 | 6.760759  6.760305  6.747467  6.725362  6.672692  6.593747  6.434067     6.157104 | 0.0  0.262471  0.655989  0.917075  1.305311  1.558054  1.921063  2.128269     2.30079 |  2.30079 |        6.0 |
    | 2000-01-01 03:05:00 |    300.0 | 6.752942  6.752494  6.739698  6.717665  6.665192   6.58659  6.427974     6.153777 | 0.0  0.260575  0.651114  0.910088  1.294947  1.544961  1.902838  2.105938    2.272298 | 2.272298 |        6.0 |
    | 2000-01-01 03:10:00 |    300.0 | 6.745183  6.744743  6.731992  6.710036  6.657772  6.579531  6.421997     6.150563 | 0.0  0.258617  0.646189  0.903054   1.28451  1.531839  1.884773  2.083979     2.24464 |  2.24464 |        6.0 |
    | 2000-01-01 03:15:00 |    300.0 | 6.737485  6.737051  6.724349  6.702475  6.650432  6.572568  6.416132     6.147456 | 0.0   0.25661  0.641211  0.895982  1.274026   1.51871  1.866872  2.062379    2.217751 | 2.217751 |        6.0 |
    | 2000-01-01 03:20:00 |    300.0 | 6.729848  6.729418  6.716768  6.694982  6.643169  6.565697  6.410373     6.144447 | 0.0  0.254575  0.636186  0.888872  1.263514  1.505592  1.849138  2.041124    2.191575 | 2.191575 |        6.0 |
    | 2000-01-01 03:25:00 |    300.0 | 6.722272  6.721847   6.70925  6.687557  6.635984  6.558915  6.404714      6.14153 | 0.0   0.25253  0.631124  0.881725  1.252988  1.492501  1.831574  2.020203    2.166059 | 2.166059 |        6.0 |
    | 2000-01-01 03:30:00 |    300.0 | 6.714757  6.714336  6.701795  6.680199  6.628874  6.552221  6.399151     6.138699 | 0.0  0.250487  0.626036  0.874549  1.242458  1.479451  1.814184  1.999603    2.141159 | 2.141159 |        6.0 |
    | 2000-01-01 03:35:00 |    300.0 | 6.707304  6.706886  6.694402  6.672907  6.621839   6.54561  6.393681     6.135948 | 0.0  0.248448  0.620935  0.867353  1.231935  1.466452   1.79697  1.979315    2.116833 | 2.116833 |        6.0 |
    | 2000-01-01 03:40:00 |    300.0 | 6.699911  6.699497  6.687073  6.665682  6.614876  6.539082  6.388299     6.133274 | 0.0  0.246412  0.615829  0.860149  1.221428  1.453511  1.779932  1.959328    2.093045 | 2.093045 |        6.0 |
    | 2000-01-01 03:45:00 |    300.0 |  6.69258  6.692171  6.679806  6.658522  6.607985  6.532633  6.383002     6.130671 | 0.0  0.244377  0.610723  0.852947  1.210945  1.440638   1.76307  1.939632    2.069761 | 2.069761 |        6.0 |
    | 2000-01-01 03:50:00 |    300.0 |  6.68531  6.684905  6.672602  6.651427  6.601165  6.526262  6.377787     6.128137 | 0.0   0.24234  0.605622  0.845757  1.200497  1.427837  1.746386  1.920217    2.046952 | 2.046952 |        6.0 |
    | 2000-01-01 03:55:00 |    300.0 | 6.678101    6.6777   6.66546  6.644397  6.594414  6.519967  6.372651     6.125666 | 0.0  0.240303  0.600529  0.838584   1.19009  1.415114  1.729878  1.901076     2.02459 |  2.02459 |        6.0 |
    | 2000-01-01 04:00:00 |    300.0 | 6.670953  6.670557  6.658381  6.637431  6.587732  6.513746  6.367592     6.123257 | 0.0  0.238269  0.595448  0.831434  1.179732  1.402476  1.713546  1.882201    2.002653 | 2.002653 |        6.0 |
    | 2000-01-01 04:05:00 |    300.0 | 6.663865  6.663474  6.651363  6.630528  6.581117  6.507596  6.362606     6.120907 | 0.0   0.23624  0.590382  0.824312  1.169428  1.389926  1.697388  1.863582    1.981116 | 1.981116 |        6.0 |
    | 2000-01-01 04:10:00 |    300.0 | 6.656839  6.656452  6.644406  6.623689  6.574569  6.501518  6.357691     6.118612 | 0.0   0.23422  0.585335   0.81722  1.159184  1.377469  1.681404  1.845214    1.959962 | 1.959962 |        6.0 |
    | 2000-01-01 04:15:00 |    300.0 | 6.649873   6.64949  6.637511  6.616912  6.568085  6.495508  6.352847      6.11637 | 0.0  0.232209   0.58031  0.810163  1.149003  1.365108  1.665592  1.827088    1.939171 | 1.939171 |        6.0 |
    | 2000-01-01 04:20:00 |    300.0 | 6.642966  6.642588  6.630676  6.610198  6.561667  6.489566  6.348069      6.11418 | 0.0  0.230209  0.575311  0.803144  1.138889  1.352845   1.64995  1.809199    1.918727 | 1.918727 |        6.0 |
    | 2000-01-01 04:25:00 |    300.0 |  6.63612  6.635745  6.623901  6.603544  6.555312   6.48369  6.343357     6.112038 | 0.0  0.228221   0.57034  0.796166  1.128845  1.340684  1.634478   1.79154    1.898615 | 1.898615 |        6.0 |
    | 2000-01-01 04:30:00 |    300.0 | 6.629332  6.628962  6.617186  6.596951  6.549019  6.477879  6.338709     6.109944 | 0.0  0.226245  0.565399  0.789232  1.118875  1.328624  1.619172  1.774106    1.878821 | 1.878821 |        6.0 |
    | 2000-01-01 04:35:00 |    300.0 | 6.622604  6.622238   6.61053  6.590418  6.542788  6.472132  6.334123     6.107895 | 0.0  0.224281   0.56049  0.782346   1.10898  1.316669  1.604032   1.75689    1.859332 | 1.859332 |        6.0 |
    | 2000-01-01 04:40:00 |    300.0 | 6.615934  6.615572  6.603934  6.583945  6.536619  6.466447  6.329598      6.10589 | 0.0  0.222331  0.555615  0.775508  1.099162  1.304819  1.589055  1.739888    1.840138 | 1.840138 |        6.0 |
    | 2000-01-01 04:45:00 |    300.0 | 6.609322  6.608965  6.597395  6.577531  6.530509  6.460824  6.325133     6.103927 | 0.0  0.220394  0.550774  0.768721  1.089424  1.293075  1.574239  1.723094    1.821226 | 1.821226 |        6.0 |
    | 2000-01-01 04:50:00 |    300.0 | 6.602768  6.602415  6.590915  6.571176  6.524459  6.455261  6.320725     6.102006 | 0.0  0.218472  0.545969  0.761986  1.079767  1.281437  1.559583  1.706503    1.802587 | 1.802587 |        6.0 |
    | 2000-01-01 04:55:00 |    300.0 | 6.596271  6.595922  6.584492  6.564878  6.518468  6.449757  6.316374     6.100124 | 0.0  0.216565  0.541201  0.755304  1.070191  1.269907  1.545084  1.690111    1.784213 | 1.784213 |        6.0 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_gate_out_high_outer_waterlevel:

High outer water level
______________________

Now, we reverse the situation by setting a higher external water level:

>>> waterlevel.sequences.sim.series = 8.0

Due to the higher downstream water level and the subsequent rise of the upstream water
levels, the gate is permanently submerged:

.. integration-test::

    >>> test("sw1d_gate_out_high_outer_waterlevel")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |   outflow | waterlevel |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0       7.0       7.0      7.15946 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0   -7.973004 | -7.973004 |        8.0 |
    | 2000-01-01 00:05:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0       7.0  7.058111     7.266914 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0  -1.937031   -7.309727 | -7.309727 |        8.0 |
    | 2000-01-01 00:10:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0   7.01392  7.142799     7.333065 | 0.0        0.0        0.0        0.0        0.0        0.0  -0.696008  -3.518954   -6.826522 | -6.826522 |        8.0 |
    | 2000-01-01 00:15:00 |     300.0 |      7.0       7.0       7.0       7.0   7.00493  7.054092  7.185874     7.391116 | 0.0        0.0        0.0        0.0        0.0  -0.164324   -2.17291  -3.608722   -6.511239 | -6.511239 |        8.0 |
    | 2000-01-01 00:20:00 |  3.192526 |      7.0       7.0       7.0  7.001167  7.030075   7.08818    7.2246     7.437656 | 0.0        0.0        0.0  -0.000002  -0.058369  -0.896545  -2.600958  -3.891835   -6.218861 | -6.218861 |        8.0 |
    | 2000-01-01 00:25:00 |  5.893766 |      7.0  7.000001  7.000853  7.011226  7.057422  7.115993  7.260948      7.47625 | 0.0        0.0  -0.000058  -0.028484  -0.531418  -1.442988  -2.833625  -4.045227   -5.974891 | -5.974891 |        8.0 |
    | 2000-01-01 00:30:00 |  8.236717 | 7.000011  7.000402  7.007351  7.027111  7.077605  7.142461  7.294472     7.508655 | 0.0  -0.000363  -0.020429  -0.237049  -1.031312  -1.704077  -3.027497  -4.144971   -5.765252 | -5.765252 |        8.0 |
    | 2000-01-01 00:35:00 |  10.27112 | 7.000339  7.003033  7.019253  7.042941  7.095021  7.167786  7.325316     7.536214 | 0.0  -0.010935  -0.142454  -0.539185  -1.330674  -1.911193  -3.177423  -4.205537   -5.583477 | -5.583477 |        8.0 |
    | 2000-01-01 00:40:00 | 12.051846 | 7.002363   7.00911  7.031828  7.057306  7.112374  7.192031  7.353473      7.55994 | 0.0  -0.067467  -0.371305  -0.790445  -1.508718  -2.087158  -3.299412  -4.237986    -5.42431 |  -5.42431 |        8.0 |
    | 2000-01-01 00:45:00 | 13.615472 |  7.00798  7.017599  7.042689  7.070705  7.129937  7.215408  7.379088     7.580575 | 0.0  -0.187242  -0.611729  -0.973763  -1.643708  -2.229134  -3.397987  -4.251832   -5.283583 | -5.283583 |        8.0 |
    | 2000-01-01 00:50:00 | 14.994033 | 7.017795  7.026957  7.052457  7.083543   7.14745  7.237968  7.402445     7.598681 | 0.0  -0.327165  -0.795068  -1.120682  -1.762541  -2.346324  -3.474331  -4.252878   -5.158173 | -5.158173 |        8.0 |
    | 2000-01-01 00:55:00 | 16.217597 | 7.030601  7.036789  7.062208  7.096125  7.164692  7.259655  7.423825     7.614706 | 0.0  -0.426875  -0.918455  -1.243488  -1.872631  -2.447354  -3.531716   -4.24439   -5.045632 | -5.045632 |        8.0 |
    | 2000-01-01 01:00:00 | 17.311513 | 7.044588   7.04743  7.072565  7.108739  7.181574  7.280398  7.443465      7.62901 | 0.0  -0.466211  -0.998235   -1.34348  -1.974161  -2.536899  -3.574052  -4.228725   -4.943928 | -4.943928 |        8.0 |
    | 2000-01-01 01:05:00 | 18.296011 |  7.05849  7.059095   7.08383  7.121602  7.198113   7.30016   7.46156     7.641881 | 0.0  -0.463395  -1.046676  -1.422168  -2.065294  -2.616601  -3.604675  -4.207828   -4.851369 | -4.851369 |        8.0 |
    | 2000-01-01 01:10:00 | 19.187169 | 7.071869  7.071655  7.096078  7.134854  7.214366  7.318948  7.478276     7.653547 | 0.0   -0.44597  -1.073966  -1.482244  -2.144876  -2.686629  -3.626047  -4.183277   -4.766561 | -4.766561 |        8.0 |
    | 2000-01-01 01:15:00 | 19.998057 | 7.084854  7.084804   7.10919  7.148564  7.230388  7.336808  7.493768     7.664189 | 0.0   -0.43284  -1.090266  -1.527327  -2.212802  -2.746883  -3.639885  -4.156259   -4.688376 | -4.688376 |        8.0 |
    | 2000-01-01 01:20:00 | 20.739591 | 7.097779  7.098267  7.122928  7.162721   7.24622  7.353806  7.508173     7.673955 | 0.0  -0.430831  -1.103976  -1.561897  -2.269763  -2.797499  -3.647406  -4.127599     -4.6159 |   -4.6159 |        8.0 |
    | 2000-01-01 01:25:00 | 21.421079 |  7.11093  7.111898  7.137045  7.177251  7.261878  7.370018  7.521623     7.682966 | 0.0   -0.43836  -1.119935  -1.590523  -2.317042  -2.838956  -3.649526  -4.097831   -4.548375 | -4.548375 |        8.0 |
    | 2000-01-01 01:30:00 | 22.050539 | 7.124448  7.125672  7.151368  7.192042  7.277351  7.385517  7.534232     7.691324 | 0.0  -0.450627   -1.13932  -1.616748  -2.356261  -2.872034  -3.646981  -4.067285   -4.485169 | -4.485169 |        8.0 |
    | 2000-01-01 01:35:00 | 22.634893 |  7.13835  7.139623  7.165812  7.206974  7.292612   7.40037  7.546104     7.699115 | 0.0  -0.463396  -1.160944  -1.642404  -2.389024  -2.897737  -3.640407  -4.036164   -4.425732 | -4.425732 |        8.0 |
    | 2000-01-01 01:40:00 | 23.180103 | 7.152584  7.153791  7.180357  7.221952  7.307628  7.414635  7.557331     7.706415 | 0.0  -0.474462  -1.182887   -1.66771  -2.416613   -2.91715  -3.630386  -4.004596   -4.369589 | -4.369589 |        8.0 |
    | 2000-01-01 01:45:00 | 23.691261 | 7.167086  7.168195  7.195007  7.236911  7.322372  7.428358  7.567987     7.713288 | 0.0  -0.483395  -1.203583  -1.691921  -2.439853  -2.931295  -3.617462  -3.972682   -4.316317 | -4.316317 |        8.0 |
    | 2000-01-01 01:50:00 | 24.172677 | 7.181806  7.182825  7.209764  7.251813  7.336827   7.44158  7.578139     7.719788 | 0.0  -0.490654  -1.222174  -1.714075  -2.459176   -2.94102  -3.602125  -3.940513   -4.265546 | -4.265546 |        8.0 |
    | 2000-01-01 01:55:00 |  24.62798 |  7.19671  7.197657  7.224617  7.266639  7.350992  7.454337   7.58784     7.725964 | 0.0  -0.496819  -1.238371  -1.733462  -2.474778  -2.946955  -3.584802  -3.908178   -4.216953 | -4.216953 |        8.0 |
    | 2000-01-01 02:00:00 | 25.060219 | 7.211777  7.212656  7.239544   7.28138  7.364875  7.466663  7.597138     7.731854 | 0.0  -0.502228  -1.252183  -1.749762  -2.486784  -2.949545  -3.565836  -3.875757   -4.170258 | -4.170258 |        8.0 |
    | 2000-01-01 02:05:00 | 25.471978 | 7.226985  7.227791  7.254522  7.296027  7.378488  7.478591  7.606073     7.737492 | 0.0  -0.506945  -1.263708  -1.762958  -2.495334  -2.949105  -3.545492  -3.843314   -4.125222 | -4.125222 |        8.0 |
    | 2000-01-01 02:10:00 | 25.865473 | 7.242312  7.243034  7.269526  7.310575  7.391846  7.490153   7.61468     7.742907 | 0.0  -0.510889  -1.273035  -1.773189  -2.500611  -2.945877  -3.523964  -3.810893    -4.08164 |  -4.08164 |        8.0 |
    | 2000-01-01 02:15:00 |  26.24263 | 7.257731  7.258359  7.284539  7.325019  7.404964  7.501379  7.622994     7.748123 | 0.0  -0.513965  -1.280229  -1.780637  -2.502826  -2.940073  -3.501398  -3.778515   -4.039334 | -4.039334 |        8.0 |
    | 2000-01-01 02:20:00 | 26.605143 | 7.273215  7.273743  7.299542  7.339354  7.417854  7.512299  7.631042     7.753163 | 0.0  -0.516139  -1.285353  -1.785469  -2.502195  -2.931892  -3.477901  -3.746182    -3.99815 |  -3.99815 |        8.0 |
    | 2000-01-01 02:25:00 |  26.95451 | 7.288738  7.289164  7.314522  7.353576  7.430532   7.52294  7.638852     7.758044 | 0.0  -0.517439  -1.288485  -1.787829  -2.498926  -2.921517  -3.453555  -3.713883   -3.957951 | -3.957951 |        8.0 |
    | 2000-01-01 02:30:00 | 27.292068 | 7.304276    7.3046  7.329466  7.367683   7.44301  7.533326  7.646447     7.762784 | 0.0  -0.517933  -1.289728  -1.787853  -2.493205  -2.909119  -3.428427  -3.681599   -3.918614 | -3.918614 |        8.0 |
    | 2000-01-01 02:35:00 |  27.61901 | 7.319807   7.32003   7.34436  7.381673  7.455299  7.543481  7.653849     7.767399 | 0.0  -0.517695  -1.289196  -1.785674  -2.485206  -2.894852  -3.402569  -3.649305   -3.880026 | -3.880026 |        8.0 |
    | 2000-01-01 02:40:00 | 27.936403 | 7.335311  7.335435  7.359193  7.395546  7.467412  7.553424  7.661078     7.771901 | 0.0  -0.516793  -1.287016  -1.781436  -2.475089  -2.878854  -3.376023  -3.616974   -3.842085 | -3.842085 |        8.0 |
    | 2000-01-01 02:45:00 | 28.245206 | 7.350769  7.350795  7.373952  7.409301  7.479359  7.563175  7.668151     7.776303 | 0.0  -0.515277  -1.283307  -1.775282  -2.463007  -2.861248  -3.348823  -3.584579   -3.804697 | -3.804697 |        8.0 |
    | 2000-01-01 02:50:00 |  28.54628 | 7.366165  7.366095  7.388627  7.422936  7.491151  7.572752  7.675084     7.780617 | 0.0  -0.513191  -1.278186  -1.767357  -2.449105  -2.842148  -3.320995   -3.55209   -3.767774 | -3.767774 |        8.0 |
    | 2000-01-01 02:55:00 | 28.840399 | 7.381482  7.381319  7.403209   7.43645  7.502795   7.58217  7.681891     7.784852 | 0.0  -0.510576  -1.271764  -1.757802   -2.43352  -2.821658  -3.292562  -3.519479   -3.731236 | -3.731236 |        8.0 |
    | 2000-01-01 03:00:00 | 29.128262 | 7.396706  7.396452  7.417687  7.449843    7.5143  7.591444  7.688586     7.789018 | 0.0  -0.507472  -1.264145  -1.746749  -2.416381  -2.799875  -3.263542  -3.486716   -3.695007 | -3.695007 |        8.0 |
    | 2000-01-01 03:05:00 | 29.410495 | 7.411824  7.411483  7.432053  7.463113  7.525672  7.600585  7.695181     7.793123 | 0.0  -0.503919  -1.255428  -1.734321   -2.39781  -2.776889  -3.233951  -3.453775   -3.659016 | -3.659016 |        8.0 |
    | 2000-01-01 03:10:00 | 29.687661 | 7.426823  7.426398  7.446301  7.476258  7.536918  7.609605  7.701686     7.797174 | 0.0  -0.499959  -1.245712  -1.720638  -2.377922  -2.752785  -3.203805   -3.42063   -3.623198 | -3.623198 |        8.0 |
    | 2000-01-01 03:15:00 | 29.960264 | 7.441692  7.441187  7.460423  7.489279  7.548042  7.618515   7.70811     7.801179 | 0.0  -0.495632  -1.235088  -1.705811  -2.356823  -2.727642  -3.173115  -3.387256   -3.587492 | -3.587492 |        8.0 |
    | 2000-01-01 03:20:00 | 30.228752 | 7.456421   7.45584  7.474412  7.502172   7.55905  7.627322  7.714462     7.805143 | 0.0  -0.490974  -1.223643  -1.689946  -2.334614  -2.701534  -3.141896   -3.35363    -3.55184 |  -3.55184 |        8.0 |
    | 2000-01-01 03:25:00 | 30.493525 | 7.471002  7.470349  7.488262  7.514937  7.569944  7.636035  7.720749     7.809072 | 0.0  -0.486019  -1.211461  -1.673143  -2.311389   -2.67453  -3.110159  -3.319731    -3.51619 |  -3.51619 |        8.0 |
    | 2000-01-01 03:30:00 | 30.754935 | 7.485426  7.484705  7.501969  7.527572  7.580728  7.644659  7.726978     7.812971 | 0.0  -0.480799  -1.198619  -1.655496  -2.287237  -2.646696  -3.077917  -3.285542    -3.48049 |  -3.48049 |        8.0 |
    | 2000-01-01 03:35:00 |  31.01329 | 7.499686  7.498902  7.515526  7.540075  7.591404  7.653201  7.733154     7.816844 | 0.0  -0.475344  -1.185189  -1.637094  -2.262242  -2.618094  -3.045181  -3.251043   -3.444697 | -3.444697 |        8.0 |
    | 2000-01-01 03:40:00 | 31.268859 | 7.513776  7.512933  7.528929  7.552444  7.601973  7.661664  7.739281     7.820695 | 0.0  -0.469682  -1.171239  -1.618018  -2.236482  -2.588781  -3.011964  -3.216221   -3.408766 | -3.408766 |        8.0 |
    | 2000-01-01 03:45:00 | 31.521872 | 7.527691  7.526793  7.542175  7.564678  7.612436  7.670054  7.745365     7.824527 | 0.0  -0.463836   -1.15683  -1.598346   -2.21003  -2.558813  -2.978276  -3.181062   -3.372659 | -3.372659 |        8.0 |
    | 2000-01-01 03:50:00 | 31.772527 | 7.541426  7.540477  7.555259  7.576774  7.622795  7.678372  7.751408     7.828343 | 0.0  -0.457832  -1.142019  -1.578146  -2.182954  -2.528241  -2.944132  -3.145555    -3.33634 |  -3.33634 |        8.0 |
    | 2000-01-01 03:55:00 | 32.020987 | 7.554977   7.55398  7.568177   7.58873  7.633049   7.68662  7.757412     7.832145 | 0.0  -0.451691   -1.12686  -1.557484  -2.155318  -2.497115  -2.909543   -3.10969   -3.299776 | -3.299776 |        8.0 |
    | 2000-01-01 04:00:00 | 32.267386 |  7.56834    7.5673  7.580928  7.600546  7.643198  7.694801   7.76338     7.835934 | 0.0  -0.445432  -1.111399  -1.536419  -2.127181  -2.465481  -2.874523   -3.07346   -3.262937 | -3.262937 |        8.0 |
    | 2000-01-01 04:05:00 |  32.51183 | 7.581512  7.580432  7.593508  7.612218  7.653241  7.702915  7.769313     7.839713 | 0.0  -0.439072  -1.095681  -1.515005  -2.098597  -2.433381  -2.839083  -3.036858   -3.225797 | -3.225797 |        8.0 |
    | 2000-01-01 04:10:00 | 32.754401 | 7.594491  7.593374  7.605914  7.623744  7.663178  7.710963  7.775213     7.843482 | 0.0  -0.432628  -1.079744  -1.493292  -2.069617  -2.400856  -2.803238   -2.99988    -3.18833 |  -3.18833 |        8.0 |
    | 2000-01-01 04:15:00 | 32.995155 | 7.607274  7.606124  7.618145  7.635123  7.673008  7.718944  7.781078     7.847242 | 0.0  -0.426114  -1.063624  -1.471323  -2.040287  -2.367943  -2.767001  -2.962523   -3.150516 | -3.150516 |        8.0 |
    | 2000-01-01 04:20:00 | 33.234129 | 7.619861  7.618681  7.630199  7.646354  7.682729  7.726858   7.78691     7.850993 | 0.0  -0.419542  -1.047352  -1.449139   -2.01065  -2.334678  -2.730384  -2.924786   -3.112336 | -3.112336 |        8.0 |
    | 2000-01-01 04:25:00 | 33.471339 | 7.632248  7.631041  7.642073  7.657433  7.692339  7.734704  7.792708     7.854735 | 0.0  -0.412924  -1.030956  -1.426776  -1.980742  -2.301093  -2.693401  -2.886668   -3.073774 | -3.073774 |        8.0 |
    | 2000-01-01 04:30:00 | 33.706784 | 7.644436  7.643205  7.653767   7.66836  7.701838  7.742481  7.798472     7.858468 | 0.0  -0.406269  -1.014462  -1.404265    -1.9506  -2.267217  -2.656065   -2.84817   -3.034815 | -3.034815 |        8.0 |
    | 2000-01-01 04:35:00 | 33.940446 | 7.656424  7.655171   7.66528  7.679132  7.711222  7.750187  7.804199     7.862191 | 0.0  -0.399587  -0.997891  -1.381634  -1.920255  -2.233079  -2.618388  -2.809295   -2.995449 | -2.995449 |        8.0 |
    | 2000-01-01 04:40:00 | 34.172296 | 7.668211  7.666939  7.676609  7.689749  7.720491  7.757821  7.809889     7.865903 | 0.0  -0.392885  -0.981262  -1.358908  -1.889735  -2.198703  -2.580384  -2.770045   -2.955665 | -2.955665 |        8.0 |
    | 2000-01-01 04:45:00 | 34.402289 | 7.679796  7.678507  7.687755  7.700208  7.729643   7.76538  7.815539     7.869604 | 0.0  -0.386168  -0.964591  -1.336109  -1.859065  -2.164111  -2.542064  -2.730424   -2.915456 | -2.915456 |        8.0 |
    | 2000-01-01 04:50:00 |  34.63037 | 7.691179  7.689876  7.698715  7.710508  7.738675  7.772862  7.821149     7.873291 | 0.0  -0.379444  -0.947893  -1.313256  -1.828266  -2.129326  -2.503441  -2.690438   -2.874816 | -2.874816 |        8.0 |
    | 2000-01-01 04:55:00 | 34.856477 |  7.70236  7.701045  7.709491  7.720648  7.747585  7.780265  7.826716     7.876964 | 0.0  -0.372716  -0.931178  -1.290363  -1.797359  -2.094364  -2.464526  -2.650091   -2.833743 | -2.833743 |        8.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_gate_out_rising_outer_waterlevel:

Rising outer water level
________________________

Next, we combine both previous examples by letting the external water level rise from
six to eight meters:

>>> import numpy
>>> waterlevel.sequences.sim.series = numpy.linspace(6.0, 8.0, len(pub.timegrids.init))

The resulting water level trajectories are plausible but indicate some numerical
inaccuracies after one and a half and two and a half hours:

.. integration-test::

    >>> test("sw1d_gate_out_rising_outer_waterlevel")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |   outflow | waterlevel |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0       7.0       7.0      6.84054 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0    7.973004 |  7.973004 |        6.0 |
    | 2000-01-01 00:05:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0       7.0  6.945481     6.739664 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0   1.817298    6.861105 |  6.861105 |   6.033898 |
    | 2000-01-01 00:10:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0  6.987374  6.874902     6.677026 | 0.0        0.0        0.0        0.0        0.0        0.0     0.6313   2.983932     6.11584 |   6.11584 |   6.067797 |
    | 2000-01-01 00:15:00 |     300.0 |      7.0       7.0       7.0       7.0  6.995552  6.954451  6.844252     6.621317 | 0.0        0.0        0.0        0.0        0.0   0.148258   1.794393   2.816057    5.601501 |  5.601501 |   6.101695 |
    | 2000-01-01 00:20:00 |     300.0 |      7.0       7.0       7.0  6.998954  6.974599   6.92887  6.817662     6.577511 | 0.0        0.0        0.0        0.0   0.052289   0.750743   2.029809   2.916157    5.106436 |  5.106436 |   6.135593 |
    | 2000-01-01 00:25:00 |     300.0 |      7.0       7.0  6.999631   6.99127  6.952533  6.909543  6.792908     6.542714 | 0.0        0.0        0.0   0.012308   0.396536   1.132069   2.098417   2.923528    4.663375 |  4.663375 |   6.169492 |
    | 2000-01-01 00:30:00 |     300.0 |      7.0  6.999913  6.995341  6.978119  6.938004  6.892393  6.770503     6.515557 | 0.0        0.0   0.004346   0.147326   0.804842   1.289135   2.146616   2.893478    4.251335 |  4.251335 |    6.20339 |
    | 2000-01-01 00:35:00 |     300.0 | 6.999969  6.998496  6.985885  6.964922  6.927511  6.876869  6.750307     6.495292 | 0.0   0.001023   0.071879   0.387088   1.046948   1.396696    2.17293   2.846119    3.859357 |  3.859357 |   6.237288 |
    | 2000-01-01 00:40:00 |     300.0 | 6.999209  6.994201  6.974784  6.954091  6.917418  6.862864  6.732421     6.481371 | 0.0    0.02534   0.240093   0.610115    1.15166   1.488093   2.188319   2.784518    3.480605 |  3.480605 |   6.271186 |
    | 2000-01-01 00:45:00 |     300.0 | 6.996132   6.98712  6.965764  6.945075  6.907159  6.849939  6.716933     6.473436 | 0.0   0.102556   0.456578   0.757273   1.208074   1.550065   2.196336   2.712586    3.109293 |  3.109293 |   6.305085 |
    | 2000-01-01 00:50:00 |     300.0 | 6.989293  6.979049  6.959036  6.937133  6.897142  6.837793  6.703795     6.471261 | 0.0   0.227979    0.63155   0.855797   1.252887   1.586761   2.194043    2.63198    2.740733 |  2.740733 |   6.338983 |
    | 2000-01-01 00:55:00 |     300.0 | 6.978889  6.971279  6.953118   6.92991  6.887671  6.826374  6.692945     6.474686 | 0.0   0.346804   0.735304   0.932574   1.293726   1.609441   2.180401   2.542072    2.370859 |  2.370859 |   6.372881 |
    | 2000-01-01 01:00:00 |     300.0 | 6.966699  6.963652   6.94702  6.923109  6.878838  6.815754  6.684396     6.483603 | 0.0   0.406347   0.787697   0.990953   1.331002    1.62545   2.156451    2.44142    1.995576 |  1.995576 |    6.40678 |
    | 2000-01-01 01:05:00 |     300.0 | 6.954684  6.955493  6.940398  6.916441  6.870592  6.806036   6.67825     6.497963 | 0.0   0.400483   0.808456    1.02919   1.362593   1.637454   2.123333   2.328187    1.610174 |  1.610174 |   6.440678 |
    | 2000-01-01 01:10:00 |     300.0 | 6.943955  6.946535  6.933106  6.909691  6.862821  6.797312  6.674694     6.517798 | 0.0   0.357629   0.805508   1.048588   1.386098   1.645128   2.081342   2.199892    1.208124 |  1.208124 |   6.474576 |
    | 2000-01-01 01:15:00 |     300.0 | 6.934646  6.937048  6.925099  6.902717  6.855422  6.789656  6.673998     6.543458 | 0.0   0.310304   0.784631   1.051546   1.400246   1.646871   2.029672   2.052866    0.769864 |  0.769864 |   6.508475 |
    | 2000-01-01 01:20:00 |     300.0 | 6.926297  6.927522  6.916515  6.895438  6.848332  6.783149  6.676593     6.575802 | 0.0   0.278292   0.754596   1.040702   1.404647   1.640971   1.966336   1.879842    0.262668 |  0.262668 |   6.542373 |
    | 2000-01-01 01:25:00 |     300.0 | 6.918317  6.918344  6.907665  6.887851  6.841534  6.777915  6.683195     6.612608 | 0.0   0.266014   0.724929   1.019953   1.399321    1.62592    1.88762   1.667543   -0.172754 | -0.172754 |   6.576271 |
    | 2000-01-01 01:30:00 |     300.0 | 6.910295  6.909652   6.89889  6.880047  6.835072  6.774179  6.693317     6.633725 | 0.0   0.267389   0.702003   0.994507    1.38471   1.600133   1.786943   1.449562    0.393689 |  0.393689 |   6.610169 |
    | 2000-01-01 01:35:00 |     300.0 | 6.902101  6.901369  6.890426  6.872201  6.829079  6.772076  6.702954     6.676849 | 0.0   0.273128   0.687249   0.969378   1.361688   1.561446   1.666573   1.345331   -0.810847 | -0.810847 |   6.644068 |
    | 2000-01-01 01:40:00 |     300.0 | 6.893808  6.893334  6.882346  6.864529  6.823734   6.77112  6.716104     6.704554 | 0.0   0.276444   0.678194   0.947509   1.331095   1.509275   1.557091   1.118742   -0.266523 | -0.266523 |   6.677966 |
    | 2000-01-01 01:45:00 |     300.0 | 6.885558  6.885415  6.874611  6.857233  6.819045  6.771726   6.73195     6.736017 | 0.0   0.275001   0.670991   0.928845   1.293656    1.44995    1.41965   0.891469   -0.681702 | -0.681702 |   6.711864 |
    | 2000-01-01 01:50:00 |     300.0 | 6.877465  6.877559  6.867156  6.850424   6.81522  6.774147  6.748758     6.765711 | 0.0   0.269762   0.662526   0.911005   1.251442   1.378919   1.257871   0.697613   -0.787083 | -0.787083 |   6.745763 |
    | 2000-01-01 01:55:00 |     300.0 | 6.869581  6.869793  6.859951  6.844176  6.812509  6.778359  6.766601     6.794318 | 0.0   0.262796   0.651129   0.891296   1.203713   1.294082   1.083472   0.488677   -0.941683 | -0.941683 |   6.779661 |
    | 2000-01-01 02:00:00 |     300.0 | 6.861913  6.862178  6.853022  6.838559  6.811067  6.784442  6.785369     6.821768 | 0.0   0.255625   0.636376   0.867356   1.148185   1.196279   0.892135   0.266551   -1.105946 | -1.105946 |   6.813559 |
    | 2000-01-01 02:05:00 |     300.0 | 6.854453  6.854785  6.846448  6.833652  6.811026  6.792529  6.804996     6.847837 | 0.0   0.248651   0.618292   0.837423   1.082761   1.084121   0.679757   0.025534   -1.277901 | -1.277901 |   6.847458 |
    | 2000-01-01 02:10:00 |     300.0 | 6.847212  6.847679  6.840336  6.829557   6.81255  6.802819   6.82544     6.872199 | 0.0   0.241361   0.596648   0.800385   1.005168   0.954351   0.439869  -0.241613   -1.459709 | -1.459709 |   6.881356 |
    | 2000-01-01 02:15:00 |     300.0 | 6.840228  6.840921  6.834798  6.826414  6.815879  6.814748  6.856706     6.888544 | 0.0   0.232812   0.570729   0.755305   0.912451   0.801506   0.205054  -0.837128   -1.654378 | -1.654378 |   6.915254 |
    | 2000-01-01 02:20:00 |     300.0 | 6.833566  6.834574  6.829957  6.824425  6.821052  6.829702  6.882839      6.90798 | 0.0   0.222051   0.539373   0.700771   0.800204   0.627764  -0.119935  -0.991062   -1.962858 | -1.962858 |   6.949153 |
    | 2000-01-01 02:25:00 |     300.0 | 6.827317   6.82872  6.825959   6.82381  6.828145  6.854565  6.889123      6.93119 | 0.0   0.208303   0.501006   0.634261   0.665001   0.428576  -0.814578  -1.024038   -2.184529 | -2.184529 |   6.983051 |
    | 2000-01-01 02:30:00 |     300.0 | 6.821589  6.823469  6.822991  6.824825  6.837678  6.877722  6.897895      6.95256 | 0.0   0.190928    0.45347   0.552397   0.501643   0.183862  -0.973984  -1.266366   -2.334871 | -2.334871 |   7.016949 |
    | 2000-01-01 02:35:00 |     300.0 | 6.816514  6.818975  6.821295  6.827905  6.850641  6.893422  6.915092     6.972673 | 0.0   0.169171   0.393878   0.450405   0.296409  -0.135698  -0.920697  -1.493951   -2.499605 | -2.499605 |   7.050847 |
    | 2000-01-01 02:40:00 |     300.0 | 6.812258  6.815449  6.821257  6.833381  6.872315  6.898461  6.935515     6.993867 | 0.0   0.141863   0.318174   0.319457   0.045692  -0.676753  -0.928698  -1.609451   -2.669145 | -2.669145 |   7.084746 |
    | 2000-01-01 02:45:00 |     300.0 | 6.809045  6.813201  6.823378  6.842075  6.887484   6.90525  6.951872     7.016669 | 0.0   0.107097   0.219503   0.148805  -0.285901   -0.79155  -1.131023  -1.676274    -2.81637 |  -2.81637 |   7.118644 |
    | 2000-01-01 02:50:00 |     300.0 | 6.807199  6.812691  6.826909  6.856775  6.888834  6.915397  6.965943     7.039706 | 0.0   0.061534   0.087023  -0.030675  -0.765673  -0.810685  -1.318048  -1.787065   -2.938932 | -2.938932 |   7.152542 |
    | 2000-01-01 02:55:00 |     300.0 | 6.807216  6.814118  6.837639  6.865737  6.891251  6.924953   6.98065     7.062263 | 0.0  -0.000548  -0.071872   -0.42955  -0.877673  -0.958229  -1.436031  -1.926278   -3.054135 | -3.054135 |   7.186441 |
    | 2000-01-01 03:00:00 |     300.0 |  6.80965  6.819433  6.847287   6.87071  6.897017  6.933735   6.99588     7.084541 | 0.0  -0.081141  -0.346902  -0.668517  -0.917142  -1.109343  -1.548404  -2.056071   -3.169966 | -3.169966 |   7.220339 |
    | 2000-01-01 03:05:00 |     300.0 | 6.815638  6.827402  6.852758  6.874612  6.904119  6.942828  7.011123      7.10673 | 0.0  -0.199603  -0.598051  -0.780389  -0.975523  -1.212249  -1.666897  -2.174977   -3.284415 | -3.284415 |   7.254237 |
    | 2000-01-01 03:10:00 |     300.0 | 6.825471   6.83554  6.856167  6.878623  6.911231  6.952666  7.026477     7.128853 | 0.0  -0.327765  -0.734678  -0.848335  -1.048885  -1.285946   -1.77789  -2.289711   -3.395842 | -3.395842 |   7.288136 |
    | 2000-01-01 03:15:00 |     300.0 | 6.837714  6.842948  6.859935  6.882951  6.918243  6.963127  7.042177     7.150927 | 0.0  -0.408107  -0.778491  -0.904067  -1.120472  -1.354217  -1.877256  -2.400583   -3.504327 | -3.504327 |   7.322034 |
    | 2000-01-01 03:20:00 |     300.0 | 6.850188  6.850224  6.864896  6.887747  6.925441  6.974024  7.058275     7.173004 | 0.0  -0.415792  -0.779622  -0.944997  -1.184796  -1.424738   -1.96956  -2.506139   -3.609974 | -3.609974 |   7.355932 |
    | 2000-01-01 03:25:00 |     300.0 | 6.861299  6.858105  6.871017  6.893214  6.933075  6.985279  7.074709      7.19512 | 0.0  -0.370363  -0.764406  -0.968451  -1.241776  -1.496218  -2.058983  -2.606797   -3.712599 | -3.712599 |   7.389831 |
    | 2000-01-01 03:30:00 |     300.0 | 6.870538  6.866749  6.878153  6.899495  6.941289  6.996899  7.091421     7.217284 | 0.0  -0.307954  -0.740149  -0.978023  -1.292064  -1.565863   -2.14689  -2.703947   -3.812136 | -3.812136 |   7.423729 |
    | 2000-01-01 03:35:00 |     300.0 | 6.878281  6.875802  6.886191   6.90665  6.950154  7.008923  7.108385     7.239487 | 0.0  -0.258102   -0.71074  -0.978664  -1.336445  -1.631971  -2.233128  -2.798616    -3.90875 |  -3.90875 |   7.457627 |
    | 2000-01-01 03:40:00 |     300.0 | 6.885319  6.884771  6.894932  6.914679    6.9597  7.021387  7.125603     7.261717 | 0.0  -0.234595  -0.683071  -0.974432  -1.375875  -1.694065  -2.317284  -2.891203   -4.002725 | -4.002725 |   7.491525 |
    | 2000-01-01 03:45:00 |     300.0 | 6.892392  6.893357  6.904061   6.92352  6.969929  7.034321  7.143082     7.283968 | 0.0  -0.235791  -0.665093  -0.969412  -1.411439  -1.752409  -2.399132  -2.981765   -4.094328 | -4.094328 |   7.525424 |
    | 2000-01-01 03:50:00 |     300.0 | 6.899919  6.901569  6.913268  6.933043  6.980821  7.047743   7.16083     7.306238 | 0.0    -0.2509  -0.661501  -0.968373  -1.444528  -1.807581  -2.478657  -3.070272   -4.183767 | -4.183767 |   7.559322 |
    | 2000-01-01 03:55:00 |     300.0 | 6.907971  6.909633  6.922389   6.94307   6.99232  7.061656  7.178853     7.328528 | 0.0  -0.268386  -0.671583  -0.975644  -1.477001  -1.860322  -2.555972  -3.156722   -4.271196 | -4.271196 |    7.59322 |
    | 2000-01-01 04:00:00 |  1.260002 | 6.916422   6.91781  6.931472  6.953419  7.004348  7.076043  7.197161     7.350823 | 0.0  -0.281717  -0.690538  -0.993285  -1.510732  -1.911645  -2.630996  -3.241284   -4.356033 | -4.356033 |   7.627119 |
    | 2000-01-01 04:05:00 |  2.579174 | 6.925351  6.926071  6.940979  6.963824  7.016985  7.090823  7.215478     7.373234 | 0.0  -0.297605  -0.710664  -1.027571  -1.547814  -1.969055  -2.708056  -3.318602   -4.439177 | -4.439177 |   7.661017 |
    | 2000-01-01 04:10:00 |  3.884353 | 6.934436  6.934803  6.950541  6.974521  7.029803  7.105991  7.234165     7.395601 | 0.0  -0.302847  -0.739462  -1.058191  -1.593041  -2.020315  -2.778742  -3.401662   -4.519994 | -4.519994 |   7.694915 |
    | 2000-01-01 04:15:00 |  5.185633 | 6.943637  6.943996  6.960288  6.985471  7.042812  7.121525  7.253156     7.417954 | 0.0  -0.306693   -0.76635   -1.09126  -1.638752  -2.072389  -2.849054   -3.48206   -4.599708 | -4.599708 |   7.728814 |
    | 2000-01-01 04:20:00 |   6.48105 | 6.953031  6.953531  6.970362  6.996646   7.05607  7.137384  7.272381     7.440314 | 0.0   -0.31313  -0.789851  -1.125636   -1.68443  -2.126363  -2.919337  -3.560175   -4.678164 | -4.678164 |   7.762712 |
    | 2000-01-01 04:25:00 |  7.768514 | 6.962693  6.963334  6.980773  7.008075    7.0696   7.15354  7.291799     7.462684 | 0.0  -0.322077  -0.812217  -1.159245  -1.730677  -2.181671  -2.989454  -3.636741   -4.755287 | -4.755287 |    7.79661 |
    | 2000-01-01 04:30:00 |  9.046615 |  6.97266  6.973389  6.991478  7.019787  7.083411   7.16997  7.311386     7.485064 | 0.0  -0.332233  -0.834992  -1.191824  -1.777426  -2.237799  -3.059281  -3.712157   -4.831117 | -4.831117 |   7.830508 |
    | 2000-01-01 04:35:00 | 10.314413 | 6.982936  6.983706  7.002449  7.031794  7.097509  7.186657  7.331122     7.507446 | 0.0  -0.342545  -0.858392  -1.224115  -1.824453  -2.294378  -3.128748  -3.786611   -4.905731 | -4.905731 |   7.864407 |
    | 2000-01-01 04:40:00 | 11.571248 | 6.993512  6.994298  7.013688  7.044094  7.111892   7.20359  7.350993     7.529826 | 0.0  -0.352521  -0.882116  -1.256727   -1.87171  -2.351158  -3.197807  -3.860199   -4.979218 | -4.979218 |   7.898305 |
    | 2000-01-01 04:45:00 |  12.81665 | 7.004375  7.005174  7.025204  7.056682  7.126556  7.220758   7.37099       7.5522 | 0.0  -0.362103   -0.90593  -1.289823  -1.919244  -2.408012  -3.266422  -3.932975    -5.05166 |  -5.05166 |   7.932203 |
    | 2000-01-01 04:50:00 | 14.050283 | 7.015519  7.016341   7.03701  7.069557   7.14149  7.238152  7.391102     7.574563 | 0.0  -0.371458  -0.929779  -1.323304  -1.967064  -2.464895  -3.334572  -4.004973    -5.12313 |  -5.12313 |   7.966102 |
    | 2000-01-01 04:55:00 | 15.271911 | 7.026943  7.027799   7.04911  7.082719   7.15669  7.255761  7.411321     7.596913 | 0.0  -0.380801   -0.95371  -1.357032  -2.015126  -2.521792   -3.40225  -4.076222   -5.193693 | -5.193693 |        8.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_gate_out_oscillation_damping:

Oscillation Damping
___________________

The numerical inaccuracies that emerged in the
:ref:`sw1d_gate_out_rising_outer_waterlevel` example can cause oscillations of relevant
size.  We could (similar to the :ref:`sw1d_channel_higher_precision` example) increase
numerical precision to reduce these oscillations.  Or we can exclude them entirely by
selecting an appropriate value for parameter |DampingRadius|, wshich reduces the
calculated positive or negative flows when level gradients are nearly zero to lessen
the integration algorithm's sensitivity to the extreme (and possibly even infinitely
fast) changes in flow rates with respect to water level changes discussed in
|Calc_Discharge_V3|:

>>> model.routingmodels[-1].parameters.control.dampingradius(0.01)

After this change, all trajectories have a smooth appearance:

.. integration-test::

    >>> test("sw1d_gate_out_oscillation_damping")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |   outflow | waterlevel |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0       7.0       7.0      6.84054 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0    7.973004 |  7.973004 |        6.0 |
    | 2000-01-01 00:05:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0       7.0  6.945481     6.739664 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0   1.817298    6.861105 |  6.861105 |   6.033898 |
    | 2000-01-01 00:10:00 |     300.0 |      7.0       7.0       7.0       7.0       7.0  6.987374  6.874902     6.677026 | 0.0        0.0        0.0        0.0        0.0        0.0     0.6313   2.983932     6.11584 |   6.11584 |   6.067797 |
    | 2000-01-01 00:15:00 |     300.0 |      7.0       7.0       7.0       7.0  6.995552  6.954451  6.844252     6.621317 | 0.0        0.0        0.0        0.0        0.0   0.148258   1.794393   2.816057    5.601501 |  5.601501 |   6.101695 |
    | 2000-01-01 00:20:00 |     300.0 |      7.0       7.0       7.0  6.998954  6.974599   6.92887  6.817662     6.577511 | 0.0        0.0        0.0        0.0   0.052289   0.750743   2.029809   2.916157    5.106436 |  5.106436 |   6.135593 |
    | 2000-01-01 00:25:00 |     300.0 |      7.0       7.0  6.999631   6.99127  6.952533  6.909543  6.792908     6.542714 | 0.0        0.0        0.0   0.012308   0.396536   1.132069   2.098417   2.923528    4.663375 |  4.663375 |   6.169492 |
    | 2000-01-01 00:30:00 |     300.0 |      7.0  6.999913  6.995341  6.978119  6.938004  6.892393  6.770503     6.515557 | 0.0        0.0   0.004346   0.147326   0.804842   1.289135   2.146616   2.893478    4.251335 |  4.251335 |    6.20339 |
    | 2000-01-01 00:35:00 |     300.0 | 6.999969  6.998496  6.985885  6.964922  6.927511  6.876869  6.750307     6.495292 | 0.0   0.001023   0.071879   0.387088   1.046948   1.396696    2.17293   2.846119    3.859357 |  3.859357 |   6.237288 |
    | 2000-01-01 00:40:00 |     300.0 | 6.999209  6.994201  6.974784  6.954091  6.917418  6.862864  6.732421     6.481371 | 0.0    0.02534   0.240093   0.610115    1.15166   1.488093   2.188319   2.784518    3.480605 |  3.480605 |   6.271186 |
    | 2000-01-01 00:45:00 |     300.0 | 6.996132   6.98712  6.965764  6.945075  6.907159  6.849939  6.716933     6.473436 | 0.0   0.102556   0.456578   0.757273   1.208074   1.550065   2.196336   2.712586    3.109293 |  3.109293 |   6.305085 |
    | 2000-01-01 00:50:00 |     300.0 | 6.989293  6.979049  6.959036  6.937133  6.897142  6.837793  6.703795     6.471261 | 0.0   0.227979    0.63155   0.855797   1.252887   1.586761   2.194043    2.63198    2.740733 |  2.740733 |   6.338983 |
    | 2000-01-01 00:55:00 |     300.0 | 6.978889  6.971279  6.953118   6.92991  6.887671  6.826374  6.692945     6.474686 | 0.0   0.346804   0.735304   0.932574   1.293726   1.609441   2.180401   2.542072    2.370859 |  2.370859 |   6.372881 |
    | 2000-01-01 01:00:00 |     300.0 | 6.966699  6.963652   6.94702  6.923109  6.878838  6.815754  6.684396     6.483603 | 0.0   0.406347   0.787697   0.990953   1.331002    1.62545   2.156451    2.44142    1.995576 |  1.995576 |    6.40678 |
    | 2000-01-01 01:05:00 |     300.0 | 6.954684  6.955493  6.940398  6.916441  6.870592  6.806036   6.67825     6.497966 | 0.0   0.400483   0.808456    1.02919   1.362593   1.637454   2.123333   2.328187    1.610014 |  1.610014 |   6.440678 |
    | 2000-01-01 01:10:00 |     300.0 | 6.943955  6.946535  6.933106  6.909691  6.862821  6.797312  6.674695     6.519367 | 0.0   0.357629   0.805508   1.048588   1.386098   1.645128   2.081342   2.199868    1.129834 |  1.129834 |   6.474576 |
    | 2000-01-01 01:15:00 |     300.0 | 6.934646  6.937048  6.925099  6.902717  6.855422  6.789656  6.674361     6.552736 | 0.0   0.310304   0.784631   1.051546   1.400246   1.646871   2.029666   2.040779    0.372318 |  0.372318 |   6.508475 |
    | 2000-01-01 01:20:00 |     300.0 | 6.926297  6.927522  6.916515  6.895438  6.848332  6.783204  6.678991     6.582176 | 0.0   0.278292   0.754596   1.040702   1.404647    1.64097   1.963561   1.809223    0.337223 |  0.337223 |   6.542373 |
    | 2000-01-01 01:25:00 |     300.0 | 6.918317  6.918344  6.907665  6.887851  6.841552  6.778321  6.685847     6.613035 | 0.0   0.266014   0.724929   1.019953   1.399321   1.625332   1.869483   1.640974    0.098015 |  0.098015 |   6.576271 |
    | 2000-01-01 01:30:00 |     300.0 | 6.910295  6.909652   6.89889   6.88005  6.835209  6.774854  6.695327     6.641758 | 0.0   0.267389   0.702003   0.994507   1.384531   1.595975   1.769337   1.453321    0.017169 |  0.017169 |   6.610169 |
    | 2000-01-01 01:35:00 |     300.0 | 6.902101  6.901369  6.890427  6.872231  6.829356  6.772851  6.706543     6.667593 | 0.0   0.273128   0.687249   0.969332   1.360304    1.55539   1.655518   1.281659   -0.010085 | -0.010085 |   6.644068 |
    | 2000-01-01 01:40:00 |     300.0 | 6.893808  6.893335  6.882359  6.864607  6.824104  6.772267  6.718599     6.696966 | 0.0   0.276444   0.678177   0.947123   1.328313   1.503386   1.532634   1.130754   -0.337878 | -0.337878 |   6.677966 |
    | 2000-01-01 01:45:00 |     300.0 | 6.885558  6.885418  6.874648  6.857364  6.819576  6.772977  6.731519      6.72951 | 0.0   0.274996   0.670836   0.927872   1.290047   1.440972   1.405449   0.974807   -0.652421 | -0.652421 |   6.711864 |
    | 2000-01-01 01:50:00 |     300.0 | 6.877467  6.877571  6.867226  6.850624  6.815869  6.774953  6.746249     6.760016 | 0.0   0.269711   0.662045   0.909445   1.246419   1.369988   1.271193   0.780177   -0.745107 | -0.745107 |   6.745763 |
    | 2000-01-01 01:55:00 |     300.0 | 6.869589  6.869819  6.860061   6.84445  6.813071  6.778409  6.763073     6.790269 | 0.0    0.26261   0.650198   0.889007    1.19773   1.290993   1.118208   0.557402   -0.955244 | -0.955244 |   6.779661 |
    | 2000-01-01 02:00:00 |     300.0 | 6.861932  6.862226  6.853178  6.838868  6.811329  6.783655  6.781684     6.819365 | 0.0   0.255209   0.634888   0.864324   1.143414    1.20149   0.939152   0.318793   -1.135998 | -1.135998 |   6.813559 |
    | 2000-01-01 02:05:00 |     300.0 | 6.854494  6.854861  6.846638   6.83392   6.81085  6.790993  6.801838     6.846752 | 0.0   0.247932   0.616184   0.834184   1.081568   1.097524   0.730634   0.058838    -1.31052 |  -1.31052 |   6.847458 |
    | 2000-01-01 02:10:00 |     300.0 | 6.847285  6.847785  6.840523  6.829696  6.811888  6.800751  6.823227     6.871815 | 0.0    0.24031    0.59409   0.797915    1.00914   0.974544    0.48667  -0.226286   -1.479429 | -1.479429 |   6.881356 |
    | 2000-01-01 02:15:00 |     300.0 |  6.84034   6.84105  6.834931  6.826345  6.814766   6.81252   6.85596     6.887981 | 0.0   0.231502   0.568245   0.754667   0.922201   0.826283   0.237796   -0.85331   -1.661613 | -1.661613 |   6.915254 |
    | 2000-01-01 02:20:00 |     300.0 | 6.833718  6.834709  6.829978  6.824093  6.819626  6.827694  6.882892     6.907361 | 0.0   0.220724   0.537777    0.70286   0.815475   0.653462  -0.105226  -1.002961   -1.971958 | -1.971958 |   6.949153 |
    | 2000-01-01 02:25:00 |     300.0 | 6.827496  6.828833  6.825824  6.823197  6.826696  6.853302  6.888626     6.930777 | 0.0   0.207392   0.501217   0.639703   0.684494   0.448833  -0.831537  -1.022685    -2.19352 |  -2.19352 |   6.983051 |
    | 2000-01-01 02:30:00 |     300.0 | 6.821767  6.823526  6.822671  6.823981  6.836391  6.877081  6.896983     6.952218 | 0.0   0.190982    0.45634   0.561445   0.522257   0.199077  -0.989907  -1.268465   -2.340477 | -2.340477 |   7.016949 |
    | 2000-01-01 02:35:00 |     300.0 | 6.816645  6.818939  6.820793  6.826918  6.849663  6.893082  6.914207     6.972281 | 0.0   0.170719   0.400049   0.462626   0.315774  -0.126603  -0.926657  -1.500783   -2.503952 | -2.503952 |   7.050847 |
    | 2000-01-01 02:40:00 |     300.0 | 6.812287  6.815287  6.820607  6.832398  6.871875  6.897998  6.934902     6.993438 | 0.0   0.145289   0.327884   0.334107   0.060082  -0.680315  -0.926112  -1.615936   -2.673808 | -2.673808 |   7.084746 |
    | 2000-01-01 02:45:00 |     300.0 | 6.808911  6.812886  6.822647  6.841237  6.887374    6.9047  6.951411     7.016273 | 0.0   0.112524   0.232567   0.164539  -0.277395  -0.794048  -1.129152  -1.679475     -2.8212 |   -2.8212 |   7.118644 |
    | 2000-01-01 02:50:00 |     300.0 | 6.806847  6.812207  6.826237  6.856299  6.888483  6.914929  6.965512     7.039369 | 0.0   0.068791   0.102766   -0.01687   -0.76995  -0.806913  -1.318349  -1.788384    -2.94321 |  -2.94321 |   7.152542 |
    | 2000-01-01 02:55:00 |     300.0 | 6.806605  6.813528  6.837222  6.865402  6.890774  6.924537  6.980253     7.061974 | 0.0    0.00809   -0.05795   -0.42413  -0.879287   -0.95566  -1.436036    -1.9274   -3.057636 | -3.057636 |   7.186441 |
    | 2000-01-01 03:00:00 |     300.0 | 6.808444  6.819085  6.846999  6.870417  6.896562  6.933326  6.995521     7.084288 | 0.0  -0.061326  -0.339189  -0.665089  -0.915838  -1.108772  -1.548225  -2.057154   -3.172863 | -3.172863 |   7.220339 |
    | 2000-01-01 03:05:00 |     300.0 | 6.814242  6.827058  6.852575  6.874322  6.903735  6.942424  7.010789     7.106506 | 0.0  -0.193266  -0.591921  -0.777777  -0.973043  -1.212135  -1.667053  -2.175967    -3.28686 |  -3.28686 |   7.254237 |
    | 2000-01-01 03:10:00 |     300.0 | 6.824248   6.83503   6.85597  6.878348  6.910897  6.952283   7.02616     7.128651 | 0.0  -0.333526  -0.732154  -0.845331  -1.046653  -1.285378  -1.778316  -2.290692   -3.397935 | -3.397935 |   7.288136 |
    | 2000-01-01 03:15:00 |     300.0 | 6.836803  6.842265  6.859609  6.882691  6.917936  6.962771  7.041877     7.150742 | 0.0  -0.418488  -0.780238   -0.90153   -1.11864  -1.353279  -1.877677  -2.401586   -3.506154 | -3.506154 |   7.322034 |
    | 2000-01-01 03:20:00 |     300.0 |  6.84954  6.849459  6.864411  6.887469  6.925149  6.973694  7.057994     7.172832 | 0.0  -0.424563  -0.784261  -0.944346  -1.183283  -1.423719  -1.969866  -2.507109   -3.611602 | -3.611602 |   7.355932 |
    | 2000-01-01 03:25:00 |     300.0 | 6.860794  6.857353  6.870421  6.892883  6.932783  6.984971  7.074449      7.19496 | 0.0  -0.375155  -0.769817  -0.970132  -1.240826  -1.495303  -2.059196  -2.607681    -3.71407 |  -3.71407 |   7.389831 |
    | 2000-01-01 03:30:00 |     300.0 | 6.870068  6.866065  6.877519  6.899096   6.94098  6.996609  7.091179     7.217135 | 0.0  -0.309111  -0.744736  -0.981334  -1.291973  -1.565208  -2.147064  -2.704733    -3.81347 |  -3.81347 |   7.423729 |
    | 2000-01-01 03:35:00 |     300.0 | 6.877782  6.875199  6.885576  6.906192  6.949816   7.00864  7.108159     7.239348 | 0.0  -0.257135  -0.713834  -0.982426  -1.337225  -1.631735  -2.233321  -2.799318   -3.909963 | -3.909963 |   7.457627 |
    | 2000-01-01 03:40:00 |     300.0 | 6.884776  6.884231  6.894364  6.914184  6.959328  7.021105  7.125388     7.261587 | 0.0   -0.23314  -0.684738  -0.977654  -1.377245  -1.694329  -2.317556   -2.89185   -4.003828 | -4.003828 |   7.491525 |
    | 2000-01-01 03:45:00 |     300.0 | 6.891824  6.892848  6.903538  6.923012  6.969531  7.034033  7.142873     7.283846 | 0.0  -0.234952  -0.665796  -0.971605   -1.41301  -1.753118  -2.399534  -2.982388   -4.095335 | -4.095335 |   7.525424 |
    | 2000-01-01 03:50:00 |     300.0 | 6.899354  6.901064  6.912771  6.932541  6.980409  7.047446  7.160624     7.306122 | 0.0  -0.250987  -0.661792  -0.969548  -1.445984  -1.808573  -2.479205  -3.070904   -4.184695 | -4.184695 |   7.559322 |
    | 2000-01-01 03:55:00 |     300.0 | 6.907427  6.909119  6.921899  6.942582  6.991906   7.06135  7.178647     7.328416 | 0.0   -0.26909  -0.671857  -0.976135   -1.47818  -1.861409  -2.556644  -3.157384    -4.27206 |  -4.27206 |    7.59322 |
    | 2000-01-01 04:00:00 |  1.248749 | 6.915903  6.917288  6.930976  6.952944  7.003938  7.075732  7.196954     7.350713 | 0.0  -0.282526   -0.69094  -0.993509  -1.511627   -1.91268   -2.63175  -3.241984   -4.356857 | -4.356857 |   7.627119 |
    | 2000-01-01 04:05:00 |  2.568071 | 6.924845  6.925549  6.940477  6.963357  7.016581   7.09051  7.215268     7.373126 | 0.0  -0.298072  -0.711151   -1.02786  -1.548494  -1.969943  -2.708843  -3.319333   -4.439968 | -4.439968 |   7.661017 |
    | 2000-01-01 04:10:00 |  3.873362 | 6.933935  6.934287  6.950039  6.974057  7.029404  7.105679  7.233955     7.395493 | 0.0  -0.302997  -0.739881  -1.058603  -1.593633  -2.021079  -2.779515  -3.402415   -4.520762 | -4.520762 |   7.694915 |
    | 2000-01-01 04:15:00 |  5.174771 | 6.943135  6.943487  6.959791  6.985009  7.042417  7.121213  7.252946     7.417846 | 0.0   -0.30667  -0.766653  -1.091722  -1.639323   -2.07307  -2.849801  -3.482815   -4.600461 | -4.600461 |   7.728814 |
    | 2000-01-01 04:20:00 |  6.470338 | 6.952527  6.953027  6.969871  6.996188  7.055677  7.137074  7.272172     7.440205 | 0.0  -0.313071  -0.790068  -1.126068  -1.684992  -2.126998   -2.92006  -3.560923   -4.678904 | -4.678904 |   7.762712 |
    | 2000-01-01 04:25:00 |  7.757968 | 6.962189  6.962833  6.980288   7.00762  7.069209  7.153232  7.291591     7.462576 | 0.0  -0.322062  -0.812395  -1.159614  -1.731215  -2.182282  -2.990157  -3.637478   -4.756017 | -4.756017 |    7.79661 |
    | 2000-01-01 04:30:00 |   9.03624 | 6.972157  6.972891  6.990997  7.019336  7.083023  7.169664  7.311179     7.484955 | 0.0  -0.332281  -0.835166  -1.192136   -1.77793  -2.238389  -3.059967  -3.712881   -4.831835 | -4.831835 |   7.830508 |
    | 2000-01-01 04:35:00 | 10.304212 | 6.982436   6.98321  7.001971  7.031346  7.097124  7.186353  7.330916     7.507337 | 0.0  -0.342636  -0.858577  -1.224395  -1.824923  -2.294946   -3.12942  -3.787323   -4.906439 | -4.906439 |   7.864407 |
    | 2000-01-01 04:40:00 | 11.561221 | 6.993015  6.993803  7.013212  7.043649   7.11151  7.203288  7.350789     7.529718 | 0.0  -0.352627  -0.882313  -1.256997  -1.872152  -2.351704  -3.198465  -3.860899   -4.979916 | -4.979916 |   7.898305 |
    | 2000-01-01 04:45:00 | 12.806796 | 7.003881  7.004682  7.024731  7.056241  7.126176  7.220459  7.370787     7.552091 | 0.0  -0.362204  -0.906132  -1.290094  -1.919667  -2.408537  -3.267067  -3.933663   -5.052348 | -5.052348 |   7.932203 |
    | 2000-01-01 04:50:00 | 14.040602 | 7.015028  7.015851  7.036539  7.069119  7.141114  7.237855    7.3909     7.574455 | 0.0  -0.371546   -0.92998  -1.323577  -1.967475  -2.465403  -3.335203  -4.005651   -5.123809 | -5.123809 |   7.966102 |
    | 2000-01-01 04:55:00 | 15.262403 | 7.026454  7.027311  7.048641  7.082283  7.156317  7.255466  7.411121     7.596804 | 0.0  -0.380877  -0.953905  -1.357305  -2.015528  -2.522286  -3.402868  -4.076888   -5.194362 | -5.194362 |        8.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_gate_out_user_defined_control:

User-defined control
____________________

All of the above examples deal with a fixed gate opening.  However, gates are often
controlled in reality, so their opening degree depends on other properties.  Therefore,
parameter |GateHeight| alternatively accepts a callback function for adjusting its
values based on the current model state.  One can, for example, define a "sluice"
function that prevents any flow for reversed water level gradients:

>>> def sluice(model) -> None:
...     con = model.parameters.control.fastaccess
...     fac = model.sequences.factors.fastaccess
...     if fac.waterlevelupstream < fac.waterleveldownstream:
...         con.gateheight = 5.0
...     else:
...         con.gateheight = 6.5
>>> ();model.routingmodels[-1].parameters.control.gateheight(callback=sluice);()  # doctest: +ELLIPSIS
(...)

At the beginning of the simulation period, the calculated inner water levels resemble
those of the previous example.  However, the gate closes when the external water level
exceeds the inner one, so they start to even off around 6.8 m in the second half of the
simulation:

.. integration-test::

    >>> test("sw1d_gate_out_user_defined_control")
    |                date | timestep |                                                                       waterlevels |                                                                                   discharges |  outflow | waterlevel |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |    300.0 |      7.0       7.0       7.0       7.0       7.0       7.0       7.0      6.84054 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0    7.973004 | 7.973004 |        6.0 |
    | 2000-01-01 00:05:00 |    300.0 |      7.0       7.0       7.0       7.0       7.0       7.0  6.945481     6.739664 | 0.0        0.0        0.0        0.0        0.0        0.0        0.0   1.817298    6.861105 | 6.861105 |   6.033898 |
    | 2000-01-01 00:10:00 |    300.0 |      7.0       7.0       7.0       7.0       7.0  6.987374  6.874902     6.677026 | 0.0        0.0        0.0        0.0        0.0        0.0     0.6313   2.983932     6.11584 |  6.11584 |   6.067797 |
    | 2000-01-01 00:15:00 |    300.0 |      7.0       7.0       7.0       7.0  6.995552  6.954451  6.844252     6.621317 | 0.0        0.0        0.0        0.0        0.0   0.148258   1.794393   2.816057    5.601501 | 5.601501 |   6.101695 |
    | 2000-01-01 00:20:00 |    300.0 |      7.0       7.0       7.0  6.998954  6.974599   6.92887  6.817662     6.577511 | 0.0        0.0        0.0        0.0   0.052289   0.750743   2.029809   2.916157    5.106436 | 5.106436 |   6.135593 |
    | 2000-01-01 00:25:00 |    300.0 |      7.0       7.0  6.999631   6.99127  6.952533  6.909543  6.792908     6.542714 | 0.0        0.0        0.0   0.012308   0.396536   1.132069   2.098417   2.923528    4.663375 | 4.663375 |   6.169492 |
    | 2000-01-01 00:30:00 |    300.0 |      7.0  6.999913  6.995341  6.978119  6.938004  6.892393  6.770503     6.515557 | 0.0        0.0   0.004346   0.147326   0.804842   1.289135   2.146616   2.893478    4.251335 | 4.251335 |    6.20339 |
    | 2000-01-01 00:35:00 |    300.0 | 6.999969  6.998496  6.985885  6.964922  6.927511  6.876869  6.750307     6.495292 | 0.0   0.001023   0.071879   0.387088   1.046948   1.396696    2.17293   2.846119    3.859357 | 3.859357 |   6.237288 |
    | 2000-01-01 00:40:00 |    300.0 | 6.999209  6.994201  6.974784  6.954091  6.917418  6.862864  6.732421     6.481371 | 0.0    0.02534   0.240093   0.610115    1.15166   1.488093   2.188319   2.784518    3.480605 | 3.480605 |   6.271186 |
    | 2000-01-01 00:45:00 |    300.0 | 6.996132   6.98712  6.965764  6.945075  6.907159  6.849939  6.716933     6.473436 | 0.0   0.102556   0.456578   0.757273   1.208074   1.550065   2.196336   2.712586    3.109293 | 3.109293 |   6.305085 |
    | 2000-01-01 00:50:00 |    300.0 | 6.989293  6.979049  6.959036  6.937133  6.897142  6.837793  6.703795     6.471261 | 0.0   0.227979    0.63155   0.855797   1.252887   1.586761   2.194043    2.63198    2.740733 | 2.740733 |   6.338983 |
    | 2000-01-01 00:55:00 |    300.0 | 6.978889  6.971279  6.953118   6.92991  6.887671  6.826374  6.692945     6.474686 | 0.0   0.346804   0.735304   0.932574   1.293726   1.609441   2.180401   2.542072    2.370859 | 2.370859 |   6.372881 |
    | 2000-01-01 01:00:00 |    300.0 | 6.966699  6.963652   6.94702  6.923109  6.878838  6.815754  6.684396     6.483603 | 0.0   0.406347   0.787697   0.990953   1.331002    1.62545   2.156451    2.44142    1.995576 | 1.995576 |    6.40678 |
    | 2000-01-01 01:05:00 |    300.0 | 6.954684  6.955493  6.940398  6.916441  6.870592  6.806036   6.67825     6.497966 | 0.0   0.400483   0.808456    1.02919   1.362593   1.637454   2.123333   2.328187    1.610014 | 1.610014 |   6.440678 |
    | 2000-01-01 01:10:00 |    300.0 | 6.943955  6.946535  6.933106  6.909691  6.862821  6.797312  6.674695     6.519367 | 0.0   0.357629   0.805508   1.048588   1.386098   1.645128   2.081342   2.199868    1.129834 | 1.129834 |   6.474576 |
    | 2000-01-01 01:15:00 |    300.0 | 6.934646  6.937048  6.925099  6.902717  6.855422  6.789656  6.674361     6.552736 | 0.0   0.310304   0.784631   1.051546   1.400246   1.646871   2.029666   2.040779    0.372318 | 0.372318 |   6.508475 |
    | 2000-01-01 01:20:00 |    300.0 | 6.926297  6.927522  6.916515  6.895438  6.848332  6.783204  6.678991     6.582176 | 0.0   0.278292   0.754596   1.040702   1.404647    1.64097   1.963561   1.809223    0.337223 | 0.337223 |   6.542373 |
    | 2000-01-01 01:25:00 |    300.0 | 6.918317  6.918344  6.907665  6.887851  6.841552  6.778321  6.685847     6.613035 | 0.0   0.266014   0.724929   1.019953   1.399321   1.625332   1.869483   1.640974    0.098015 | 0.098015 |   6.576271 |
    | 2000-01-01 01:30:00 |    300.0 | 6.910295  6.909652   6.89889   6.88005  6.835209  6.774854  6.695327     6.641758 | 0.0   0.267389   0.702003   0.994507   1.384531   1.595975   1.769337   1.453321    0.017169 | 0.017169 |   6.610169 |
    | 2000-01-01 01:35:00 |    300.0 | 6.902101  6.901369  6.890427  6.872231  6.829356  6.772851  6.706543     6.667391 | 0.0   0.273128   0.687249   0.969332   1.360304    1.55539   1.655518   1.281659         0.0 |      0.0 |   6.644068 |
    | 2000-01-01 01:40:00 |    300.0 | 6.893808  6.893335  6.882359  6.864607  6.824104  6.772267  6.718579      6.69002 | 0.0   0.276444   0.678177   0.947123   1.328313   1.503386   1.532634   1.131428         0.0 |      0.0 |   6.677966 |
    | 2000-01-01 01:45:00 |    300.0 | 6.885558  6.885418  6.874648  6.857364  6.819576  6.772973  6.730727     6.710034 | 0.0   0.274996   0.670836   0.927872   1.290047   1.440972    1.40563   1.000696         0.0 |      0.0 |   6.711864 |
    | 2000-01-01 01:50:00 |    300.0 | 6.877467  6.877571  6.867226  6.850624  6.815868  6.774803  6.742521     6.727743 | 0.0   0.269711   0.662045   0.909445   1.246419   1.370027   1.278568   0.885445         0.0 |      0.0 |   6.745763 |
    | 2000-01-01 01:55:00 |    300.0 | 6.869589  6.869819  6.860061  6.844449  6.813021  6.777567  6.753675     6.743396 | 0.0    0.26261   0.650198   0.889007   1.197742   1.292653   1.154451   0.782659         0.0 |      0.0 |   6.779661 |
    | 2000-01-01 02:00:00 |    300.0 | 6.861932  6.862226  6.853178  6.838857  6.811017   6.78107  6.764039     6.757197 | 0.0   0.255209   0.634888   0.864328    1.14394   1.210727   1.035536   0.690062         0.0 |      0.0 |   6.813559 |
    | 2000-01-01 02:05:00 |    300.0 | 6.854494  6.854861  6.846634  6.833846  6.809791  6.785118  6.773557     6.769319 | 0.0   0.247932   0.616185   0.834322   1.084872   1.125749   0.923377   0.606113         0.0 |      0.0 |   6.847458 |
    | 2000-01-01 02:10:00 |    300.0 | 6.847285  6.847784  6.840492  6.829411  6.809243  6.789518  6.782225     6.779918 | 0.0   0.240311   0.594141   0.798865   1.020605   1.038854   0.818864   0.529925         0.0 |      0.0 |   6.881356 |
    | 2000-01-01 02:15:00 |    300.0 | 6.840339  6.841042  6.834802  6.825547  6.809262  6.794089  6.790063     6.789139 | 0.0   0.231518   0.568636   0.758315   0.951523   0.950913   0.722329    0.46107         0.0 |      0.0 |   6.915254 |
    | 2000-01-01 02:20:00 |    300.0 | 6.833714   6.83467  6.829592  6.822244   6.80973  6.798668  6.797094     6.797125 | 0.0   0.220854   0.539439   0.713107   0.878273   0.862645    0.63369   0.399298         0.0 |      0.0 |   6.949153 |
    | 2000-01-01 02:25:00 |    300.0 | 6.827474  6.828703  6.824875  6.819482  6.810538  6.803111  6.803343     6.804011 | 0.0   0.207995   0.506338   0.663562    0.80164   0.774709    0.55259   0.344294         0.0 |      0.0 |   6.983051 |
    | 2000-01-01 02:30:00 |    300.0 | 6.821684  6.823178  6.820656  6.817231   6.81158  6.807295  6.808833     6.809922 | 0.0   0.192985   0.469244   0.609874   0.722436    0.68773   0.478498   0.295522         0.0 |      0.0 |   7.016949 |
    | 2000-01-01 02:35:00 |    300.0 | 6.816402  6.818135  6.816937  6.815447  6.812754  6.811126  6.813589     6.814966 | 0.0   0.176082   0.428229   0.552208   0.641426   0.602289    0.41076   0.252213         0.0 |      0.0 |   7.050847 |
    | 2000-01-01 02:40:00 |    300.0 | 6.811674  6.813617  6.813719  6.814076  6.813967  6.814531  6.817644     6.819235 | 0.0   0.157593    0.38351   0.490788   0.559311    0.51887   0.348606    0.21344         0.0 |      0.0 |   7.084746 |
    | 2000-01-01 02:45:00 |    300.0 | 6.807541  6.809664  6.811002  6.813061  6.815135  6.817464  6.821032     6.822799 | 0.0   0.137761   0.335404   0.425968   0.476735   0.437808    0.29117   0.178239         0.0 |      0.0 |   7.118644 |
    | 2000-01-01 02:50:00 |    300.0 | 6.804039  6.806313  6.808783   6.81234  6.816186  6.819898  6.823787     6.825714 | 0.0   0.116738   0.284298   0.358247   0.394282   0.359252   0.237535   0.145714         0.0 |      0.0 |   7.152542 |
    | 2000-01-01 02:55:00 |    300.0 | 6.801201  6.803593  6.807055  6.811856  6.817065  6.821826  6.825938     6.828016 | 0.0   0.094622   0.230626   0.288239   0.312466   0.283166   0.186796   0.115094         0.0 |      0.0 |   7.186441 |
    | 2000-01-01 03:00:00 |    300.0 | 6.799055  6.801525  6.805802  6.811554  6.817735   6.82325  6.827509     6.829731 | 0.0   0.071516    0.17488   0.216638   0.231706   0.209364   0.138132    0.08575         0.0 |      0.0 |   7.220339 |
    | 2000-01-01 03:05:00 |    300.0 | 6.797628  6.800124  6.805006  6.811392  6.818177  6.824185  6.828519     6.830874 | 0.0   0.047585    0.11763   0.144168     0.1523   0.137567   0.090861   0.057185         0.0 |      0.0 |   7.254237 |
    | 2000-01-01 03:10:00 |    300.0 | 6.796935  6.799396  6.804646  6.811334  6.818385  6.824645  6.828982     6.831455 | 0.0   0.023092   0.059538   0.071551   0.074418   0.067471   0.044464    0.02903         0.0 |      0.0 |   7.288136 |
    | 2000-01-01 03:15:00 |    300.0 | 6.796983  6.799336  6.804702  6.811362  6.818365  6.824649  6.828909     6.831475 | 0.0  -0.001604   0.001371  -0.000508  -0.001873  -0.001201  -0.001414   0.001031         0.0 |      0.0 |   7.322034 |
    | 2000-01-01 03:20:00 |    300.0 | 6.797772  6.799922  6.805181   6.81146  6.818129   6.82422  6.828295     6.830942 | 0.0  -0.026311  -0.055603  -0.071569  -0.076471  -0.068617  -0.047153  -0.026669         0.0 |      0.0 |   7.355932 |
    | 2000-01-01 03:25:00 |    300.0 | 6.799242   6.80111  6.805967  6.811637  6.817728  6.823416  6.827222     6.829859 | 0.0   -0.04901  -0.108419  -0.134621  -0.143473  -0.130095  -0.089918  -0.054169         0.0 |      0.0 |   7.389831 |
    | 2000-01-01 03:30:00 |    300.0 |  6.80131  6.802774  6.806994  6.811881  6.817229  6.822341  6.825764     6.828268 | 0.0  -0.068901  -0.152121  -0.186339  -0.198561  -0.181921  -0.128147   -0.07954         0.0 |      0.0 |   7.423729 |
    | 2000-01-01 03:35:00 |    300.0 | 6.803837  6.804776  6.808204  6.812184  6.816694  6.821099  6.824003     6.826244 | 0.0  -0.084265  -0.184326  -0.224686  -0.239811   -0.22197  -0.159881  -0.101199         0.0 |      0.0 |   7.457627 |
    | 2000-01-01 03:40:00 |    300.0 | 6.806646  6.806992  6.809547  6.812544  6.816166  6.819775  6.822034     6.823888 | 0.0  -0.093632  -0.204451  -0.249193    -0.2672  -0.249624   -0.18344  -0.117791         0.0 |      0.0 |   7.491525 |
    | 2000-01-01 03:45:00 |    300.0 | 6.809543  6.809323  6.810974  6.812966  6.815673  6.818426  6.819949     6.821321 | 0.0   -0.09654   -0.21307  -0.260662  -0.281779   -0.26534  -0.197857  -0.128374         0.0 |      0.0 |   7.525424 |
    | 2000-01-01 03:50:00 |    300.0 | 6.812354  6.811677  6.812451  6.813453  6.815227  6.817084  6.817834     6.818669 | 0.0  -0.093714  -0.211423  -0.260653  -0.285008  -0.270138  -0.203056  -0.132558         0.0 |      0.0 |   7.559322 |
    | 2000-01-01 03:55:00 |    300.0 | 6.814954  6.813967  6.813944  6.814001  6.814833  6.815773  6.815761     6.816058 | 0.0  -0.086668  -0.201192  -0.250966  -0.278366  -0.265227  -0.199703  -0.130578         0.0 |      0.0 |    7.59322 |
    | 2000-01-01 04:00:00 |    300.0 | 6.817267  6.816112  6.815414  6.814598  6.814492  6.814516   6.81379     6.813593 | 0.0  -0.077091  -0.184339  -0.233336  -0.263188  -0.251812  -0.188917  -0.123221         0.0 |      0.0 |   7.627119 |
    | 2000-01-01 04:05:00 |    300.0 | 6.819258  6.818042   6.81681  6.815224  6.814204  6.813335  6.811978     6.811361 | 0.0  -0.066367  -0.162866   -0.20937  -0.240632  -0.231029   -0.17201  -0.111613         0.0 |      0.0 |   7.661017 |
    | 2000-01-01 04:10:00 |    300.0 | 6.820919  6.819706  6.818071  6.815847   6.81397  6.812263  6.810377     6.809422 | 0.0  -0.055369  -0.138535  -0.180585  -0.211739  -0.203942  -0.150318  -0.096952         0.0 |      0.0 |   7.694915 |
    | 2000-01-01 04:15:00 |    300.0 | 6.822253   6.82107  6.819143  6.816429  6.813791  6.811333  6.809033     6.807816 | 0.0   -0.04448  -0.112679  -0.148405  -0.177532  -0.171575  -0.125089  -0.080286         0.0 |      0.0 |   7.728814 |
    | 2000-01-01 04:20:00 |    300.0 | 6.823266  6.822118  6.819981  6.816929  6.813667  6.810582  6.807983     6.806568 | 0.0  -0.033753  -0.086154  -0.114099    -0.1391  -0.134962  -0.097411  -0.062413         0.0 |      0.0 |   7.762712 |
    | 2000-01-01 04:25:00 |    300.0 | 6.823958  6.822844  6.820559  6.817308  6.813594  6.810041  6.807254     6.805691 | 0.0  -0.023087  -0.059421  -0.078692  -0.097627  -0.095196  -0.068168  -0.043868         0.0 |      0.0 |    7.79661 |
    | 2000-01-01 04:30:00 |    300.0 |  6.82433  6.823251  6.820866  6.817537  6.813567  6.809734  6.806862     6.805191 | 0.0  -0.012376  -0.032689  -0.042925  -0.054363  -0.053452  -0.038058  -0.024988         0.0 |      0.0 |   7.830508 |
    | 2000-01-01 04:35:00 |    300.0 | 6.824377   6.82334  6.820904  6.817602  6.813579  6.809667  6.806812     6.805071 | 0.0  -0.001579   -0.00606    -0.0073  -0.010555  -0.010962  -0.007649  -0.005993         0.0 |      0.0 |   7.864407 |
    | 2000-01-01 04:40:00 |    300.0 | 6.824099  6.823118   6.82068  6.817505  6.813626  6.809838    6.8071      6.80533 | 0.0   0.009259   0.020361   0.027805   0.032619   0.031035   0.022522   0.012927         0.0 |      0.0 |   7.898305 |
    | 2000-01-01 04:45:00 |    300.0 | 6.823502  6.822599  6.820224  6.817272  6.813705  6.810216  6.807698     6.805956 | 0.0   0.019926   0.045882   0.061078   0.072777   0.070157   0.051233   0.031304         0.0 |      0.0 |   7.932203 |
    | 2000-01-01 04:50:00 |    300.0 |   6.8226  6.821812  6.819581  6.816942  6.813812  6.810754  6.808555     6.806921 | 0.0   0.030052   0.069398   0.090835   0.107319   0.103751    0.07686   0.048279         0.0 |      0.0 |   7.966102 |
    | 2000-01-01 04:55:00 |    300.0 | 6.821427  6.820798  6.818804   6.81656  6.813946    6.8114  6.809611     6.808179 | 0.0   0.039114    0.08982   0.115732   0.134804   0.130345   0.098039   0.062861         0.0 |      0.0 |        8.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.interfaces import routinginterfaces

# ...from musk
from hydpy.models.sw1d import sw1d_model


class Model(modeltools.AdHocModel, routinginterfaces.RoutingModel_V3):
    """|sw1d_gate_out.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="SW1D-Gate-Out",
        description=(
            "submodel for calculating flow under a submerged gate at a channel outlet"
        ),
    )
    __HYDPY_ROOTMODEL__ = False

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = ()
    INTERFACE_METHODS = (
        sw1d_model.Perform_Preprocessing_V5,
        sw1d_model.Determine_MaxTimeStep_V5,
        sw1d_model.Determine_Discharge_V6,
        sw1d_model.Get_MaxTimeStep_V1,
        sw1d_model.Get_Discharge_V1,
        sw1d_model.Get_PartialDischargeDownstream_V1,
        sw1d_model.Get_DischargeVolume_V1,
        sw1d_model.Set_TimeStep_V1,
        sw1d_model.Perform_Postprocessing_V2,
    )
    ADD_METHODS = (
        sw1d_model.Pick_WaterLevelDownstream_V1,
        sw1d_model.Reset_DischargeVolume_V1,
        sw1d_model.Calc_WaterLevelUpstream_V1,
        sw1d_model.Calc_WaterLevel_V4,
        sw1d_model.Calc_MaxTimeStep_V5,
        sw1d_model.Calc_Discharge_V3,
        sw1d_model.Update_DischargeVolume_V1,
        sw1d_model.Pass_Discharge_V1,
    )
    OUTLET_METHODS = ()
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.StorageModel_V1,
    )
    SUBMODELS = ()

    storagemodelupstream = modeltools.SubmodelProperty(
        routinginterfaces.StorageModel_V1, sidemodel=True
    )
    storagemodelupstream_is_mainmodel = modeltools.SubmodelIsMainmodelProperty()
    storagemodelupstream_typeid = modeltools.SubmodelTypeIDProperty()

    routingmodelsupstream = modeltools.SubmodelsProperty(
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        sidemodels=True,
    )


tester = Tester()
cythonizer = Cythonizer()
