from __future__ import annotations
import asyncio
from typing import cast

import httpx

from tgpromo.__version__ import __version__
from tgpromo.exceptions import APIUnavailableException
from tgpromo.logging import logger


class _ResilientHttpProxy:
    trigger_healthcheck_on_status = {503, 502}

    def __init__(self, owner_client: BaseClient, http_client: httpx.AsyncClient):
        self._owner = owner_client
        self._http = http_client

    def __getattr__(self, name):
        attr = getattr(self._http, name)
        if not callable(attr):
            return attr

        if self._owner.is_api_unreachable:
            raise APIUnavailableException('tgpromo API is temporarily unavailable')

        async def wrapper(*args, **kwargs):
            try:
                response = await attr(*args, **kwargs)
            except (httpx.ConnectError, httpx.ConnectTimeout) as e:
                logger.warning('Connection error. Triggering health check.')
                self._owner.start_health_check()
                raise APIUnavailableException('tgpromo API is temporarily unavailable') from e
            except Exception as e:
                logger.exception('Unexpected error during request: %s', e)
                raise APIUnavailableException('Unexpected error in tgpromo client') from e

            if response.status_code in self.trigger_healthcheck_on_status:
                logger.warning('Received %s. Triggering health check.', response.status_code)
                self._owner.start_health_check(skip_initial_sleep=True)
            return response
        return wrapper


class BaseClient:
    healthcheck_path: str = '/health'
    healthcheck_interval: float = 10.0

    def __init__(
        self,
        token: str,
        base_url: str,
        user_agent_id: str
    ):
        logger.info('Initializing %s client with base_url=%s', user_agent_id, base_url)

        self._token = token

        self._closed = False
        self._api_unreachable = False

        self._health_task = None

        self._raw_http = httpx.AsyncClient(
            base_url=base_url,
            headers=self._build_headers(token, user_agent_id),
            timeout=httpx.Timeout(10.0, connect=5.0),
        )
        self._http: httpx.AsyncClient = cast(httpx.AsyncClient, _ResilientHttpProxy(self, self._raw_http))
        logger.debug('%s client initialized with base_url=%s', user_agent_id.title(), base_url)

    @property
    def is_api_unreachable(self) -> bool:
        return self._api_unreachable

    @property
    def is_closed(self) -> bool:
        return self._closed

    @staticmethod
    def _build_headers(token: str, user_agent_id: str) -> dict:
        return {
            'Authorization': f'Bearer {token}',
            'User-Agent': f'tgpromo-{user_agent_id}-client/{__version__}'
        }

    def start_health_check(self, skip_initial_sleep: bool = False) -> None:
        self._api_unreachable = not skip_initial_sleep

        if not self._health_task:
            logger.info('Health check task started (skip_initial_sleep=%s)', skip_initial_sleep)
            self._health_task = asyncio.create_task(self._health_check_loop(skip_initial_sleep))
        else:
            logger.debug('Health check already running.')

    async def _health_check_loop(self, skip_initial_sleep: bool):
        logger.info('Starting health check loop (skip_initial_sleep=%s)', skip_initial_sleep)

        while not self._closed:
            if not skip_initial_sleep:
                logger.debug('Waiting %s seconds before next health check...', self.healthcheck_interval)
                await asyncio.sleep(self.healthcheck_interval)
            skip_initial_sleep = False

            try:
                resp = await self._raw_http.get(self.healthcheck_path, timeout=5.0)

                if resp.status_code < 500:
                    logger.info('API is reachable. Exiting health check loop.')
                    break
                else:
                    logger.warning('Health check returned status %s', resp.status_code)
            except asyncio.CancelledError:
                raise
            except Exception as e:
                logger.warning('Health check failed: %s', e)

            self._api_unreachable = True

        self._api_unreachable = False
        self._health_task = None
        logger.debug('Health check loop terminated.')

    async def aclose(self):
        if not self._closed:
            logger.info('Closing BaseClient...')
            if self._health_task:
                self._health_task.cancel()
                try:
                    await self._health_task
                except asyncio.CancelledError:
                    logger.debug('Health check task cancelled.')

            await self._raw_http.aclose()
            self._closed = True
            self._health_task = None
            logger.info('BaseClient closed.')

    async def __aenter__(self):
        return self

    async def __aexit__(self, *_):
        await self.aclose()
