from typing import Type, TypeVar

from httpx import AsyncClient

from tgpromo.clients.partner.models import PartnerAPIResponse, PartnerAPIError
from tgpromo.clients.partner.exceptions import PartnerAPIException, PartnerException
from tgpromo.logging import logger


T = TypeVar('T')


class BaseAPI:
    response_model_cls = PartnerAPIResponse
    exception_cls = PartnerAPIException
    error_model_cls = PartnerAPIError

    def __init__(self, http: AsyncClient):
        self._http = http
        logger.debug('BaseAPI initialized with provided HTTP client')

    async def _request(self, method: str, url: str, result_model: Type[T], **kwargs) -> T:
        logger.debug('Sending %s request to %s with params: %s', method.upper(), url, kwargs)
        response = await self._http.request(method, url, **kwargs)
        logger.debug('Received response [%s] from %s', response.status_code, url)

        try:
            data = response.json()
            parsed = self.response_model_cls[result_model].model_validate(data)
        except Exception as e:
            logger.exception('Failed to parse JSON from response: %s', e)
            raise PartnerException('Invalid API response structure') from e

        if not parsed.success:
            logger.warning('API returned error response: %s', parsed.error)
            raise self.exception_cls(parsed.error)

        logger.debug('API call successful: result=%s', parsed.result)
        return parsed.result
