import { aws_iam as iam, aws_imagebuilder as imagebuilder } from 'aws-cdk-lib';
import { Vpc } from 'aws-cdk-lib/aws-ec2';
import { Construct } from 'constructs';
export interface IEbsParameters {
    /**
       * Size of the volume in GiB
       */
    volumeSize: number;
    /**
       * Type of the volume
       *
       * @default gp2
       */
    volumeType?: string;
    /**
       * KMS Key Alias for the volume
       * If not specified, the default AMI encryption key alias will be used
       * Custom KMS Keys Alias need to exist in the other accounts for distribution to work correctly.
       * @default alias/aws/ebs
       */
    kmsKeyId?: string;
    /**
       * Whether the volume is encrypted
       *
       * @default true
       */
    encrypted?: boolean;
    /**
       * Whether the volume is deleted when the instance is terminated
       *
       * @default true
       */
    deleteOnTermination?: boolean;
}
export interface VolumeProps {
    /**
       * Name of the volume
       */
    readonly deviceName: string;
    /**
       * EBS Block Store Parameters
       *
       * By default, the 'kmsKeyId' of EBS volume is set to 'amiEncryptionKey.keyId',
       * and 'encrypted' is set to 'true'. If you wish to use a different KMS Key,
       * you may do so. However, please make sure that the necessary permissions
       * and compliance requirements for the KMS Key are already set up.
       */
    readonly ebs: IEbsParameters;
}
/**
 * Build commands for the component
 */
export interface IActionCommands {
    commands: string[];
}
export interface IStepCommands {
    name: string;
    action: 'ExecuteBash' | 'ExecutePowerShell' | 'UpdateOS';
    inputs?: IActionCommands;
}
/**
 * Phases for the component
 */
export interface IPhases {
    name: 'build' | 'validate' | 'test';
    steps: IStepCommands[];
}
export interface IInputParameter {
    type: string;
    default: string;
    description: string;
}
export type IParameters = Record<string, IInputParameter>;
/**
 * Component data
 */
export interface IComponentDocument {
    name?: string;
    description?: string;
    schemaVersion?: string;
    phases: IPhases[];
}
/**
 * Component props
 */
export interface IComponentProps {
    name?: string;
    description?: string;
    componentVersion?: string;
    schemaVersion?: string;
    parameters?: IParameters;
    platform?: 'Linux' | 'Windows';
    componentDocument: IComponentDocument;
}
export interface ImagePipelineProps {
    readonly subnetId?: string;
    readonly securityGroups?: string[];
    readonly distributionRegions?: string[];
    /**
       * Vpc to use for the Image Builder Pipeline
       */
    readonly vpc: Vpc;
    /**
       * List of component props
       */
    readonly components: IComponentProps[];
    /**
       * Name of the instance profile that will be associated with the Instance Configuration.
       */
    readonly profileName?: string;
    /**
       * Additional policies to add to the instance profile associated with the Instance Configurations
       */
    readonly additionalPolicies?: iam.ManagedPolicy[];
    /**
       * UserData script that will override default one (if specified)
       *
       * @default - none
       */
    readonly userDataScript?: string;
    /**
       * Image recipe version (Default: 0.0.1)
       */
    readonly imageRecipeVersion?: string;
    /**
       * The source (parent) image that the image recipe uses as its base environment. The value can be the parent image ARN or an Image Builder AMI ID
       */
    readonly parentImage: string;
    /**
       * List of instance types used in the Instance Configuration (Default: [ 't3.medium', 'm5.large', 'm5.xlarge' ])
       */
    readonly instanceTypes?: string[];
    /**
       * Platform type Linux or Windows (Default: Linux)
       */
    readonly platform?: 'Linux' | 'Windows';
    /**
       * Email used to receive Image Builder Pipeline Notifications via SNS
       */
    readonly email?: string;
    /**
       * List of security group IDs for the Infrastructure Configuration
       */
    readonly securityGroupIds?: string[];
    /**
       * Subnet ID for the Infrastructure Configuration
       */
    readonly ebsVolumeConfigurations?: VolumeProps[];
    /**
       * Set to true if you want to enable continuous vulnerability scans through AWS Inpector
       */
    readonly enableVulnScans?: boolean;
    /**
       * Store vulnerability scans through AWS Inspector in ECR using this repo name (if option is enabled)
       */
    readonly vulnScansRepoName?: string;
    /**
       * Store vulnerability scans through AWS Inspector in ECR using these image tags (if option is enabled)
       */
    readonly vulnScansRepoTags?: string[];
    /**
       * This variable represents an array of shared account IDs.
       * It is optional and readonly.
       * If it is provided, this AMI will be visible to the accounts in the array.
       *  In order to share the AMI with other accounts, you must specify a KMS key ID for the EBS volume configuration as AWS does not allow sharing AMIs encrypted with the default AMI encryption key.
       * @type {Array<string>}
       * @readonly
       */
    readonly distributionAccountIds?: string[];
    /**
       * Flag indicating whether the debug image pipeline is enabled or not.
       * This variable is optional. Default value is false.
       *  Functionally, this will flag to return as finished immediately after first check to see if the image pipeline has finished.
       *  This is useful for debugging the image pipeline.  However, there will be no AMI value returned.
       *
       * @type {boolean}
       * @readonly
       */
    readonly debugImagePipeline?: boolean;
    /**
       * The alias of the KMS key used for encryption and decryption of content in the distribution.
       * This property is optional and readonly.
       * The default encryption key is not compatible with cross-account AMI sharing.
       * If you specify distributionAccountIds, you must specify a non-default encryption key using this property. Otherwise, Image Builder will throw an error.
       * Keep in mind that the KMS key in the distribution account must allow the EC2ImageBuilderDistributionCrossAccountRole role to use the key.
       * @type {string}
       * @readonly
       */
    readonly distributionKmsKeyAlias?: string;
}
export declare class ImagePipeline extends Construct {
    imageRecipeComponents: imagebuilder.CfnImageRecipe.ComponentConfigurationProperty[];
    imagePipelineArn: string;
    imageId: string;
    constructor(scope: Construct, id: string, props: ImagePipelineProps);
    private stringToMD5;
}
