from __future__ import annotations

import json
import re
from ast import ClassDef, Module
from copy import copy
from pathlib import Path
from typing import TYPE_CHECKING

import astor

from ethbinductor.bind import create_def_node
from ethbinductor.conf import BB_cls_index, abi_sample_module_tree

if TYPE_CHECKING:
	from typing import Any


def abi2py(filepath: 'str | Path', cls_name: 'str | None' = None) -> str:
	if not isinstance(filepath, Path):
		file = Path(filepath)
	if not cls_name:
		cls_name = file.stem

	with open(filepath, 'r') as f:
		return abi2pys(json.load(f), cls_name)


def abi2pys(abi: list[dict[str, Any]], cls_name: str) -> str:
	abi_sm_tree: Module = copy(abi_sample_module_tree)
	cls_tree: ClassDef = abi_sm_tree.body[BB_cls_index]
	assert isinstance(cls_tree, ClassDef)

	cls_name = re.sub(r'[^A-Za-z0-9_]', '', cls_name)
	cls_tree.name = cls_name

	# Add methods to class
	for obj in abi:
		if obj['type'] == 'function':
			func = create_def_node(obj)
			cls_tree.body.append(func)

	return astor.to_source(abi_sm_tree)


# FIXME: Move into tests after add cli..
if __name__ == '__main__':
	# TODO: Lint & autoformat generated code..
	with open('examples/eth_erc20_bind.py', 'w') as code_file:
		data: str = abi2py('examples/abi/erc20.json') + (
			'\n# Generated by ETHBinductor: '
			'https://github.com/BlackCatDevel0per/ETHBinductor\n'
		)
		code_file.write(data)

# TODO: Add convert data providers (file, etherscan & etc.)
